/* Author: Jeff Dalton
 * Updated: Tue Apr 25 20:29:44 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2003, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.awt.Component;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.TableModelEvent;

import java.awt.BorderLayout;
import java.awt.event.*;

import java.util.*;

import ix.icore.process.event.*;

import ix.icore.domain.*;

import ix.iface.util.SortingKVTable;
import ix.iface.util.IFUtil;
import ix.util.*;
import ix.util.lisp.*;

public class StateViewTable extends JPanel implements StateViewer {

    protected Ip2 ip2;
    protected ViewTable viewTable;
    protected PropertyViewTable propertyViewer;

    /**
     * Constructs a viewer for the indicated agent.
     */
    public StateViewTable(Ip2 ip2) {
	super();
	this.ip2 = ip2;
	this.viewTable = new ViewTable(this);
	this.propertyViewer = new PropertyViewTable(this);

	setLayout(new BorderLayout());
	JTable tab = viewTable.getJTable();
	add(tab.getTableHeader(), BorderLayout.NORTH);
	add(tab, BorderLayout.CENTER);

    }

    public Component getView(PanelFrame frame) {
	return viewTable.getJTable();
    }

    /**
     * Sets the viewer back to something approximating its initial state.
     */
    public void reset() {
	viewTable.reset();
	propertyViewer.reset();
    }

    /*
     * Methods in ProcessStatusListener interface
     */

    /** Ignored by this viewer. */
    public void statusUpdate(ProcessStatusEvent event) { }

    /** Ignored by this viewer. */
    public void newBindings(ProcessStatusEvent event, Map bindings) { }

    public void stateChange(ProcessStatusEvent event, Map delta) {
	for (Iterator i = delta.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    LList pattern = (LList)e.getKey();
	    Object value = e.getValue();
	    viewTable.recordNewValue(pattern, value);
	    propertyViewer.recordNewValue(pattern, value);
	}
    }

    public void stateDeletion(ProcessStatusEvent event, Map delta) {
	for (Iterator i = delta.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    LList pattern = (LList)e.getKey();
	    Object value = e.getValue();
	    viewTable.deleteEntry(pattern, value);
	    propertyViewer.deleteEntry(pattern, value);
	}
    }

    static class ViewTable extends SortingKVTable {

	// /\/: Static so that PropertyViewTable can subclass it.

	StateViewTable stateViewer;

	ViewTable(StateViewTable stateViewer) {
	    this(stateViewer, "Pattern", "Value");
	}

	ViewTable(StateViewTable stateViewer, 
		  String keyColName,
		  String valColName) {
	    super(keyColName, valColName);
	    this.stateViewer = stateViewer;
	    // setKeySortComparator(new PatternObjectComparator());
	}

	protected String keyToString(Object key) {
	    // We redefine this to omit the outermost open and close parans.
	    return Lisp.elementsToString((LList)key);
	}

	protected RowPopupMenu makePopupMenu() {
	    return new StatePopupMenu();
	}

	protected class StatePopupMenu extends RowPopupMenu {
	    JMenu viewMenu = new JMenu("View Object Properties");
	    StatePopupMenu() {
		super();
		add(makeMenuItem("Change"));
		add(makeMenuItem("Delete"));
		add(makeMenuItem("Copy to Messenger"));
		add(new SortMenu("Sort Patterns"));
	    }
	    public void setRow(int row) {
		super.setRow(row);
		remove(viewMenu);
		Set objects = getRowObjects(row);
		if (!objects.isEmpty()) {
		    addObjectMenuItems(viewMenu, objects);
		    add(viewMenu);
		}
	    }
	    void addObjectMenuItems(JMenu menu, Set objects) {
		menu.removeAll();
		for (Iterator i = objects.iterator(); i.hasNext();) {
		    menu.add(makeObjectMenuItem(i.next()));
		}
	    }
	    JMenuItem makeObjectMenuItem(final Object obj) {
		String pname = Lisp.printToString(obj);
		return IFUtil.makeMenuItem(pname, new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			stateViewer.propertyViewer.viewObject(obj);
		    }
		});
	    }
	}

	protected Set getRowObjects(int row) {
	    Object key = keyList.get(row);
	    return getRowObjects(key, getValue(key));
	}

	protected Set getRowObjects(Object key, Object value) {
	    // Looks at pattern.cdr() and value
	    LList pattern = (LList)key;
	    Set result = getMentionedObjects(pattern.cdr());
	    result.addAll(getMentionedObjects(value));
	    return result;
	}

	protected Set getMentionedObjects(Object item) {
	    //\/ Looks at only top level.
	    Set result = new TreeSet(new LexicographicComparator());
	    List candidates = item instanceof LList
		? (List)item
		: Lisp.list(item);
	    for (Iterator i = candidates.iterator(); i.hasNext();) {
		Object elt = i.next();
		if ((elt instanceof Symbol || elt instanceof String)
		    && isObjectName(elt))
		  result.add(elt);
	    }
	    return result;
	}

	protected boolean isObjectName(Object name) {
	    //\/ Slow, because it iterates through the key-list each time.
	    for (Iterator i = keyList.iterator(); i.hasNext();) {
		LList pattern = (LList)i.next();
		if (pattern.size() >= 2 && pattern.get(1) == name)
		    return true;
	    }
	    return false;
	}

	protected void doPopupAction(ActionEvent event, int row, Object key) {
	    String command = event.getActionCommand();
	    LList pattern = (LList)key;
	    Object value = getValue(pattern);
	    Ip2 ip2 = stateViewer.ip2;
	    if (command.equals("Change")) {
		PatternAssignment pv = new PatternAssignment(pattern, value);
		ip2.frame.getSendPanelVisible().initConstraint(pv);
		ip2.frame.getSendPanelVisible().initToMe();
	    }
	    else if (command.equals("Delete")) {
		PatternAssignment pv = new PatternAssignment(pattern, value);
		Constraint c = 
		    new Constraint("world-state", "effect",
				   Lisp.list(pv));
		ip2.getModelManager().deleteConstraint(c);
	    }
	    else if (command.equals("Copy to Messenger")) {
		PatternAssignment pv = new PatternAssignment(pattern, value);
		ip2.frame.getSendPanelVisible().initConstraint(pv);
	    }
	    // "View ..." subitems each have their own ActionListener
	    else
		throw new ConsistencyException
		    ("Nothing to do for " + command);
	}

    }

}
