/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Dec  9 18:05:27 2008 by Jeff Dalton
 * Copyright: (c) 2001 - 2005, 2007, 2008, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.process.*;
import ix.icore.plan.*;
import ix.icore.plan.inspect.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Constructs a plan from the current contents of the model.
 *
 * @see ix.icore.plan.Plan
 */
class PlanMaker {

    // /\/ Duplicates dcl in Ip2ModelManager
    public static final Symbol
	S_WORLD_STATE = Symbol.intern("world-state"),
	S_CONDITION   = Symbol.intern("condition"),
	S_EFFECT      = Symbol.intern("effect");

    Gensym.Generator nameGen = new Gensym.Generator();

    Ip2 ip2;
    PanelController controller;
    Ip2ModelManager modelManager;
    VariableManager variableManager; // /\/

    Plan plan;
    ListOfPlanNode topNodes;
    ListOfPlanRefinement refinements;
    ListOfPlanIssue topIssues;
    ListOfPlanIssueRefinement issueRefinements;

    Map varToDclMap;
    Map itemToNodeMap = new HashMap(); // ActivityItem -> PlanNode

    Set<ActivityItem> topSelectedActivities = null;
    Set<ActivityItem> allSelectedActivities = null;

    PlanMaker(Ip2 ip2) {
        this.ip2 = ip2;
	this.controller = ip2.getController();
	this.modelManager = (Ip2ModelManager)ip2.getModelManager();
	this.variableManager = modelManager.variableManager; // /\/
    }

    Plan getPlan() {
	buildPlan();
	return plan;
    }

    Plan getPlan(ActivityItem topNode) {
        return getPlan(topNode, false);
    }

    Plan getPlanWithReportBack(ActivityItem topNode) {
        return getPlan(topNode, true);
    }

    Plan getPlan(ActivityItem topNode, boolean reportBack) {
        // /\/: This has been added late in the day and so is not done
        // as nicely as perhaps it might be.  Also, it should be
        // possible to specify a list or set of top-nodes to include.
        topSelectedActivities = Collections.singleton(topNode);
        allSelectedActivities = new HashSet<ActivityItem>();
        collectSubtrees(topSelectedActivities, allSelectedActivities);
        if (reportBack)
            for (ActivityItem item: allSelectedActivities)
                item.getAbout().ensureId();
        buildPlan();
        plan.setPlanIssues(null);
        plan.setWorldState((ListOfPatternAssignment)null);
        if (reportBack) {
            final Name ipcName = Name.valueOf(ip2.getAgentIPCName());
            PlanInspector pi = new PlanInspector(plan);
            pi.walkActivities(new TaskItemVisitor() {
                public void visit(TaskItem item, TaskItem parent,
                                  List children) {
                    item.setSenderId(ipcName);
                    item.setReportBack(YesNo.YES);
                    item.setRef(item.getId());
                }
            });
        }
        return plan;
    }

    private void collectSubtrees(Collection<? extends PNode> roots,
                                 Set<ActivityItem> result) {
        for (PNode r: roots) {
            result.add((ActivityItem)r);
            collectSubtrees(r.getChildren(), result);
        }
    }

    private boolean isSelected(PNode item) {
        return allSelectedActivities == null
            || allSelectedActivities.contains(item);
    }

    private Plan buildPlan() {
	plan = new Plan();
	topNodes = new LinkedListOfPlanNode();
	refinements = new LinkedListOfPlanRefinement();
	topIssues = new LinkedListOfPlanIssue();
	issueRefinements = new LinkedListOfPlanIssueRefinement();
	varToDclMap = new HashMap();

	List variables = getVariables(modelManager.getVarEnv());
	ListOfPlanVariableDeclaration varDcls =
	    makeVariableDeclarations(variables);
	Collect.extendMap(varToDclMap, variables, varDcls);

	ListOfConstrainer topConstraints = getTopLevelConstraints();
	ListOfPatternAssignment worldState = getWorldState();

	walkNodes();
	walkIssues();

	topConstraints.addAll(buildNonSiblingOrderings());

	Annotations annotations = modelManager.getAnnotations();

	if (!varDcls.isEmpty())
	    plan.setPlanVariableDeclarations(varDcls);
	if (!topIssues.isEmpty())
	    plan.setPlanIssues(topIssues);
	if (!issueRefinements.isEmpty())
	    plan.setPlanIssueRefinements(issueRefinements);
	if (!topNodes.isEmpty())
	    plan.setPlanNodes(topNodes);
	if (!refinements.isEmpty())
	    plan.setPlanRefinements(refinements);
	if (!topConstraints.isEmpty())
	    plan.setConstraints(topConstraints);
	if (!worldState.isEmpty())
	    plan.setWorldState(worldState);
	if (annotations != null && !annotations.isEmpty())
	    // Safe to just set because we soon deep-copy the plan.
	    plan.setAnnotations(annotations);

	plan = (Plan)new Pass2().copy(plan);

	return plan;

    }

    /**
     * Constructs a deep copy of a plan in which each Variable
     * has been replaced by its value, if it has a value, or else
     * by a PlanVariable.
     */
    class Pass2 extends ObjectCopier {
	public Object mapElement(Object obj) {
	    return obj instanceof Variable
		? mapVariable((Variable)obj)
		: copy(obj);
	}
	Object mapVariable(Variable var) {
	    Object val = var.getValue();
	    return val != null ? val : getPV(var);
	}
	PlanVariable getPV(Variable var) {
	    PlanVariableDeclaration dcl =
		(PlanVariableDeclaration)varToDclMap.get(var);
	    return dcl.getPlanVariable();
	}
    }

    Name genId(String base) {
	return nameGen.nextName(base);
    }

    List getVariables(Map varEnv) {
	// Here we start with all variables and keep only the ones in
	// a specific env, thus retaining the order in which they were
	// created.
	List variables = new LinkedList(Variable.getAllVariables());
	variables.retainAll(varEnv.values());
	// Now remove variables that already have values.
	for (Iterator i = variables.iterator(); i.hasNext();) {
	    Variable v = (Variable)i.next();
	    if (v.getValue() != null)
		i.remove();
	}
	return variables;
    }

    ListOfPlanVariableDeclaration makeVariableDeclarations
				     (List variables) {
	ListOfPlanVariableDeclaration result =
	    new LinkedListOfPlanVariableDeclaration();
	Collect.map(result, variables, new Function1() {
	    public Object funcall(Object arg) {
		Variable v = (Variable)arg;
		return new PlanVariableDeclaration(genId("var"), v);
	    }
	});
	return result;
    }

    ListOfConstrainer getTopLevelConstraints() {
	ListOfConstrainer result = new LinkedListOfConstrainer();
	result.addAll(variableManager.getConstraints());
	result.addAll(modelManager.getOtherConstraints());
	return result;
    }

    ListOfPatternAssignment getWorldState() {
	final List savable = Parameters.getList("plan-state-to-save");
	Map worldStateMap = modelManager.getWorldStateMap();
	return PatternAssignment.mapToAssignments
	    (savable.contains("*") // keep everything?
	     ? worldStateMap
	     : Collect.filter(new StableHashMap(),
			      worldStateMap,
			      new Predicate2() {
		   public boolean trueOf(Object key, Object value) {
		       LList k = (LList)key;
		       return k != Lisp.NIL
			   && savable.contains(k.car().toString());
		   }
	     }));
    }

    void walkIssues() {
	List issueItems = controller.getIssueAgenda().getItems();
	for (Iterator i = issueItems.iterator(); i.hasNext();) {
	    IssueItem item = (IssueItem)i.next();
	    if (item.getLevel() == 0)
		walkTopIssue(item);
	}
    }

    void walkTopIssue(IssueItem item) {
	Issue issue = (Issue)item.getAbout();
	PlanIssue pi = new PlanIssue(genId("issue"), issue);
	topIssues.add(pi);
	if (item.isExpanded()) {
	    buildIssueExpansion(item, pi);
	}
    }

    void buildIssueExpansion(IssueItem item, PlanIssue pi) {

	// Construct the IssueRefinement.
	PlanIssueRefinement ref = new PlanIssueRefinement();
	ref.setId(genId("issue-refinement"));
	issueRefinements.add(ref);

	// Connect the PlanIssue and the IssueRefinement
	pi.setExpansion(ref.getId());
	ref.setExpands(pi.getId());

	// Variables
	List variables = getVariables(item.getVarTable());
	if (!variables.isEmpty()) {
	    ListOfPlanVariableDeclaration varDcls =
		makeVariableDeclarations(variables);
	    Collect.extendMap(varToDclMap, variables, varDcls);
	    ref.setPlanVariableDeclarations(varDcls);
	}

	// Walk children
	List childItems = item.getChildren();
	ListOfPlanIssue childPlanIssues = new LinkedListOfPlanIssue();
	for (Iterator i = childItems.iterator(); i.hasNext();) {
	    IssueItem childItem = (IssueItem)i.next();
	    Issue childIssue = (Issue)childItem.getAbout();
	    PlanIssue childPlanIssue =
		new PlanIssue(genId(pi.getId().toString()),
			      childIssue);
	    childPlanIssues.add(childPlanIssue);
	    if (childItem.isExpanded()) {
		// Finally, the recursion.
		buildIssueExpansion(childItem, childPlanIssue);
	    }
	}
	if (!childPlanIssues.isEmpty()) 
	    ref.setPlanIssues(childPlanIssues);
    }

    void walkNodes() {
        if (topSelectedActivities != null) {
            for (ActivityItem item: topSelectedActivities) {
                walkTopNode(item);
            }
            return;
        }
	for (Iterator i = modelManager.getNodes().iterator(); i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    if (item.getLevel() == 0)
		walkTopNode(item);
	}
    }

    void walkTopNode(ActivityItem item) {
	PlanNode node = makePlanNode("node", item);
	topNodes.add(node);
	if (item.isExpanded()) {
	    buildExpansion(item, node);
	}
	else {
	    // There can be constraints even if it hasn't been expanded.
	    // Sometimes constraints are explicitly added, for instance
	    // to restrict resource use in I-Globe.
	    ListOfConstrainer constraints = getNodeConstraints(item);
	    if (!constraints.isEmpty()) {
		PlanRefinement ref = makePlanRefinement(node);
		ref.setConstraints(constraints);
		ref.setIsNotExpansion();
	    }
	}
    }

    PlanNode makePlanNode(String idBase, ActivityItem item) {
	Activity activity = (Activity)item.getAbout();
	PlanNode node = new PlanNode(genId(idBase), activity);
	itemToNodeMap.put(item, node);
	return node;
    }

    PlanNode getPlanNode(ActivityItem item) {
	PlanNode pn = (PlanNode)itemToNodeMap.get(item);
	Debug.expect(pn != null, "Can't find PlanNode for", item);
	return pn;
    }

    PlanNode getPlanNode(PNode n) {
	return getPlanNode((ActivityItem)n);
    }

    void buildExpansion(ActivityItem item, PlanNode node) {
	// /\/: Confusing because an ActivityItem is also sometimes
	// called a node in the model manager.

	// Construct the refinement that will describe the expansion
	// of the node.
	PlanRefinement ref = makePlanRefinement(node);

	// Variables
	List variables = getVariables(item.getVarTable());
	if (!variables.isEmpty()) {
	    ListOfPlanVariableDeclaration varDcls =
		makeVariableDeclarations(variables);
	    Collect.extendMap(varToDclMap, variables, varDcls);
	    ref.setPlanVariableDeclarations(varDcls);
	}

	// Walk children
	List childItems = item.getChildren();
	ListOfPlanNode childNodes = new LinkedListOfPlanNode();
	for (Iterator i = childItems.iterator(); i.hasNext();) {
	    ActivityItem childItem = (ActivityItem)i.next();
	    PlanNode childNode = makePlanNode(node.getId().toString(),
					      childItem);
	    childNodes.add(childNode);
	    if (childItem.isExpanded()) {
		// Finally, the recursion.
		buildExpansion(childItem, childNode);
	    }
	}
	checkItemToNodeMap(childItems, childNodes);
	if (!childNodes.isEmpty()) 
	    ref.setPlanNodes(childNodes);

	// Constraints
	ListOfConstrainer constraints = getNodeConstraints(item);
	if (!constraints.isEmpty())
	    ref.setConstraints(constraints);

    }

    PlanRefinement makePlanRefinement(PlanNode node) {
	PlanRefinement ref = new PlanRefinement();
	ref.setId(genId("refinement"));
	refinements.add(ref);
	// Connect the node and the refinement.
	node.setExpansion(ref.getId());
	ref.setExpands(node.getId());
	return ref;
    }

    void checkItemToNodeMap(List childItems, List childNodes) {
	Debug.expect(childItems.size() == childNodes.size());
	for (Iterator i = childItems.iterator(),
		      n = childNodes.iterator();
	     i.hasNext();) {
	    ActivityItem item = (ActivityItem)i.next();
	    PlanNode node = (PlanNode)n.next();
	    Debug.expect(getPlanNode(item) == node);
	}
    }

    ListOfConstrainer getNodeConstraints(ActivityItem item) {
	// /\/: For now, we have to (re)construct the constraints.
	ListOfConstrainer constraints = new LinkedListOfConstrainer();
	// Orderings
	constraints
	    .addAll(buildChildSiblingOrderings(item));
	// World-state constraints
	constraints
	    .addAll(modelManager.getNodeConditions(item));
	constraints
	    .addAll(buildConstraints(S_WORLD_STATE,
				     S_EFFECT,
				     modelManager.getNodeEffects(item)));
	constraints
	    .addAll(Collect.ensureList(modelManager
				       .getOtherNodeConstraints(item)));
	constraints
	    .addAll(Collect.ensureList(modelManager
				       .getNodeTimeConstraints(item)));
	fixConstraintRanges(item, constraints);
	return constraints;
    }

    List buildConstraints(Symbol type, Symbol relation, List assigns) {
	List result = new LinkedList();
	if (assigns == null) return result;
	for (Iterator i = assigns.iterator(); i.hasNext();) {
	    PatternAssignment pv = (PatternAssignment)i.next();
	    result.add(new Constraint(type, relation, Lisp.list(pv)));
	}
	return result;
    }

    void fixConstraintRanges(PNode node, ListOfConstrainer constraints) {
	for (ListIterator<Constrainer> i = constraints.listIterator()
		 ; i.hasNext();) {
	    Constrainer c = i.next();
	    if (c instanceof Constraint) {
		Constraint con = (Constraint)c;
		if (con.getConstraintRange() != null) {
		    // The constraint contains a constraint-range
		    i.set(fixConstraintRange(node, con));
		}
	    }
	}
    }

    Constraint fixConstraintRange(PNode node, Constraint c) {
	// The ConstraintRange refers to node-ends using the names
	// in the node's nameToChildMap.  We have to change it to
	// use the names / ids we've given the PlanNodes.
	Constraint result = new Constraint(c.getType(), c.getRelation(), null);
	LListCollector params = new LListCollector();
	for (Object param: c.getParameters()) {
	    if (param instanceof ConstraintRange) {
		ConstraintRange range = (ConstraintRange)param;
		ConstraintRange r = (ConstraintRange)Util.clone(range);
                if (range.getFrom() != null) {
                    PNodeEnd fromNE = node.refToTimePoint(range.getFrom());
                    PlanNode from = getPlanNode(fromNE.getNode());
                    Name fromName = fromNE.getNode() == node
                        ? Name.valueOf("self")
                        : from.getId();
                    r.setFrom(new NodeEndRef(fromNE.getEnd(), fromName));
                }
		PNodeEnd toNE = node.refToTimePoint(range.getTo());
		PlanNode to = getPlanNode(toNE.getNode());
		Name toName = toNE.getNode() == node
		    ? Name.valueOf("self")
		    : to.getId();
		r.setTo(new NodeEndRef(toNE.getEnd(), toName));
		params.add(r);
	    }
	    else
		params.add(param);
	}
	result.setParameters(params.contents());
	return result;
    }

    // /\/: Really shouldn't have to build the orderings here
    // rather than somewhere in the MM or CM.

    List buildChildSiblingOrderings(ActivityItem item) {
	// Returns only orderings between siblings.
	// Note that we don't have to record any links between
	// parent and child, even though we could use "self"
	// as a node-id to express them, because we'll always
	// add links when the plan is loaded to ensure that
	// begin_of parent is before  begin_of each child, and
	// that end_of each child is before end_of their parent
	// (and of course that begin_of each node is before
	// the end_of that node).
	// Any "unusual" parent-child link, such as b/self --> e/2,
        // will be implied by the standard links and so needn't
	// be explicitly expressed.
	final List orderings = new LinkedList();
	walkChildOrderings(item, new OrdWalker() {
	    public void visit(PNodeEnd before, PNodeEnd after) {
		if (before.getNode().isSiblingOf(after.getNode()))
		    orderings.add(makeOrdering(before, after));
	    }
	});
	return orderings;
    }

    // This is a bit tricky.  We need to build the non-sibgling orderings
    // in the same order they'd be in if the Plan were loaded into a
    // clean model-manager and then turned into a new Plan, and that can
    // be different from the order we get if we just iterate over
    // the nodes that are now in the MM.  [/\/: Not absolutely clear
    // why, but probably because when a node is expanded, the children
    // are added at the end of the node list rather than inserted
    // after their parent, but when we load a Plan, children are
    // placed right after their parent.]  So we have to walk the
    // nodes in an order that visits children right after their
    // parent.  This doesn't matter for correctness, but it makes
    // it easier to compare plans.  For one thing, it depends on
    // the order of top-level nodes, but not on the order in
    // which they were expanded.  It also makes it so that loading
    // a Plan then getting a new Plan from the MM doesn't change
    // the order.

    List buildNonSiblingOrderings() {
	// Called after all PlanNodes have been created and there
	// is a complete itemToNodeMap.
	final List orderings = new LinkedList();
	modelManager.walkTopNodes(new Proc() {
	    public void call(Object node) {
		final ActivityItem item = (ActivityItem)node;
                if (!isSelected(item))
                    return;
		walkNodeOrderings(item, new OrdWalker() {
		    public void visit(PNodeEnd before, PNodeEnd after) {
			if (isSelected(before.getNode())
                              && isSelected(after.getNode())
                              && isNonSiblingOrdering(before, after))
			    orderings.add(makeOrdering(before, after));
		    }
		});
		modelManager.walkNodeChildren(item, this);
	    }
	});
	return orderings;
    }

    boolean isNonSiblingOrdering(PNodeEnd before, PNodeEnd after) {
	// The method name isn't quite right. /\/
	// We know the orderings are all from successor lists.
	// The method should return true iff the ordering has to be top-level.
	ActivityItem b = (ActivityItem)before.getNode();
	ActivityItem a = (ActivityItem)after.getNode();
	// Skip all links between parent node-ends and children,
	// because they're either default links that will be added
	// automatically, or else they're implied by such links.
	if (b == a.getParent() || a == b.getParent())
	    return false;
	// Skip links between siblings **except at the top level**.
	if (b.getParent() != null && b.isSiblingOf(a))
	    return false;
	// Oherwise, approve the ordering
	return true;
    }

    /*
     * Ordering utilities
     */

    interface OrdWalker {
	public void visit(PNodeEnd before, PNodeEnd after);
    }

    void walkChildOrderings(PNode node, OrdWalker w) {
	for (Iterator i = node.getChildren().iterator(); i.hasNext();) {
	    PNode child = (PNode)i.next();
	    walkNodeOrderings(child, w);
	}
    }

    void walkNodeOrderings(PNode node, OrdWalker w) {
	walkEndOrderings(node.getBegin(), w);
	walkEndOrderings(node.getEnd(), w);
    }

    void walkEndOrderings(PNodeEnd nodeEnd, OrdWalker w) {
	// Skip link from begin to end of the same node.
	PNodeEnd skip = null;
	if (nodeEnd.getEnd() == End.BEGIN)
	    skip = nodeEnd.getNode().getEnd();
	for (Iterator i = nodeEnd.getSuccessors().iterator(); i.hasNext();) {
	    PNodeEnd after = (PNodeEnd)i.next();
	    if (after != skip)
		w.visit(nodeEnd, after);
	}
    }

    Ordering makeOrdering(PNodeEnd before, PNodeEnd after) {
	ActivityItem beforeItem = (ActivityItem)before.getNode();
	ActivityItem afterItem = (ActivityItem)after.getNode();
	Name beforeName = beforeItem.isParentOf(afterItem)
	    ? Name.valueOf("self")
	    : getPlanNode(beforeItem).getId();
	Name afterName = afterItem.isParentOf(beforeItem)
	    ? Name.valueOf("self")
	    : getPlanNode(afterItem).getId();
	if (beforeName.toString().equals("self")
	    || afterName.toString().equals("self"))
	    System.out.println
		(before + " " + getPlanNode(beforeItem).getId() +
		 " ---> " + after + " " + getPlanNode(afterItem).getId());
	return new Ordering
	    (new NodeEndRef(before.getEnd(), beforeName),
	     new NodeEndRef(after.getEnd(), afterName));
    }

}
