/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Feb 19 04:27:14 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.domain.ObjectProperty;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.context.*;

/**
 * Maintains an context-layered, object-oriented view of a world-state.
 */
public class ObjectWorld {

    public static final Symbol S_TRUE = Symbol.intern("true");

    protected ContextMap objectTable = new ContextHashMap(100);

    public ObjectWorld() {
    }

    public ObjectWorld(Map worldState) {
	addState(worldState);
    }

    public Object getPropValue(Symbol prop, Object obj) {
	Object value = _getPropValue(prop, obj);
	return value instanceof LLQueue
	    ? ((LLQueue)value).contents()
	    : value;
    }

    protected Object _getPropValue(Symbol prop, Object obj) {
	Map properties = (Map)objectTable.get(obj);
	if (properties != null) {
	    Object value = properties.get(prop);
	    if (value != null) 
		return value;
	}
	return ObjectView.NO_VALUE;
    }

    public void setPropValue(Symbol prop, Object obj, Object value) {
	Map properties = (Map)objectTable.get(obj);
	if (properties == null) {
	    properties = new ContextHashMap(5);
	    objectTable.put(obj, properties);
	}
	properties.put(prop, value);
    }

    public void addPropValueIfNew(Symbol prop, Object obj, Object v) {
	LLQueue exists = (LLQueue)_getPropValue(prop, obj);
	if (exists == null || exists == ObjectView.NO_VALUE) {
	    exists = new LLQueue();
	    exists.add(v);
	    setPropValue(prop, obj, exists);
	}
	else if (!exists.contains(v)) {
	    exists.add(v);
	}
    }

    public void removePropValueElement(Symbol prop, Object obj, Object v) {
	LLQueue exists = (LLQueue)_getPropValue(prop, obj);
	if (exists == null)
	    throw new ConsistencyException
		("Property value (" + prop + " " + obj + ") " +
		 "does not exist");
	Debug.expect(exists.remove(v),
		     "No value " + v + " to remove from " +
		     " the value of property (" + prop + " " + obj + ")");
    }

    public void addState(Map delta) {
	for (Iterator i = delta.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    LList pattern = (LList)e.getKey();
	    Object value = e.getValue();
	    if (isPropertyPattern(pattern))
		setPropValue((Symbol)pattern.get(0), pattern.get(1), value);
	    else if (value == S_TRUE && isDistributedSetPattern(pattern))
		addPropValueIfNew((Symbol)pattern.get(0), pattern.get(1),
				  pattern.get(2));
	}
    }

    public void deleteState(Map delta) {
	for (Iterator i = delta.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    LList pattern = (LList)e.getKey();
	    Object value = e.getValue();
	    if (isPropertyPattern(pattern))
		setPropValue((Symbol)pattern.get(0), pattern.get(1),
			     ObjectView.NO_VALUE);
	    else if (value == S_TRUE && isDistributedSetPattern(pattern))
		removePropValueElement((Symbol)pattern.get(0), pattern.get(1),
				       pattern.get(2));
	}
    }

    public boolean isPropertyPattern(LList pattern) {
	// (prop object) = value
	return pattern.car() instanceof Symbol
	    && pattern.cdr() != Lisp.NIL
	    && pattern.cdr().cdr() == Lisp.NIL;
    }

    public boolean isDistributedSetPattern(LList pattern) {
	// (prop object value) = TRUE
	return pattern.car() instanceof Symbol
	    && pattern.length() == 3;
    }

}
