/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu May 11 02:54:00 2006 by Jeff Dalton
 * Copyright: (c) 2005 - 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import java.awt.Component;
import java.awt.Container;
import java.awt.BorderLayout;
import java.awt.Frame;
import java.awt.Dimension;
import java.awt.event.*;

import java.io.File;
import java.io.IOException;

import javax.swing.text.StyleConstants;
import javax.swing.text.Document;
import javax.swing.text.AbstractDocument;
import javax.swing.text.DocumentFilter;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.AttributeSet;
import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLDocument;

// import java.io.IOException;

import java.util.*;

import ix.iface.util.HtmlTableEditorPane;
import ix.iface.util.HtmlStringWriter;
import ix.iface.util.HtmlTableWalker;

import ix.iface.util.IconImage;
import ix.iface.util.IFUtil;
import ix.iface.util.ToolFrame;
import ix.iface.util.KeyValueTable; // for comparator

import ix.icore.plan.Plan;

import ix.icore.domain.PatternAssignment;
import ix.icore.domain.ObjectProperty;
import ix.icore.domain.Constraint;

import ix.icore.process.event.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

public class ObjectWhiteboard implements StateViewer {

    Ip2 ip2;
    Ip2ModelManager mm;		// shouldn't need to have this /\/

    WhiteboardFrame frame;

    SortedMap nameToViewMap = null;
    ObjectView view = null;	// the current ObjectView
    SortedSet objects;		// the objects we're currently viewing
    List properties;		// the properties to display

    static boolean hasBeenVisible = false; // N.B. one value for all views /\/

    // Maps to Facts
    Map fromPattern = new HashMap();
    Map fromTD = new HashMap();

    StructuralEquality structEqual = new StructuralEquality();

    final Object BLANK = new UniqueObject("BLANK");

    /** Create a viewer for the indicated agent. */
    public ObjectWhiteboard(Ip2 ip2) {
	this(ip2, null);
    }

    /** Create a viewer for the indicated agent with references
	to any existing views. */
    protected ObjectWhiteboard(Ip2 ip2, SortedMap nameToViews) {
	// This method is called with a non-null nameToViews
	// when creating a new view window from an existing view.
	this.ip2 = ip2;
	this.mm = (Ip2ModelManager)ip2.getModelManager();
	this.nameToViewMap = nameToViews;
	this.frame = new WhiteboardFrame();

	if (nameToViewMap == null) {
	    // This is the first one created.
	    String viewDir = Parameters.getParameter("object-view-directory");
	    if (viewDir != null)
		loadViews(viewDir);
	}
	else
	    frame.populateSelectViewMenu();

	ip2.getModelManager().addProcessStatusListener(this);
	ip2.addResetHook(new ResetHook());

    }

    public void setVisible(boolean v) {
	frame.setVisible(v);
	if (!hasBeenVisible) {
	    hasBeenVisible = true;
	    whenFirstMadeVisible();
	}
    }

    void whenFirstMadeVisible() {

	List views = Parameters.getList("initial-whiteboards");
	if (!views.isEmpty() && views.get(0).equals(":all"))
	    views = new ArrayList(nameToViewMap.keySet());

	if (!views.isEmpty()) {
	    Iterator i = views.iterator();
	    String view_1 = (String)i.next();
	    Debug.noteln("Setting first view to", view_1);
	    setObjectView(view_1);
	    while (i.hasNext()) {
		String view_i = (String)i.next();
		Debug.noteln("Creating view for", view_i);
		ObjectWhiteboard w = 
		    new ObjectWhiteboard(ip2, nameToViewMap);
		w.setVisible(true);
		w.setObjectView(view_i);
	    }
	}
    }

    class ResetHook implements Runnable {
	public void run() {
	    Debug.noteln("Resetting", ObjectWhiteboard.this);
	    reset();
	}
    }

    /* Methods from the StateViewer interface */

    public Component getView(PanelFrame panelFrame) {
	return frame;
    }

    public void reset() {
	if (view != null)
	    setObjectView(view);
    }

    /* ProcessStatusListener methods */

    /** Ignored by this viewer. */
    public void statusUpdate(ProcessStatusEvent event) { }

    /** Ignored by this viewer. */
    public void newBindings(ProcessStatusEvent event, Map bindings) { }

    public void stateChange(ProcessStatusEvent event, Map delta) {
	if (view == null)
	    return;
	// See if there are any new, relevant objects.
	// /\/: If an object changes its type (and potentially other
	// properties as well), we might want to remove its row.
	// Also some objects might now belong in the table because
	// some other object's properties changed.  So we can't
	// rally get away with looking only in the delta for new objects.
	Set newObjects = view.getNewObjects(delta, objects, stateLookupFn);
	if (!newObjects.isEmpty()) {
	    Debug.noteln("New objects to view", newObjects);
	    // /\/: Various ways of adding rows to the HTMLDocument
	    // were tried, but there didn't seem to be any good way
	    // to add a row at the end of the table.  So we do
	    // this instead:
	    loadViewContentsSavingState();
	}
	else {
	    // Just Change any values that need changing.
	    processStateDelta(delta, false);
	}
    }

    public void stateDeletion(ProcessStatusEvent event, Map delta) {
	// /\/: If an object is included only because of its type,
	// and the type constraint is deleted, then the object's
	// row probably should be removed from the table.  Other
	// deletions might also call for that, or even for objects
	// to be added.  But we won't handle any of those cases for now.
	if (view == null)
	    return;
	processStateDelta(delta, true);
    }

    void processStateDelta(Map delta, boolean isDeletion) {
	for (Iterator i = delta.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    LList pattern = (LList)e.getKey();
	    Object value = e.getValue();
	    Fact f = getFact(pattern);
	    // If there's no fact for the pattern, then it's not
	    // one we're displaying and we can ignore the change.
	    if (f != null) {
		if (isDeletion)
		    f.delete();
		else
		    f.changeValue(value);
	    }
	}
    }

    /* End of ProcessStatusListener methods */

    /* End of StateViewer methods */

    public ObjectView getObjectView() {
	return view;
    }

    public ObjectView getObjectView(String name) {
	ObjectView view = (ObjectView)nameToViewMap.get(name);
	if (view != null)
	    return view;
	else
	    throw new IllegalArgumentException
		("There is no view named " + Strings.quote(name));
    }

    public void setObjectView(String name) {
	setObjectView(getObjectView(name));
    }

    public void setObjectView(ObjectView view) {
	this.view = view;
	this.properties = Collect.ensureList(view.getProperties());
	loadViewContents();
	frame.noticeView(view);
    }

    void addObject(Symbol name) {
	view.addObject(name);
	loadViewContentsSavingState();
    }

    void addObject(Symbol name, Object type) {
	LList pattern = Lisp.list(ObjectView.TYPE, name);
	PatternAssignment pv = new PatternAssignment(pattern, type);
	mm.addConstraint
	    (new Constraint("world-state", "effect", Lisp.list(pv)));
    }

    /** Reload the current view without losing any editing the user
	may have done. */
    void loadViewContentsSavingState() {
	// /\/: Use a "finally"?
	Map saved = saveFactEditState();
	loadViewContents();
	if (saved.isEmpty())
	    Debug.noteln("No fact state needed to be restored.");
	else
	    restoreFactEditState(saved);
    }

    /** Reload the current view from the agent's current world-state. */
    void loadViewContents() {
	Map worldState = mm.getWorldStateMap();
	this.objects = view.getInitialObjects(worldState, stateLookupFn);
	String htmlText = makeHtmlTable(worldState);
	frame.setHtml(htmlText);
	analyseHTML(worldState, frame.getHTMLDocument()); // clears facts 1st
    }

    private Function1 stateLookupFn = new Function1() {
	public Object funcall(Object a) {
	    LList pattern = (LList)a;
	    return mm.getWorldStateValue(pattern);
	}
    };

    /** Adds views from the indicated directory. */
    void loadViews(String viewDir) {
	Debug.noteln("Reading views from", viewDir);
	try {
	    // /\/: Add views or replace?  For now, it's replace.
	    nameToViewMap = readViews(viewDir);
	    frame.populateSelectViewMenu();
	}
	catch (Throwable t) {
	    Debug.displayException
		("Problem reading views from directory " + viewDir, t);
	    if (!Parameters.isInteractive())
		throw new RethrownException(t);
	}
    }

    /**
     * Tries to read an ObjectView from each file in the indicated
     * directory that might contain one.  If any view does not have
     * a name, it is given the name of the corresponding file.
     * View names must be unique within the directory.
     *
     * Returns a map from names to {@link ObjectView}s.
     */
    public SortedMap readViews(String directoryName) {
	// Find all the files in the directory that might contain plans.
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	Map filesToViews = fsm.readAllObjects(ObjectView.class, directoryName);
	SortedMap views = new TreeMap();
	for (Iterator i = filesToViews.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    File f = (File)e.getKey();
	    ObjectView v = (ObjectView)e.getValue();
	    String name = v.getName();
	    if (name == null) {	// if no name, give it the file's
		name = Strings.beforeLast(".", f.getName());
		v.setName(name);
	    }
	    if (views.get(name) != null)
		throw new IllegalArgumentException
		    ("There are two views named " + Strings.quote(name) +
		     " in " + directoryName);
	    views.put(name, v);
	}
	return views;
    }

    /**
     * Constructs an HTML table that describes the current view.
     * However, property values are not included; instead each
     * "td" element that will contain a value instead contains
     * the text "BLANK".  The correct values are filled-in
     * later by calling analyseHTML after the HTML has been
     * installed as a document.
     *
     * <p>For a description of the HTML's structure, see
     * {@link HtmlTableWalker}.</p>
     *
     * @see WhiteboardFrame#setHtml(String htmlText)
     * @see #analyseHTML(Map state, HTMLDocument doc)
     */
    String makeHtmlTable(Map worldState) {
	HtmlStringWriter html = new ViewHtmlStringWriter();
	// Table
	html.tag("table", "border=\"1\" cellspacing=\"0\"");
	html.newLine();
	// Top row
	html.tag("tr");
	html.tagged("th", view.getObjectHeader("Object"));
	for (Iterator i = properties.iterator(); i.hasNext();) {
	    ObjectProperty prop = (ObjectProperty)i.next();
	    html.tagged("th", prop.getName().toString());
	}
	html.end("tr");
	html.newLine();
	// For each object
	for (Iterator oi = objects.iterator(); oi.hasNext();) {
	    Object obj = oi.next();
	    html.tag("tr");
	    // Make the object a th, but specify "" attributes
	    // to avoid the default (which would change the bgcolor).
	    html.tagged("th", "", obj.toString());
	    // For each property
	    for (Iterator pi = properties.iterator(); pi.hasNext();) {
		ObjectProperty prop = (ObjectProperty)pi.next();
		// Object value = getPropValue(prop, obj, worldState);
		// String text = view.write(prop, value);
		String text = "BLANK";// filled-in later
		html.tagged("td", text);
	    }
	    html.end("tr");
	    html.newLine();
	}
	// End table
	html.end("table");
	html.newLine();
	// Finished.
	String htmlString = html.toString();
	html.close();
	return htmlString;
    }

    Object getPropValue(ObjectProperty prop, Object obj, Map state) {
	LList pattern = Lisp.list(prop.getName(), obj);
	Object value = state.get(pattern);
	return value != null ? value : ObjectView.NO_VALUE;
    }

    /**
     * Sets up connections between the world state and the HTML.
     * Finds each part of an HTML table that represent the value
     * of an object property, creates a {@link Fact} to connect
     * the world-state pattern with the HTML element, and asks
     * the Fact to write in the current value of the property.
     *
     * <p>The HTML must have the same structure as that created
     * by {@link #makeHtmlTable(Map worldState)}.</p>
     */
    void analyseHTML(Map state, HTMLDocument doc) {
	clearFacts();
	new HtmlAnalyser(state).walkHTML(doc);
    }

    /**
     * The variation on {@link HtmlTableWalker} used by the
     * {@link #analyseHTML(Map state, HTMLDocument doc) analyseHTML}
     * method.
     */
    class HtmlAnalyser extends HtmlTableWalker {

	Map state;

	HtmlAnalyser(Map state) {
	    super(objects, properties);
	    this.state = state;
	}

	public void walkTD(Element td, Object rowItem, Object colItem) {
	    analyseTD(td, rowItem, (ObjectProperty)colItem);
	}

	void analyseTD(Element td, Object obj, ObjectProperty prop) {
	    Debug.expectSame(HTML.Tag.TD, getTag(td));
	    Fact f = new Fact(obj, prop, td);
	    f.recordYourself();

	    // Now put in the value's text.
	    Object value = getPropValue(prop, obj, state);
	    f.changeValue(value);
	}

    }

    /** Returns the HTML.Tag of an Element. */
    HTML.Tag getTag(Element elt) {
	return (HTML.Tag)elt.getAttributes()
	                     .getAttribute(StyleConstants.NameAttribute);
    }

    /** The type of HtmlStringWriter used when constructing the
	HTML table that describes an ObjectView. */
    static class ViewHtmlStringWriter extends HtmlStringWriter {

	public ViewHtmlStringWriter() {
	    super();
	    setDefaultAttributes
		("tr", "align=\"left\"");
	    setDefaultAttributes
		("th", "align=\"center\" bgcolor=\"#99ccff\"");
	}

    }

    /**
     * What we need to know about world-state entries ("facts").
     * Used to connect world-state patterns with table cells.
     */
    class Fact {
	LList pattern;
	Object value = BLANK;
	Object obj;
	ObjectProperty prop;
	Element td;
	boolean wasEdited = false;

	Fact(Object obj, ObjectProperty prop, Element td) {
	    this.obj = obj;
	    this.prop = prop;
	    this.pattern = Lisp.list(prop.getName(), obj);
	    this.td = td;
	}

	void recordYourself() {
	    fromPattern.put(pattern, this);
	    fromTD.put(td, this);
	}

	void edited() {
	    Debug.noteln("Changing the value of", pattern);
	    wasEdited = true;
	}

	Object getValue() {
	    Debug.expect(value != BLANK, "Blank value for", this);
	    Debug.expect(value != null, "Null value for", this);
	    return value;
	}

	Object getValueFromTable() {
	    String text = getTableText();
	    Debug.noteln("Text for " + pattern, Strings.quote(text));
	    try {
		return view.read(prop, text, value);
	    }
	    catch (Exception e) {
		throw new RethrownException
		    ("Problem with the value for " + pattern + ":", e);
	    }
	}

	String getTableText() {
	    Document doc = td.getDocument();
	    int start = td.getStartOffset(), end = td.getEndOffset();
	    try {
		// /\/: -1 because we know there's an extra newline.
		return doc.getText(start, end-start -1);
	    }
	    catch (BadLocationException e) {
		throw new RethrownException(e);
	    }
	}

	void changeValue(Object newValue) {
	    value = newValue;
	    // Now change the document.
	    String text = view.write(prop, newValue);
	    writeTableText(text);
	}

	void writeTableText(String text) {
	    HTMLDocument doc = (HTMLDocument)td.getDocument();
	    int start = td.getStartOffset();
	    int end = td.getEndOffset();
	    Debug.noteln("TD TEXT", Strings.quote(text));
	    try {
		frame.changingDocumnent = true;
		// /\/: Tried:
		//   doc.setInnerHTML(td, text);
		// and
		//   doc.setOuterHTML(td, "<td>" + text + "</td>");
		// One of those nicer methods ought to work, but
		// setting the inner HTML adds an extra line, and setting
		// the outer would require that we find the new td element.

		// /\/: It's surprising that it has to be
		// end-start-1 rather then end-start.
		// Also see editIsAllowed in HtmlTableEditorPane.
		doc.replace(start, end-start-1, text, null);
	    }
	    catch (BadLocationException e) {
		throw new RethrownException(e);
	    }
//  	    catch (IOException e) {
//  		throw new RethrownException(e);
//  	    }
	    finally {
		frame.changingDocumnent = false;
		wasEdited = false;
	    }
	    // Debug.noteln("getText() ==>\n", frame.htmlPane.getText());
	}

	void delete() {
	    changeValue(ObjectView.NO_VALUE);
	}

	public String toString() {
	    return "Fact[" + pattern + "=" + value + "]";
	}

    }

    Fact getFact(LList pattern) {
	return (Fact)fromPattern.get(pattern);
    }

    Fact getFact(Element elt) {
	return (Fact)fromTD.get(elt);
    }

    Collection getFacts() {
	return fromPattern.values();
    }

    Plan getFactsAsPlan() {
	// N.B. Doesn't handle deletions
	Map state = new TreeMap(new KeyValueTable.PatternObjectComparator());
	for (Iterator i = getFacts().iterator(); i.hasNext();) {
	    Fact f = (Fact)i.next();
	    if (f.wasEdited)
		throw new IllegalStateException
		    ("You must commit changes before sending the view.");
	    Object value = f.getValue();
	    if (value != ObjectView.NO_VALUE)
		state.put(f.pattern, value);
	}
	Plan p = new Plan();
	p.setWorldState(state);
	return p;
    }

    void clearFacts() {
	fromPattern.clear();
	fromTD.clear();
    }

    void submitChanges() {
	Map changes = new StableHashMap();
	Map deletions = new StableHashMap();
	// Collect changes and deletions
	for (Iterator i = getFacts().iterator(); i.hasNext();) {
	    Fact f = (Fact)i.next();
	    if (f.wasEdited) {
		Object oldValue = f.getValue();
		Object newValue = f.getValueFromTable();
		if (structEqual.equal(oldValue, newValue))
		    f.wasEdited = false; // Not really a change
		else if (newValue == ObjectView.NO_VALUE)
		    deletions.put(f.pattern, oldValue);
		else
		    changes.put(f.pattern, newValue);
	    }
	}
	// Submit deletions
	if (!deletions.isEmpty()) {
	    Debug.noteln("Deletions:", deletions);
	    for (Iterator i = deletions.entrySet().iterator(); i.hasNext();) {
		PatternAssignment pv =
		    new PatternAssignment((Map.Entry)i.next());
		mm.deleteConstraint
		    (new Constraint("world-state", "effect", Lisp.list(pv)));
	    }
	}
	// Submit changes
	if (changes.isEmpty())
	    Debug.noteln("No changes to submit");
	else {
	    Debug.noteln("Submitting changes", changes);
	    mm.handleEffects(PatternAssignment.mapToAssignments(changes));
	}
    }

    Map saveFactEditState() {
	Map result = new StableHashMap();
	for (Iterator i = getFacts().iterator(); i.hasNext();) {
	    Fact f = (Fact)i.next();
	    if (f.wasEdited)
		result.put(f, f.getTableText());
	}
	return result;
    }

    void restoreFactEditState(Map savedState) {
	// At this point we have new Fact objects and want to
	// give them some state from the old Facts.
	Debug.noteln("Restoring state", savedState);
	for (Iterator i = savedState.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Fact f = (Fact)e.getKey();
	    String text = (String)e.getValue();
	    Debug.expect(f.wasEdited, "should have been edited", f);
	    Fact newFact = (Fact)fromPattern.get(f.pattern);
	    if (newFact == null)
		Debug.noteln("No new Fact matching", f);
	    else {
		newFact.writeTableText(text);
		newFact.edited();
	    }
	}
    }

    /** An ObjectWhiteboard's GUI. */
    class WhiteboardFrame extends ToolFrame implements ActionListener {

	Container contentPane;

	JMenu viewMenu;
	JMenu selectViewMenu;

	HtmlTableEditorPane htmlPane = new WhiteboardEditorPane();
	JScrollPane htmlScroll;

	JButton submitButton;
	JButton newRowButton;

	boolean changingDocumnent = false;

	WhiteboardFrame() {
	    super(ip2.getAgentDisplayName() + " Object View Table");

	    setIconImage(IconImage.getIconImage(this));
	    setJMenuBar(makeMenuBar());

	    contentPane = getContentPane();

	    htmlScroll = new JScrollPane(htmlPane);
	    htmlScroll.setBorder
		(BorderFactory.createTitledBorder("Object View"));

	    contentPane.add(htmlScroll, BorderLayout.CENTER);

	    JPanel buttons = makeButtonPanel();
	    contentPane.add(buttons, BorderLayout.SOUTH);

	    pack();
	    setSize(500, 300);
	    validate();
	}

	JMenuBar makeMenuBar() {
	    JMenuBar bar = new JMenuBar();
	    // File menu
	    JMenu fileMenu = new JMenu("File");
	    bar.add(fileMenu);
	    fileMenu.add(IFUtil.makeMenuItem("New View Window", this));
	    fileMenu.add(IFUtil.makeMenuItem("Send View", this));
	    fileMenu.add(IFUtil.makeMenuItem("Close", this));
	    // View menu
	    JMenu viewMenu = new JMenu("View");
	    bar.add(viewMenu);
	    viewMenu.add(selectViewMenu = new JMenu("Select"));
	    viewMenu.add(IFUtil.makeMenuItem("Redisplay", this));
	    selectViewMenu.setEnabled(false);
	    return bar;
	}

	void populateSelectViewMenu() {
	    JMenu menu = selectViewMenu;
	    menu.setEnabled(false);
	    menu.removeAll();
	    for (Iterator i = nameToViewMap.keySet().iterator()
		     ; i.hasNext();) {
		String name = (String)i.next();
		JMenuItem item = IFUtil.makeMenuItem(name, this);
		item.setActionCommand("Select View");
		menu.add(item);
	    }
	    menu.setEnabled(true);
	}

	JPanel makeButtonPanel() {
	    JPanel panel = new JPanel(); 	// defaults to FlowLayout
	    submitButton = IFUtil.makeButton("Commit changes", false, this);
	    newRowButton = IFUtil.makeButton("New Row", false, this);
	    panel.add(submitButton);
	    panel.add(newRowButton);
	    return panel;
	}

	/** Action interpreter */
	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("I-Plan Option Tool frame action:", command);
	    if (command.equals("Close")) {
		setVisible(false);
	    }
	    else if (command.equals("New View Window")) {
		new ObjectWhiteboard(ip2, nameToViewMap).setVisible(true);
	    }
	    else if (command.equals("Send View")) {
		sendView();
	    }
	    else if (command.equals("Select View")) {
		JMenuItem item = (JMenuItem)e.getSource();
		String viewName = item.getText();
		Debug.noteln("Select view", viewName);
		setObjectView(viewName);
	    }
	    else if (command.equals("Redisplay")) {
		if (view != null)
		    loadViewContents();
	    }
	    else if (command.equals("Commit changes")) {
		submitChanges();
		submitButton.setEnabled(false);
	    }
	    else if (command.equals("New Row")) {
		displayNewRowDialog();
	    }
	    else
		throw new ConsistencyException
		    ("Nothing to do for " + command);
	}

	void displayNewRowDialog() {
	    NewRowDialog dialog =
		new NewRowDialog
		    (this, ip2.getAgentDisplayName() + " Define Object", view);
	    dialog.show();
	    if (dialog.getObject() == null)	// cancelled?
		return;
	    Symbol object = Symbol.intern(dialog.getObject());
	    Object type = dialog.getType();
	    if (type.equals("")) {
		// No type was specified
		addObject(object);
	    }
	    else {
		// The type has to be one of the ones in the view.  /\/
		Debug.expect(view.getTypes().contains(type));
		addObject(object, type);
	    }
	}

	HTMLDocument getHTMLDocument() {
	    return (HTMLDocument)htmlPane.getDocument();
	}

	void setHtml(String htmlText) {
	    // Debug.noteln("HTML text:\n", htmlText);
	    htmlPane.setDocText(htmlText);
	    // htmlPane.describeHtmlDocument(); // debug
	    // Debug.noteln("getText() ==>\n", htmlPane.getText());
	}

	void noticeView(ObjectView view) {
	    // This doesn't show the change: border.setTitle(title);
	    htmlScroll.setBorder
		(BorderFactory.createTitledBorder(view.getName()));
	    submitButton.setEnabled(false);
	    newRowButton.setEnabled(true);
	    setTitle(ip2.getAgentDisplayName() + " " +
		     view.getName() + " View");
	}

	void sendView() {
	    if (view == null)
		throw new IllegalStateException
		    ("No view has been selected.");
	    LList pat = 
		Lisp.list(Symbol.intern("load-plan"), getFactsAsPlan());
	    ip2.getFrame().getSendPanelVisible()
		.initActivity(new ix.icore.Activity(pat));
	}

	class WhiteboardEditorPane extends HtmlTableEditorPane {

	    protected void editedElement(Element td) {
		Debug.noteln("Editing TD", td);
		Debug.expectSame(HTML.Tag.TD, getTag(td));
		Fact f = getFact(td);
		Debug.expect(f != null, "unmapped element", td);
		if (!changingDocumnent) {
		    f.edited();
		    submitButton.setEnabled(true);
		}
	    }
	    
	}

    }

    /**
     * A modal dialog that requests the information needed to
     * create a new row in the table.
     */
    static class NewRowDialog extends JDialog implements ActionListener {

	JTextField nameText = new JTextField(15);
	JComboBox typeChoice = new JComboBox();

	String object = null;
	Object type = null;

	NewRowDialog(Frame owner, String title, ObjectView view) {
	    super(owner, title, true); // modal
	    JPanel contents = new JPanel();
	    setContentPane(contents);
	    contents.setLayout(new BorderLayout());
	    contents.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
	    // Object name
	    contents.add(wrap(nameText, view.getObjectHeader("Object")),
			 BorderLayout.WEST);
	    // Object type
	    String blank = "";
	    typeChoice.addItem(blank);
	    if (view.getTypes() != null) {
		for (Iterator i = view.getTypes().iterator(); i.hasNext();) {
		    typeChoice.addItem(i.next());
		}
	    }
	    typeChoice.setSelectedItem(blank);
	    contents.add(wrap(typeChoice, "Type"), BorderLayout.EAST);
	    // The buttons
	    JButton cancel = IFUtil.makeButton("Cancel", this);
	    JButton ok = IFUtil.makeButton("OK", this);
	    Box buttons = Box.createHorizontalBox();
	    buttons.add(Box.createHorizontalGlue());
	    buttons.add(ok);
	    buttons.add(Box.createRigidArea(new Dimension(5, 0)));
	    buttons.add(cancel);
	    buttons.add(Box.createHorizontalGlue());
	    contents.add(buttons, BorderLayout.SOUTH);
	    pack();
	    setLocationRelativeTo(owner);
	}

	String getObject() {
	    return object;	// null means the dialog was cancelled
	}

	Object getType() {
	    return type;
	}

	JPanel wrap(Component c, String title) {
	    JPanel p = new JPanel();
	    p.add(c);
	    p.setBorder(BorderFactory.createTitledBorder(title));
	    return p;
	}

	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("ObjectWhiteboard.NewRowDialog command", command);
	    if (command.equals("OK")) {
		String maybe = nameText.getText().trim();
		if (maybe.equals("")) {
		    Util.displayAndWait(this, "Empty object name");
		    return;
		}
		object = maybe;
		type = typeChoice.getSelectedItem();
		dispose();
	    }
	    else if (command.equals("Cancel")) {
		dispose();
	    }
	    else
		throw new ConsistencyException
		    ("Don't know what to do for", command);
	}

    }

}

// Issues:
// * Right-click in cell to see the value class?  Otherwise, it's
//   difficult to tell whether something is a symbol, string, etc.
//
// * Should we keep rows that would otherwise be deleted
//   if anything on them has been edited?
//
// * To Do (not in any particular order):
//
// - Add a row if a new object gets a type that's being viewed.
//
// - Remove a row if an object loses a type.  (This already happens
//   for a redisplay() and other cases that call loadViewContents,
//   but not for state-deletions or for changes to a different type.)
//
//   (Also the user can pick up new objects, and drop rows for ones
//   that have lost a viewed type, by re-selecting the view.)
//
// - Let the user add a row for a specified object. 
//   Send the type assertion to the model?  (It may not be obvious
//   what the type should be, so maybe the user should have to select
//   from a menu.)
//
// - Distinguish between no value and the empty string (or symbol?).
//   [Done]
//
// - Let the user edit views.  (The XML Tree Editor can be used initially.)
//
// - Let the user specify a comparator that will determine the row order.
//
// - Let the user decide what to do if a state update would affect
//   a cell the user has edited (but not yet submitted).
//
// - Deal with option changes.
//
// - Provide an "undo" for unsubmitted changes to the table.
//   (Perhaps also allow going further back, across submissions.)
