/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Aug 13 22:15:38 2007 by Jeff Dalton
 * Copyright: (c) 2001 - 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import java.awt.Toolkit;
import java.awt.Component;
import java.awt.Container;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.Dimension;

import java.awt.event.*;

import java.net.URL;

import java.util.*;

import ix.iface.ui.util.UIUtil;

import ix.ichat.ChatFrame;
import ix.ispace.ISpaceTool;
import ix.itest.SendPanel;

import ix.iplan.IPlanTool;
import ix.iplan.IPlanOptionTool;

import ix.icore.*;
import ix.icore.event.AgentNameEvent;
import ix.icore.domain.Domain;
import ix.icore.domain.Refinement;
import ix.icore.domain.Constraint;
import ix.icore.domain.PatternAssignment;
import ix.icore.plan.Plan;
import ix.iview.*;
import ix.iface.domain.DomainParser;
import ix.iface.domain.DomainWriter;
import ix.iface.util.LogoPanel;
import ix.iface.util.IconImage;
import ix.iface.util.ToolController;
import ix.iface.util.ToolManager;
import ix.iface.util.CatchingActionListener;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.XML;

import ix.iface.ui.HelpFrame;
// import ix.iface.ui.AboutFrame;  // we don't (yet?) use one /\/

/**
 * The main frame of the I-P2 user interface.
 */
public class Ip2Frame extends PanelFrame implements ActionListener {

    protected Container contentPane;

    protected Ip2AboutFrame aboutFrame;	// should just be class AboutFrame /\/

    // /\/: Do we need these fields?
    // /\/: Should it be productViewer or stateViewer?
    protected AgendaViewer activityViewer;
    protected AgendaViewer issueViewer;
    protected StateViewer stateViewer;
    protected AnnotationViewer annotationViewer;

    protected Ip2 ip2;

    protected JPanel logoPanel;

    protected JMenuItem undoMenuItem;

    public Ip2Frame(Ip2 ip2) {
	super(ip2);
	this.ip2 = ip2;
    }

    protected void setUp() {
	setIconImage(IconImage.getIconImage(this));
	activityViewer = ip2.activityViewer;
	issueViewer = ip2.issueViewer;
	stateViewer = ip2.stateViewer;

	contentPane = getContentPane();

	setSize(500, 400);	// preliminary
	setJMenuBar(makeMenuBar());
	// Layout manager defaults to BorderLayout.
	// contentPane.setLayout(new FlowLayout()); // probably not too clever

	// Package issue viewer
	JPanel issuePanel = new JPanel();
	issuePanel.add((Component)issueViewer);
	JScrollPane issueScroll = new JScrollPane(issuePanel);
	issueScroll.setBorder(BorderFactory.createTitledBorder("Issues"));

	// Package activity viewer
	JPanel activityPanel = new JPanel();
	activityPanel.add((Component)activityViewer);
	JScrollPane activityScroll = new JScrollPane(activityPanel);
	activityScroll
	    .setBorder(BorderFactory.createTitledBorder("Activities"));

	// Package state viewer
	// stateViewer.setBorder(BorderFactory.createTitledBorder("inner"));
	JPanel statePanel = new JPanel();
	statePanel.add((Component)stateViewer);
	JScrollPane stateScroll = new JScrollPane(statePanel);
	stateScroll.setBorder(BorderFactory.createTitledBorder("State"));

	// Put the viewers in nested split panes.

	// JSplitPane split1 = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
	// JSplitPane split2 = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
	split1.setTopComponent(issueScroll);
	split1.setBottomComponent(split2);
	split2.setTopComponent(activityScroll);
	split2.setBottomComponent(stateScroll);
	// split.setDividerLocation(0.80); // the one after pack() matters /\/
	contentPane.add(split1);

	// Add something that identifies the system etc.
	logoPanel = ip2.makeLogoPanel();
	contentPane.add(logoPanel, BorderLayout.SOUTH);

    }

    // Needs to be accessible in becomeVisible().
    JSplitPane split1 = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
    JSplitPane split2 = new JSplitPane(JSplitPane.VERTICAL_SPLIT);

    protected void becomeVisible() { // sep method so can do it last
	// It is expected that this is called only once.

	if (Parameters.haveParameter("frame-size")) {
	    Dimension size = Parameters.getDimension("frame-size");
	    pack();		// needed for divider adjust /\/
	    setSize(size);
	}
	else
	    setReasonableSize();

	// Seem to need this here, rather than, say, later.
	super.validate();

	// Adjust the split pane dividers.
	// /\/: If we nest split1 in split2 instead of split2 in split1,
	// the divider locations are bizarre.
	split1.setDividerLocation(0.33);
	split2.setDividerLocation(0.55);

	// Finally, make it visible
	setVisible(true);
    }

    protected void setReasonableSize() {
	// pack() is mostly to get a good width, so remember the
	// original height because we'll probably want to keep it.
	Dimension origSize = getSize();
	int origHeight = (int)Math.round(origSize.getHeight());

	pack();

	// Keep the frame from being too tall.
	// The way to do it is somewhat empirical ... /\/
	Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
	Dimension size = getSize();
	int width = (int)Math.round(size.getWidth());
	int height = (int)Math.round(size.getHeight());
	int maxHeight = (int)((7.0 / 8.0) * screenSize.getHeight());
	int scrollWidth = 30;	// scrollbar will appear /\/
	int desiredHeight = Math.max(Math.min(height, maxHeight),
				     origHeight);
	if (height != desiredHeight)
	    setSize(new Dimension(width + scrollWidth, desiredHeight));
    }

    /**
     * Handles a change in the panel's symbol-name.
     */
    public void symbolNameChanged(AgentNameEvent e) {
	super.symbolNameChanged(e); // sets frame title
	// Change the panel that identifies the system etc.
	contentPane.remove(logoPanel);
	logoPanel = ip2.makeLogoPanel();
	contentPane.add(logoPanel, BorderLayout.SOUTH);	
	validate();
    }

    protected void addIp2Tools() {
	addTool(new ToolController("Domain Editor") {
	    public Object createTool() {
		return makeDomainEditor(ip2, ip2.getDomain());
	    }
	});
	addTool(new ToolController("I-Space") {
	    public Object createTool() {
		return new ISpaceTool(ip2);
	    }
	});
	addTool(new ToolController("I-Var") {
	    public Object createTool() {
		return new IVarTool(ip2);
	    }
	});
	addTool(new ToolController("I-Plan") {
	    public Object createTool() {
		return new IPlanTool(ip2);
	    }
	});
	if (ip2.showOptions) {
	    addTool(new ToolController("I-Plan Options") {
		public Object createTool() {
		    return new IPlanOptionTool(ip2);
		}
	    });
	}
	addTool(new ToolController("Messenger") {
	    public Object createTool() {
		return new ChatFrame
		    (ip2, ip2.getAgentDisplayName() + " Messenger");
	    }
	});
	addTool(new ToolController("HTML Viewer") {
	    public Object createTool() {
		return makeHTMLViewer();
	    }
	});
	addTool(new ToolController("Timeline Viewer") {
	    public Object createTool() {
		return new TimelineViewer(ip2);
	    }
	});
    }

    protected HTMLViewer makeHTMLViewer () {
	// /\/: Separate method because it's not a one-liner and
	// a subclass might want to explicitly create tool controllers
	// rather than inherit the above addIp2Tools() method.
	// /\/: Have to specify an "original help page"
	String page = Parameters.getParameter("initial-html-viewer-home");
	// /\/: We have to get the URL here, because the HelpFrame
	// class [probably?] won't do it in the right way.
	if (page != null) {
	    URL url = XML.toURL(page);
	    if (url != null)
		return new HTMLViewer(url);
	    else
		Util.displayAndWait(Ip2Frame.this, new String[] {
		    "Can't find " + page,
		    "The HTML viewer will display the default",
		    "page instead."
		});
	}
	return new HTMLViewer("ix-info.html");
    }

    /*
     * HTML Viewer
     */

    public HTMLViewer ensureHTMLViewer() {
	return (HTMLViewer)
	    toolManager.findToolElseError("HTML Viewer")
	        .ensureToolVisible();
    }

    class HTMLViewer extends HelpFrame {
	HTMLViewer(String file) {
	    super(file);
	    setup();
	}
	HTMLViewer(URL url) {
	    super(url);
	    setup();
	}
	protected void setup() {
	    setTitle(ip2.getAgentDisplayName() + " HTML Viewer");
	    setIconImage(IconImage.getIconImage("ip2-help-icon.gif"));
	}
    }

    /*
     * Help
     */

    protected HelpFrame helpFrame = null;

    public HelpFrame ensureHelpFrame() {
	return ensureHelpFrame("ip2-help.html");
    }

    public HelpFrame ensureHelpFrame(String file) {
	if (helpFrame == null)
	    helpFrame = new Ip2HelpFrame(file);
	else
	    helpFrame.displayURL(UIUtil.resourceURL(file)); // /\/
	helpFrame.setVisible(true);
	return helpFrame;
    }

    // /\/: This class should not have to exist.
    class Ip2HelpFrame extends HelpFrame {
	Ip2HelpFrame(String file) {
	    super(file);
	    setTitle(ip2.getAgentDisplayName() + " Help");
	    setIconImage(IconImage.getIconImage("ip2-help-icon.gif"));
	}
    }

    /*
     * About I-X Process Panel
     */

    public void ensureAboutIP2() {
	ensureAboutFrame(ip2.getAgentTypeName(), makeAboutText());
    }

    public void ensureAboutFrame(String subject, String aboutText) {
	String title = ip2.getAgentDisplayName() +" About " + subject;
	if (aboutFrame == null)
	    aboutFrame = new Ip2AboutFrame(title);
	else
	    aboutFrame.setTitle(title);
	aboutFrame.setText(aboutText);
	aboutFrame.setVisible(true);
    }

    // /\/: This class should not have to exist.
    // /\/: And we shouldn't need to use a TextAreaFrame.
    // Unfortunately, the text area in an ix.iface.ui.AboutFrame
    // is private, so we can't set its caret position.
    public static class Ip2AboutFrame extends TextAreaFrame {
	public Ip2AboutFrame(String title) {
	    super(0, 0, title, new String[]{"Cancel"});
	    setEditable(false);
	}
	protected void finishFrame() {
	    // Doesn't make visible, unlike super method.
	    frame.pack();
	    frame.setSize(400, 300);
	    frame.validate();
	}
	public void setText(String text) {
	    Debug.noteln("About text", text);
	    super.setText(text);
	    setCaretPosition(0); // go to start of about text
	}
    }

    public String makeAboutText() {
	List about = new LinkedList();
	// I-X version
	about.add("I-X version: " + ix.Release.version
		  + ", release date " + ix.Release.date);
	// I-DE version
	about.add("I-DE version: " + ix.iview.IDERelease.version
		  + ", release date " + ix.iview.IDERelease.date);
	// Agent class
	about.add("Agent class: " + ip2.getClass().getName());
	// Startup time
	about.add("Started running: " +
		  new ISODateFormat()
		      .formatDateTime(ip2.getAgentStartupDate()));
	// Java version
	about.add("Java version: " + System.getProperty("java.version")
		  + " from " + System.getProperty("java.vendor"));
	// Interactive?
	if (Parameters.isInteractive())
	    about.add("Interactive.");
	// Parameters
	Hashtable props = Parameters.getParameters();
	SortedSet keys = new TreeSet(props.keySet());
	about.add("");
	if (keys.isEmpty()) {
	    about.add("Parameters: none");
	}
	else {
	    about.add("Parameters: ");
	    for (Iterator i = keys.iterator(); i.hasNext();) {
		String key = (String)i.next();
		String val = (String)props.get(key);
		about.add("   " + key + " = " + val
			  + (Parameters.usedParameter(key)
			     ? "" : " [unused]"));
	    }
	}
	// Let the agent add things
	ip2.addAboutInfo(about);
	// Let the controller add things
	ip2.getController().addAboutInfo(about);
	// Let the file-syntax manager add things
	XML.fileSyntaxManager().addAboutInfo(about);
	// Concat into string
	return Strings.joinLines(about);
    }

    /*
     * About Syntax
     */

    public void ensureAboutSyntax() {
	ensureAboutFrame("Syntax", makeSyntaxText());
    }

    public String makeSyntaxText() {
	List syntax = new LinkedList();
	List issueSyntax = ip2.getController().getIssueAgenda()
	                      .getSyntaxList();
	List activitySyntax = ip2.getController().getActivityAgenda()
	                         .getSyntaxList();
	// Issue syntax
	addHandlerSyntax(syntax, "Issue", issueSyntax);
	syntax.add("");
	// Activity Syntax
	addHandlerSyntax(syntax, "Activity", activitySyntax);
	syntax.add("");
	// Expansion syntax
	addExpansionSyntax(syntax);
	// Concatenate
	return Strings.joinLines(syntax);
    }

    private void addHandlerSyntax(List syntax, String type, List patterns) {
	if (patterns.isEmpty())
	    syntax.add(type + " patterns from handlers: none");
	else {
	    syntax.add(type + " patterns from handlers:");
	    for (Iterator i = sortedStrings(patterns).iterator();
		 i.hasNext();) {
		syntax.add("   " + i.next());
	    }
	}
    }

    private SortedSet sortedStrings(List patterns) {
	SortedSet s = new TreeSet();
	for (Iterator i = patterns.iterator(); i.hasNext();) {
	    LList pat = (LList)i.next();
	    s.add(PatternParser.unparse(pat));
	}
	return s;
    }

    private void addExpansionSyntax(List syntax) {
	// Get a list of all the expansion patterns.
	List patterns =
	    (List)Collect.map(new LinkedList(),
			      ip2.getDomain().getRefinements(),
			      Fn.accessor(Refinement.class, "getPattern"));
	if (patterns.isEmpty())
	    syntax.add("Activity patterns from domain refinements: none");
	else {
	    syntax.add("Activity patterns from domain refinements:");
	     for (Iterator i = sortedStrings(patterns).iterator();
		 i.hasNext();) {
		syntax.add("   " + i.next());
	    }
	}
    }

    /*
     * Menu bar
     */

    protected JMenuBar makeMenuBar() {
	JMenuBar bar = new JMenuBar();

	// File menu
        JMenu fileMenu  = new JMenu("File");
	bar.add(fileMenu);
	fileMenu.add(makeMenuItem("Load Plan/State From ..."));
	fileMenu.add(makeMenuItem("Save Plan/State As ..."));
	fileMenu.add(makeMenuItem("Send Plan"));
	fileMenu.add(makeMenuItem("Send State"));
	fileMenu.addSeparator();
	fileMenu.add(makeMenuItem("Load Domain From ..."));
	fileMenu.add(makeMenuItem("Save Domain As ..."));
	fileMenu.add(makeMenuItem("Send Domain"));
	fileMenu.addSeparator();
        JMenu resetMenu = new JMenu("Reset");
	fileMenu.add(resetMenu);
	fileMenu.addSeparator();
	fileMenu.add(makeMenuItem("Exit"));

	// File -> Reset menu
	resetMenu.add(makeMenuItem("Reset All"));
	resetMenu.add(makeMenuItem("Reset All But State"));

	// New menu
        JMenu newMenu = new JMenu("New");
	bar.add(newMenu);
	newMenu.add(makeMenuItem("New Issue"));
	newMenu.add(makeMenuItem("New Activity"));
	newMenu.add(makeMenuItem("New Constraint"));
	newMenu.add(makeMenuItem("New Annotation"));

        // Edit menu
        JMenu editMenu = new JMenu("Edit");
        bar.add(editMenu);
        undoMenuItem = makeMenuItem("Undo");
        editMenu.add(undoMenuItem);

	// Tools menu
	bar.add(toolManager.getToolsMenu());
	addIp2Tools();

	// Help menu
        JMenu helpMenu = new JMenu("Help");
	bar.add(helpMenu);
	helpMenu.add(makeMenuItem("Help"));
	helpMenu.add(makeMenuItem("About " + ip2.getAgentTypeName()));
	helpMenu.add(makeMenuItem("About Syntax"));

	bar.add(Box.createHorizontalGlue()); // let later items move right

	// Test menu
	// Note that some tests might have already been added.
	bar.add(testMenu);
	if (testMenu.getMenuComponentCount() == 0)
	    testMenu.setEnabled(false); 	// initially
	ip2.addTestMenuItems();

	return bar;
    }

    /*
     * Action interpreter
     */

    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("I-P2 action:", command);
	if (command.equals("Exit")) {
	    if (Util.dialogConfirms(ip2.getFrame(),
                                    "Are you sure you want to exit?"))
		ip2.exit();
	}
	else if (command.equals("Reset All")) {
	    if (Util.dialogConfirms
                  (ip2.getFrame(),
                   "Are you sure you want to reset everything?"))
	      ip2.reset();
	}
	else if (command.equals("Reset All But State")) {
	    if (Util.dialogConfirms
                  (ip2.getFrame(), 
                   "Are you sure you want to reset all but state?"))
		ip2.resetAllButState();
	}
	else if (command.equals("Load Plan/State From ...")) {
	    ip2.loadPlan();
	}
	else if (command.equals("Save Plan/State As ...")) {
	    ip2.savePlanAs();
	}
	else if (command.equals("Send Plan")) {
	    Plan plan = ip2.getModelManager().getPlan();
	    Activity activity =
		new Activity(Lisp.list(LoadPlanHandler.S_LOAD_PLAN,
				       plan));
	    getSendPanelVisible().initActivity(activity);
	}
	else if (command.equals("Send State")) {
	    Ip2ModelManager mm = (Ip2ModelManager)ip2.getModelManager();
	    Map state = mm.getWorldStateMap();
	    Plan plan = new Plan();
	    plan.setWorldState(state);
	    Activity activity =
		new Activity(Lisp.list(LoadStateHandler.S_LOAD_STATE,
				       plan));
	    getSendPanelVisible().initActivity(activity);
	}
	else if (command.equals("Load Domain From ...")) {
	    DomainParser.loadDomain(this, ip2.getDomain());
	}
	else if (command.equals("Save Domain As ...")) {
	    DomainWriter.saveDomain(this, ip2.getDomain());
	}
	else if (command.equals("Send Domain")) {
	    Domain domain = ip2.getDomain();
	    Activity activity =
		new Activity
		    (Lisp.list(LoadDomainHandler.S_LOAD_DOMAIN,
			       domain));
	    getSendPanelVisible().initActivity(activity);
	}
	else if (command.equals("New Issue")) {
	    issueViewer.getNewItemFromUser();
	}
	else if (command.equals("New Activity")) {
	    activityViewer.getNewItemFromUser();
	}
	else if (command.equals("New Constraint")) {
	    newConstraint();
	}
	else if (command.equals("New Annotation")) {
	    annotationViewer.getNewItemFromUser();
	}
        else if (command.equals("Undo")) {
            try {
                ip2.getIp2ModelManager().undo();
            }
            catch (UndoException.NoFurtherUndo nfu) {
                Util.displayAndWait(this, "Cannot undo further.");
            }
        }
	else if (command.equals("Help")) {
	    ensureHelpFrame();
	}
	else if (command.startsWith("About ")
		 && command.endsWith(ip2.getAgentTypeName())) {
	    ensureAboutIP2();
	}
	else if (command.equals("About Syntax")) {
	    ensureAboutSyntax();
	}
	else {
	    throw new UnsupportedOperationException
		("Nothing to do for " + command);
	}
    }

    protected void newConstraint() {
	String spec = JOptionPane.
	    showInputDialog(this, "Enter constraint as pattern=value");
	if (spec == null)
	    return;		// user cancelled
	String[] parts = Strings.breakAtFirst("=", spec);
	String pat = parts[0].trim();
	String val = parts[1].trim();
	if (val.equals(""))
	    val = "true";
	LList pattern = Lisp.elementsFromString(pat);
	Object value = Lisp.readFromString(val);
	Constraint c =
	    new Constraint("world-state", "effect",
			   Lisp.list(new PatternAssignment(pattern, value)));
	ip2.getModelManager().addConstraint(c);
    }

    /*
     * Right-button popup for agenda items.
     */

    public void adjustAgendaItemPopup(AbstractAgendaItemPopupMenu menu,
				      AgendaItem item) {
	// Do nothing except in subclasses.
    }

}

// Issues:
// * Perhaps a more abstract name, such as PanelUserInterface, instead of
//   Ip2Frame.
