/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu May 29 18:13:59 2008 by Jeff Dalton
 * Copyright: (c) 2001 - 2008, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.ip2.event.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.icore.process.ProcessModelManager;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * Adds expansion actions to activities based on refinements in
 * the agent's domain.
 */
public class ExpandHandler extends ActivityHandler
                           implements DomainListener {

    protected Ip2 ip2;

    protected Agenda actAgenda;
    protected Ip2ModelManager modelManager;

    public ExpandHandler(Ip2 ip2) {
        // super("Expand actions");
        super("Refine actions");
	this.ip2 = ip2;
	this.actAgenda = ip2.getController().getActivityAgenda();
	this.modelManager = ip2.getIp2ModelManager();
	ip2.getDomain().addDomainListener(this); // clean up /\/
    }

    public boolean appliesTo(AgendaItem item) {
	// This stops expand actions being considered if e.g. a
	// new refinement comes along and a node is already expanded.
	// However, it doesn't stop expand actions being added to
	// a node that is just about to be expanded, such as
	// when loading a plan.
	return !item.isExpanded();
    }

    public void addHandlerActions(AgendaItem item) {
	item.addItemListener(new ItemListener());
	// Expansion and manual primitive options from refinements
	// in the domain.
	for (Iterator i = ip2.getDomain().getRefinements().iterator();
	     i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    MatchEnv e = matchRefinement(item, r);
	    if (e == null) {
		continue;
	    }
//  	    else if (Collect.isEmpty(r.getNodes())
//  		     && Collect.isEmpty(r.getConstraints())) {
//  		// When there are no subnodes or constraints, the
//  		// refinement name still specifies a way of doing
//  		// it manually.
//  		item.addAction(new HandlerAction.Manual(r.getName()));
//  	    }
	    else {
		// Add action to expand using the refinement.
		item.addAction(makeExpandAction(item, r, e));
	    }
	}
    }

    /**
     * Factory method that returns an instance of a suitable subclass
     * of {@link ExpandAction}.
     *
     * @param item  the AgendaItem that would be expanded
     * @param r     the Refinement that would be used to expand it
     * @param env   the result of matching the Refinement's pattern
     *               against the item's
     */
    protected ExpandAction makeExpandAction(AgendaItem item,
					    Refinement r, 
					    MatchEnv env) {
	return new ExpandAction(item, r, env);
    }

    protected MatchEnv matchRefinement(AgendaItem item, Refinement r) {
	if (modelManager.getAdviceManager().isUsableRefinement(r))
	    return Matcher.match(r.getPattern(), item.getPattern());
	else
	    return null;
    }

    protected class ItemListener extends AgendaItemAdapter {
	public ItemListener() { }
	public void agendaItemEdited(AgendaItemEvent e) {
	    AgendaItem item = (AgendaItem)e.getSource();
	    // The item may have been expanded.
	    if (item.isExpanded())
		ip2.getController()
		    .checkActionValidity(ExpandHandler.this, e);
	}
    }

    public void refinementAdded(RefinementEvent event) {
	Refinement r = event.getRefinement();
	Debug.noteln("Expand Handler noting new ", r);
	// Arrange for calls to reviseHandlerActions as needed.
	ip2.getController().reconsiderHandler(this, r);
    }

    public void reviseHandlerActions(AgendaItem item, Object reason) {
	// See if the activity (item) now has a new way to expand.
	Refinement r = (Refinement)reason;
	MatchEnv env = matchRefinement(item, r);
	if (env == null) {
	    return;
	}
//  	else if (Collect.isEmpty(r.getNodes())
//  		 && Collect.isEmpty(r.getConstraints())) {
//  	    Debug.noteln("New manual primitive for", item);
//  	    item.insertAction(new HandlerAction.Manual(r.getName()));
//  	}
	else {
	    Debug.noteln("New expansion for", item);
	    item.insertAction(new ExpandAction(item, r, env));
	}
    }

    public HandlerAction makeManualExpandAction(AgendaItem item,
						Refinement r) {
	return new ManualExpandAction(item, r);
    }

    /**
     * Handles an activity by expanding it into subactivities
     * as specified by a refinement schema.
     */
    public class ExpandAction extends HandlerAction {

	AgendaItem item;
	Refinement refinement;
	MatchEnv env;

	ListOfConstraint filters;
	List filterEnvs = Collections.EMPTY_LIST;

	ExpandAction(AgendaItem item, Refinement refinement, MatchEnv env) {
	    this.item = item;
	    this.refinement = refinement;
	    this.env = env;
	    this.filters = refinement.getFilterConstraints();
	    // this.shortDescription = "Expand using " + refinement.getName();
            this.shortDescription = "Refine using " + refinement.getName();
	}

	public boolean isStillValid() {
	    // Not if item has been expanded.
	    if (item.isExpanded())
		return false;
	    // Yes iff the refinement still matches the item pattern,
	    // given the current Variable values.
	    // /\/: Do we want everything matchRefinement does,
	    // or should we just match the patterns?
	    env = matchRefinement(item, refinement);
	    return env != null;
	}

	public boolean isReady() {
	    return !item.isExpanded() // ?? /\/
		&& (filters.isEmpty() || !filterEnvs.isEmpty());
	}

	public void computeStatus() {
	    if (!filters.isEmpty())
		filterEnvs = modelManager.evalFilters(filters, env);
	}

	public ActionUnreadyReason getUnreadyReason() {
	    if (item.isExpanded())
		return new SimpleUnreadyReason(this, "Already expanded");
	    List satisfiable = modelManager.testFilters(filters, env);
	    return new UnsatisfiedConditionsUnreadyReason
		(this, filters, satisfiable, env);
	}

	public void handle(AgendaItem item) {
	    Debug.expect(isReady(),
	        this + " asked to handle " + item + " when not ready");

	    Debug.expect(!item.isExpanded(), "already expanded", item);

	    // That an option's been selecte implies that execution
	    // has begun.  /\/: But subnodes must be put in first
	    // or they will get the wrong status as they are added
	    // (because parent will be executing and orderings won't
	    // yet be in so it will look like all preconditions have
	    // been met).
	    Debug.expect(item == this.item);
	    do_expansion(item);
	    Debug.expect(item.getStatus() == Status.POSSIBLE);
	    item.fireAgendaItemEdited(); // /\/
	    setSatusAfterExpansion(item);
	}

	protected void setSatusAfterExpansion(AgendaItem item) {
	    // /\/: We know any conds have been satisfied -- otherwise
	    // we wouldn't have let the refinement be used -- and so
	    // the begin end of the activity can be COMPLETE.  And
	    // if there are no children, the whole activity can be
	    // COMPLETE.
	    item.setStatus
		(Collect.isEmpty(item.getChildren())
		 ? Status.COMPLETE 
		 : Status.EXECUTING);
	}

	protected void do_expansion(AgendaItem item) {
	    Debug.noteln("Expanding using", refinement);
	    // Take the most complete env
	    MatchEnv expandEnv = getInstantiationEnv();
	    Debug.noteln("Using env", expandEnv);
	    Debug.noteln("Filters", filters);
	    // Instantiate the refinement, create subnodes, and
	    // install orderings.  item.getRefinement() is then
	    // an instantiated copy.
	    item.expandOneLevel(refinement, expandEnv);
	    // /\/: Since activity fields aren't context-layered, it's
	    // not clear how we should handle recording the expansion
	    // refinement name.
	    // ((ActivityItem)item).recordExpansionRefinementName();
	    // Make any Variable bindings specified by expandEnv.
	    modelManager.applyEnv(expandEnv);
	    // Make sure we respect the ways the filter conds
	    // might bind now that the refinement has been
	    // instantiated.
	    if (!filters.isEmpty())
		reevaluateFilters(item);
	    // Add the item's children to this agenda.
	    // This adds a time constraint between the ends of each child.
	    for (Iterator i = Collect.iterator(item.getChildren());
		 i.hasNext();) {
		AgendaItem child = (AgendaItem)i.next();
		actAgenda.addItem(child);
	    }
	    // Add issues from the refinement to the issue agenda.
	    List issues = item.getRefinement().getIssues();
	    for (Iterator i = Collect.iterator(issues); i.hasNext();) {
		Issue issue = (Issue)i.next();
		ip2.getController().addIssue(issue);
	    }
	    // Add the orderings as time constraints
	    // /\/: We will eventually have to add time constraints
	    // as orderings as well, though the Refinement getOrderings()
	    // might take care of that.
	    // /\/: Refinement getConstraints() ought to already have
	    // the orderings as time constraints?  Or not - because
	    // we don't want them explicitly in the plan.
	    ListOfOrdering ords = item.getRefinement().getOrderings();
	    modelManager
		.addOrderingsAsTimeConstraints(item, ords);
	    // Add constraints from the refinement.
	    // /\/: At present, they are assumed to be at begin or end
	    // of self (depending on type + relation), hence apply to
	    // the activity being expanded.
	    List constraints = item.getRefinement().getConstraints();
	    if (constraints != null) {
		modelManager.addConstraints(item, constraints);
	    }
	    modelManager.getTPNManager().describeTPN(Debug.out);
	}

	protected MatchEnv getInstantiationEnv() {
	    // Returns an env that must be used.  If there's only
	    // one branch in the filterEnvs, we can use it; otherwise,
	    // we have to go back to the env from matching the
	    // refinement's pattern against the activity's.
	    if (!filters.isEmpty() && filterEnvs.size() == 1)
		return (MatchEnv)filterEnvs.get(0);
	    else
		return env;
	}

	protected void reevaluateFilters(AgendaItem item) {
	    // Get filter conds from the instantiated refinement.
	    ListOfConstraint conds =
		item.getRefinement().getFilterConstraints();
	    // Don't need to pass an env this time.
	    // The model manager returns the live branches.
	    List branches = modelManager.reevaluateFilters(conds);
	    Debug.expect(!branches.isEmpty(),
			 "Cannot match filter conditions " + conds);
	    // The branches tell us all the ways we could satisfy
	    // all the filters by binding.  Each branch is a way to
	    // satisy all the filters.
	}

    }

    /**
     * Handles an item by expanding it into subitems as specified
     * explicitly by the user.
     */
    private class ManualExpandAction extends ExpandAction {

	ManualExpandAction(AgendaItem item, Refinement refinement) {
	    super(item, refinement, new MatchEnv());
	    this.shortDescription = "Expand as below";
	}

	// /\/: We know manual expansions don't add any conditions
	// to the activity, and that conditions are added only by
	// expansion -- things that happen to be true in I-P2 at
	// present but may stop being true in the future.

	public void handle(AgendaItem item) {
	    do_expansion(item);
	    if (item.getStatus() == Status.POSSIBLE)
		item.setStatus(Collect.isEmpty(item.getChildren())
			       ? Status.COMPLETE : Status.EXECUTING);
	}

    }

}
