/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed May  7 17:53:16 2008 by Jeff Dalton
 * Copyright: (c) 2001, 2004, 2006 - 2008, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.icore.process.PNodeEnd;
import ix.icore.domain.ListOfConstraint;

import ix.util.*;
import ix.util.match.*;

/**
 * Generates HandlerActions that let a user do nothing or
 * make an {@link AgendaItem} complete.
 */
public class CompletionHandler extends SimpleCompletionHandler {

    Ip2 ip2;
    Ip2ModelManager modelManager;

    public CompletionHandler(Ip2 ip2) {
	super();
	this.ip2 = ip2;
	this.modelManager = ip2.getIp2ModelManager();
    }

    public void addHandlerActions(AgendaItem item) {
	// Standard options that all items have.
	// The ActivityAgenda.OtherReportsActivity wantsActionsFrom
	// method must allow actions from this class.  /\/
	item.addAction(new NoAction(item));
	item.addAction(new DoneAction(item));
	item.addAction(new NotApplicableAction(item));
    }

    public static class NoAction extends HandlerAction.NoAction {

	// /\/: The AgendaItem is needed because getActionDescription
	// may need to call makeDescription.
	AgendaItem item;

	public NoAction(AgendaItem item) {
	    // this.shortDescription = "No Action";
	    this.shortDescription = makeDescription(item);
	    this.item = item;
	}

	// /\/: Is this ever called any more [02 Oct 06]?
	// It will probably confuse a lot of things (such as logging)
	// if there are "handlings" that aren't supposed to count.
	public void handle(AgendaItem item) {
	    // N.B. no status change.
	    Debug.noteln("No action for", item);
	    // Have to clear the handledBy field.
	    Debug.expect(item.getHandledBy() == this);
	    item.setHandledBy(null);
	}

	public String getActionDescription() {
	    // /\/: If we're reloading viewers, the current description
	    // may not be right.
	    return shortDescription = makeDescription(item);
	}

	String makeDescription(AgendaItem item) {
	    if (item instanceof ActivityItem) {
		ActivityItem ai = (ActivityItem)item;
		String rName = ai.getExpansionRefinementName();
		if (rName != null)
		    // return "Expanded by " + rName;
                    return "Refined by " + rName;
	    }
	    String how = item.getHowHandled();
	    if (how != null)
		return how;
            if (item.getStatus() == Status.COMPLETE)
                // /\/: We'll say "Done" if the item is COMPLETE
                // though in some cases it should be "N/A" instead.
                // Perhaps we should always remember how the item
                // was handled, but would that always make sense
                // when a plan from one kind of agent is loaded in another?
                return "Done";
            else
                return NO_ACTION_DESCRIPTION;
	}

    }

    @Deprecated
    public class SimpleDoneAction extends HandlerAction.Manual {

	AgendaItem item;

	public SimpleDoneAction(AgendaItem item) {
	    this(item, "Done");
	}

	public SimpleDoneAction(AgendaItem item, String descr) {
	    this.item = item;
	    this.shortDescription = descr;
	}

	public boolean isReady() {
	    return Collect.isEmpty
		(ip2.getModelManager().getNodeConditions(item));
	}

	public ActionUnreadyReason getUnreadyReason() {
	    return new SimpleUnreadyReason(this, new String[] {
		"Cannot be used when there are conditions.",
		"Use \"Satisfy conditions\" instead."});
	}

    }

    public class NotApplicableAction extends DoneAction {
	public NotApplicableAction(AgendaItem item) {
	    super(item, "N/A", "N/A");
	}
    }

    public class DoneAction extends HandlerAction {

	AgendaItem item;
	boolean wasExpanded;
	String expandedDescr;

	MatchEnv env = SimpleMatcher.emptyEnv;
	ListOfConstraint filters;
	List filterEnvs = Collections.EMPTY_LIST;

	Set unboundVariables;

	public DoneAction(AgendaItem item) {
	    this(item, "Done", "Start");
	}

	public DoneAction(AgendaItem item, String descr, String expandedDescr) {
	    this.shortDescription = descr;
	    this.item = item;
	    this.expandedDescr = expandedDescr;
	    this.wasExpanded = item.isExpanded();
	    if (this.wasExpanded && !item.getChildren().isEmpty())
		this.shortDescription = expandedDescr;
	    this.filters = modelManager.getNodeConditions(item);
	}

	public boolean isReady() {
	    // The item may have been expanded since the last time we looked.
	    //\/ Can that happen yet?
	    if (item.isExpanded() != wasExpanded) {
		Debug.expect(item.isExpanded());
		wasExpanded = true;
		filters = modelManager.getNodeConditions(item);
		computeStatus();
	    }
	    return filters.isEmpty() || !filterEnvs.isEmpty();
	}

	public void computeStatus() {
	    if (!filters.isEmpty())
		filterEnvs = modelManager.evalFilters(filters, env);
	}

	public ActionUnreadyReason getUnreadyReason() {
	    List satisfiable = modelManager.testFilters(filters, env);
	    return new UnsatisfiedConditionsUnreadyReason
		(this, filters, satisfiable, env);
	}

	public void handle(AgendaItem item) {

	    Debug.expect(item == this.item);

	    // Get filter conds from the instantiated refinement.
	    // List conds = item.getRefinement().getFilterConditions();
	    ListOfConstraint conds = filters;
	    if (conds.isEmpty()) {
		adjustStatus(item);
		return;
	    }

	    // Don't need to pass an env this time.
	    // The model manager returns the live branches.
	    List branches = modelManager.reevaluateFilters(conds);
	    Debug.expect(!branches.isEmpty(),
			 "Cannot match filter conditions " + conds);

	    // The branches tell us all the ways we could satisfy
	    // all the filters by binding.  Each branch is a way to
	    // satisy all the filters.

	    Set condVars = Variable.varsAnywhereIn(conds);
	    Set unboundVars = Variable.unboundVarsIn(condVars);

	    Debug.noteln("Cond vars", condVars);
	    Debug.noteln("Unbound", unboundVars);

	    if (unboundVars.isEmpty()) {
		//modelManager.satisfyConds(item, conds, new MatchEnv());
		adjustStatus(item);
		return;
	    }

	    // We now have to get the user to bind all the unbound
	    // vars from the conditions, and then we can satisfy
	    // the conds.

	    unboundVariables = unboundVars;  // remember
	    BindingViewer binder =
		new BindingViewer(ip2, unboundVars) {
		    public void handleBindings(Map newBindings) {
			if (newBindings != null) {
			    bind(newBindings);
			}
		    }
		};
	}

	protected void bind(Map newBindings) {
	    try {
		modelManager.bindVariables(newBindings);
	    }
	    catch (IllegalArgumentException e) {
		// The user might have picked incompatible values
		// when selecting for more than one variable.
		Debug.displayException(e);
	    }
	    Set stillUnbound = Variable.unboundVarsIn(unboundVariables);
	    Debug.noteln("Still unbound", stillUnbound);
	    if (stillUnbound.isEmpty()) {
		//modelManager.satisfyConds(item, filters, new MatchEnv());
		adjustStatus(item);
	    }
	    else {
		// /\/: User quit before binding enough vars
		item.setHandledBy(null);
	    }
	}

	protected void adjustStatus(AgendaItem item) {
	    item.setStatus(Collect.isEmpty(item.getChildren())
  			   ? Status.COMPLETE : Status.EXECUTING);
	}

    }

    private static List<PNodeEnd> allWithoutStatusExcept
	             (PNodeEnd exception, List<PNodeEnd> list, Status status) {
	List<PNodeEnd> result = new LinkedList<PNodeEnd>();
	for (PNodeEnd ne: list)
	    if (ne != exception && ne.getStatus() != status)
		result.add(ne);
	return result;
    }

}
