/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Nov 24 16:35:44 2008 by Jeff Dalton
 * Copyright: (c) 2001 - 2002, 2004, 2007, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import javax.swing.*;

import java.util.*;

import ix.iface.util.CatchingActionListener;

import ix.icore.Variable;
import ix.iface.util.GridColumn;
import ix.iface.util.ToolFrame;
import ix.iface.util.VerticalPanel;

import ix.util.*;
import ix.util.lisp.*;

public abstract class BindingViewer implements ActionListener {

    protected Ip2 ip2;
    protected Set vars;
    protected Map possibleValues;

    protected JFrame frame;
    protected Container contentPane;

    protected VarPanel varPanel;

    public BindingViewer(Ip2 ip2, Set vars) {
	this.ip2 = ip2;
	this.vars = vars;
	this.possibleValues = getPossibleValues(vars);
	setUpFrame();
    }

    public Set getVars() {
	return vars;
    }

    protected void setUpFrame() {
	frame = new ToolFrame(ip2.getAgentDisplayName()
			      + " Variable Binder");
	contentPane = frame.getContentPane();

	// default is equiv to setLayout(new BorderLayout());
	// contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));

        // /\/: Changed to be like IVarTool, JD 24 Nov 08.
// 	contentPane.add(varPanel = new VarPanel(vars), BorderLayout.CENTER);

        varPanel = new VarPanel(vars);
        VerticalPanel p = new VerticalPanel();
        p.addFixedHeight(varPanel);
        p.add(new JPanel());    // to stretch /\/
        contentPane.add(p, BorderLayout.CENTER);
        
        contentPane.add(makeButtonPanel(), BorderLayout.SOUTH);

	frame.pack();
	frame.setVisible(true);
    }

    protected ButtonPanel makeButtonPanel() {
	return new ButtonPanel("Bind", "Finished");
    }

    class ButtonPanel extends JPanel {

	ButtonPanel(String... labels) {
	    // Stick with default flowlayout for now.
	    for (String label: labels) {
		add(makeButton(label));
	    }
	}

	protected JButton makeButton(String command) {
	    JButton b = new JButton(command);
	    b.addActionListener
		(CatchingActionListener.listener(BindingViewer.this));
	    return b;
	}

    }

    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("BindingViewer action:", command);
	if (command.equals("Bind"))
	    bindCommand();
	else if (command.equals("Finished"))
	    closeCommand();
	else
	    throw new ConsistencyException("Nothing to do for", command);
    }

    protected void bindCommand() {
	handleBindings(varPanel.getBindings());
	resetPossibleValues();
    }

    protected void closeCommand() {
	frame.setVisible(false);
    }

    public abstract void handleBindings(Map newBindings);

    protected Map getPossibleValues(Set vars) {
	return ip2.getIp2ModelManager().getVariableManager()
	             .getPossibleValues(vars);
    }

    protected void resetPossibleValues() {
	resetPossibleValues(getPossibleValues(vars));
    }

    protected void resetPossibleValues(Map possibles) {
	this.possibleValues = possibles;
	varPanel.resetPossibles(possibles);
	frame.validate();
    }

    protected void reloadView(Set vars) {
	this.vars = vars;
	this.possibleValues = getPossibleValues(vars);
	varPanel.reloadTable();
	frame.validate();
	// frame.pack();  // not all subclasses want to
    }

    class VarPanel extends JPanel {

	GridColumn varColumn = new GridColumn("Variable");
	GridColumn valColumn = new GridColumn("Value");

	Map varFields = new HashMap();
	List varList;

	VarPanel(Set vars) {
	    setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
	    setBorder(BorderFactory.createTitledBorder("Variables"));
	    add(varColumn);
	    add(Box.createHorizontalStrut(5));
	    add(valColumn);
	    loadTable();
	}

	void loadTable() {
	    varList = new LinkedList(vars);
	    Collections.sort(varList);
	    for (Iterator i = varList.iterator(); i.hasNext();) {
		Variable v = (Variable)i.next();

		// The variable
		JTextField varText = new JTextField(v.getName().toString());
		varText.setEditable(false);
		varColumn.add(varText);

		// The value
		JComponent valField = makeValField(v);
		varFields.put(v, valField);
		valColumn.add(valField);
	    }

	}

	void reloadTable() {
	    varColumn.reset();
	    valColumn.reset();
	    loadTable();
	    invalidate();
	}

	JComponent makeValField(Variable v) {
	    // /\/: We treat a var specially if it's bound because
	    // resetPossibleValues is sometimes called before the var manager
	    // knows about some new bindings, because the model-manager
	    // calls fireNewBindings before telling the var manager
	    // about new bindings.
	    Set p = v.isBound()
		? Collections.singleton(v.getValue())
		: (Set)possibleValues.get(v);
	    // Debug.noteln("Possible values for " + v + " = " + p);
//  	    if (v.isBound()) {
//  		JTextField tf =
//  		    new JTextField(v.getValue().toString() + "   ");
//  		tf.setEnabled(false);
//  		return tf;
//  	    }
	    return p == null
		? (JComponent)new JTextField(20)
		: (JComponent)new ValComboBox(v);
	}

	void resetPossibles(Map possibles) {
	    for (Iterator i = possibles.keySet().iterator(); i.hasNext();) {
		Variable v = (Variable)i.next();
		JComponent oldField = (JComponent)varFields.get(v);
		JComponent newField = makeValField(v);
		valColumn.replace(oldField, newField);
		varFields.put(v, newField);
	    }
	}

	Map getBindings() {
	    Map bindings = new HashMap();
	    for (Iterator i = varList.iterator(); i.hasNext();) {
		Variable var = (Variable)i.next();
		JComponent valField = (JComponent)varFields.get(var);
		Object val = null;
		if (!valField.isEnabled())
		    ;
		else if (valField instanceof JTextField) {
		    JTextField tf = (JTextField)valField;
		    String text = tf.getText(); 
		    text = text.trim();
		    if (!text.equals("")) {
			val = Lisp.readFromString(text);
			tf.setEnabled(false);
		    }
		}
		else {
		    ValComboBox vbox = (ValComboBox)valField;
		    Object selected = vbox.getSelectedItem();
		    if (selected instanceof Value) {
			val = ((Value)selected).get();
			vbox.setEnabled(false);
		    }
		}
		if (val != null)
		    bindings.put(var, val);
	    }
	    return bindings;
	}

    }

    class ValComboBox extends JComboBox {

	public ValComboBox(Variable var) {
	    super();
	    if (var.isBound()) {
		Value val = new Value(var.getValue());
		addItem(val);
		setSelectedItem(val);
		setEnabled(false);
		return;
	    }
	    addItem(" ");
	    Set possible = (Set)possibleValues.get(var);
	    for (Iterator i = possible.iterator(); i.hasNext();) {
		Object val = i.next();
		addItem(new Value(val));
	    }
	    setSelectedItem(" ");
	}

    }

    /** 
     * Class to hold values in a ValComboBox so that we don't
     * have to convert back from strings.
     */
    static class Value {

	Object v;

	Value(Object v) {
	    this.v = v;
	}
	Object get() {
	    return v;
	}
	public String toString() {
	    return Lisp.printToString(v);
	}

    }

}
