/* Author: Jeff Dalton
 * Updated: Tue Aug 22 16:59:20 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2002, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import java.awt.Component;
import java.awt.Color;
import java.awt.Font;
import java.awt.event.*;
import java.util.*;

import ix.ip2.event.*;

import ix.icore.*;
import ix.icore.domain.Refinement;

import ix.iface.util.GridColumn;
import ix.iface.util.PriorityComboBox;
import ix.iface.util.CatchingActionListener;

import ix.util.*;
import ix.util.lisp.*;

/**
 * A viewer for a set of AgendaItems.  The items are managed by
 * an AgendaManager (usually an Agenda).  The view has the shape
 * of a table but it not represented that way internally; instead,
 * there are objects representing rows and columns.
 */
public abstract class AgendaViewTable extends JPanel
    implements AgendaViewer, AgendaListener {

    protected Ip2 ip2;

    AgendaManager agendaManager;
    ItemEditor itemEditor;

    // The columns
    GridColumn descriptionCol = new GridColumn("Description");
    GridColumn commentsCol = new GridColumn("Annotations");
    GridColumn priorityCol = new GridColumn("Priority");
    GridColumn actionCol = new GridColumn("Action           ");

    final int descriptionWidth
        = Parameters.getInt("description-column-width", 30);
    final int commentsWidth
        = Parameters.getInt("annotations-column-width", 20);

    /**
     * Lets us find the Row object that represents an item.
     * An item should not be in the table until after the
     * fields of the row have been added to the appropriate
     * comumns.
     */
    HashMap itemToRowMap = new HashMap();

    /**
     * A List of rows in the order in which they were first added
     * to the table.  Note this this is NOT the order in which they
     * are listed in the table, because subitem rows can be added
     * at any time and must be inserted below their parent and hence
     * above any more recently added top-level entries (and their
     * descendents).
     */
    List rows = new LinkedList();

    /** 
     * Says which action to use when an item has more than one that
     * have the same description.  The two keys are the item and
     * the description.
     */
    TwoKeyHashMap actionShadowingTable = new TwoKeyHashMap();

    /**
     * Constructs a viewer for the indicated agent.
     */
    public AgendaViewTable(Ip2 ip2) {
        super();
	this.ip2 = ip2;
	setLayout(new BoxLayout(this, BoxLayout.X_AXIS));

	descriptionCol.setPreferredSize
	    (new JTextField("", descriptionWidth).getPreferredSize());

	commentsCol.setPreferredSize
	    (new JTextField("", commentsWidth).getPreferredSize());

	priorityCol.setPreferredSize
	    (new PriorityComboBox().getPreferredSize());

	add(descriptionCol);
	add(commentsCol);
	add(priorityCol);
	add(actionCol);
    }

    public void setAgendaManager(AgendaManager agendaManager) {
	this.agendaManager = agendaManager;
    }

    public void ensureItemEditor() {
 	if (itemEditor == null) {
 	    itemEditor = makeItemEditor();
 	}
	itemEditor.setVisible(true);
    }

    /**
     * Creates the item editor.  This method is defined in subclasses
     * to instantiate the desired ItemEditor class.
     */
    protected abstract ItemEditor makeItemEditor();

    /**
     * Sets the viewer back to something approximating its initial state.
     */
    public synchronized void reset() {
	actionShadowingTable.clear();
	itemToRowMap.clear();
	rows.clear();
	clearTable();
	invalidate();
    }

    /**
     * Clears what's displayed without resetting anything else.
     */
    public synchronized void clearTable() {
	Debug.noteln("Clearing item viewing table");
	descriptionCol.reset();
	commentsCol.reset();
	priorityCol.reset();
	actionCol.reset();
    }

    /**
     * Clears the table then puts everything back, taking account
     * of any changes in open/closed status.
     */
    public synchronized void redisplay() {
	clearTable();
	restoreTable();
	invalidate();
	SwingUtilities.getRoot(this).validate();
    }

    /**
     * Undoes a <code>clearTable()</code>, taking into acount any
     * changes in open/closed status.
     */
    public void restoreTable() {
	// The rows List has the top-level nodes in the same order
	// in which they should appear in the table; but for subrows
	// (subitem rows), which could have been added at any time,
	// we have to go via the items.
	Debug.noteln("Restoring item viewing table");
	for (Iterator ri = rows.iterator(); ri.hasNext();) {
	    Row row = (Row)ri.next();
	    if (row.item.getLevel() == 0) {
		row.simplyAddToTable();
		if (row.open) {
		    restoreSubtree(row.item);
		}
	    }
	}
    }

    /**
     * The part of the implementation of <code>restoreTable()</code>
     * that restores the section of the table that corresponds to the
     * subitem tree of an item whose row is "open".
     */
    protected void restoreSubtree(AgendaItem item) {
	for (Iterator ci = item.getChildren().iterator(); ci.hasNext();) {
	    AgendaItem child = (AgendaItem)ci.next();
	    Row childRow = (Row)itemToRowMap.get(child);
	    childRow.simplyAddToTable();
	    if (childRow.open) {
		restoreSubtree(child);
	    }
	}
    }

    /*
     * Finding HandlerActions from descriptions ...
     */

    // /\/: Complicated, at present, by the ready/not-ready indicator.

    protected HandlerAction findHandlerAction(AgendaItem item,
					      String shortDescription) {
//  	String descr = shortDescription.startsWith("-")
//  	    ? shortDescription.substring(1)
//  	    : shortDescription;
	String descr = shortDescription;
	HandlerAction shadowingAction
	    = (HandlerAction)actionShadowingTable.get(item, descr);
	return (shadowingAction != null)
	    ? shadowingAction
	    : item.findAction(descr);
    }

    protected void setShadowingAction(AgendaItem item,
				      String shortDescription,
				      HandlerAction action) {
	actionShadowingTable.put(item, shortDescription, action);
    }

    protected String actionChoiceDescr(HandlerAction act) {
	String d = act.getActionDescription();
	// return act.isReady() ? d : "-" + d;
	return d;
    }


    /*
     * Instructions from the frame menu bar
     */

    public void getNewItemFromUser() {
	ensureItemEditor();
	itemEditor.showNewItem();
    }

    /*
     * Instructions from the item editor
     */

    public AgendaItem makeItem(LList pattern) {
	return agendaManager.makeItem(pattern);
    }

    public void addItem(AgendaItem i) {
	agendaManager.addItem(i);
    }

    public void saveExpansion(Refinement data) {
//  	Ip2Frame root = (Ip2Frame)SwingUtilities.getRoot(this);
//  	root.getDomainEditor().saveExpansion(data);
	ip2.frame.getDomainEditor().saveExpansion(data);
    }

    public void expandItem(AgendaItem i, Refinement instructions) {
	agendaManager.expandItem(i, instructions);
    }


    /*
     * Useful methods for adding test items.  They assume that the
     * text can be parsed w/o error.  Note that the ItemEditor does
     * not call these methods -- they are called only "internally",
     * specifically for test items.
     */

    /*
    public void addItem(String text) {
	// Assumes that the text can be parsed w/o error.
	// Note that the ItemEditor does not call this method --
	// it's called only "internally", for test items.
	addItem(agendaManager.makeItem(text));
    }

    public void addItem(Priority priority, String text) {
	// Assumes that the text can be parsed w/o error.
	// Note that the ItemEditor does not call this method --
	// it's called only "internally", for test items.
	AgendaItem i = agendaManager.makeItem(text);
	i.setPriority(priority);
	addItem(i);
    }
    */


    /*
     * Instructions from the controller (aka agenda manager)
     */

    /*
     * Item added
     */

    // Method for ControllerListener interface
    public void itemAdded(AgendaEvent event, AgendaItem i) {
	itemAdded(i);
    }

    public synchronized void itemAdded(AgendaItem i) {
	Debug.noteln("Viewer adding item", i);

	// Create a row for the item
	Row row = new Row(i);
	row.addToTable();
	
	adjustSizes();

    }

    protected void adjustSizes() {
	// This method is sometimes called before there's a root,
	// in particular when a panel (I-LEED) preloads some items.  /\/
	Component root = SwingUtilities.getRoot(this);
	if (root == null) {
	    Debug.noteln("Tried to adjust sizes without root frame");
	    return;
	}
	JFrame r = (JFrame)root;
	r.validate();
    }

    /*
     * Item removed
     */

    public void itemRemoved(AgendaEvent event, AgendaItem i) {
	// /\/: What if the item has children?
	Debug.noteln("Viewer removing item", i);
	if (i.getParent() != null) {
	    throw new RuntimeException
		("Attempt to remove AgendaViewTable item with children "
		 + i);
	}
	Row row = (Row)itemToRowMap.get(i);
	rows.remove(row);
	itemToRowMap.remove(i);
	redisplay();
    }

    /*
     * Item handled
     */

    public void itemHandled(AgendaEvent e,
			    AgendaItem i,
			    HandlerAction h) {
	// Called when an item has been handled in some way other than
	// the user selecting an action from the menu - in particular when
	// the user has specified a manual expansion.  The JComboBox's
	// listener will be told when we set the selected item and may
	// need to know this is not the usual case.  At present, we let
	// it know by disabling the combobox.  /\/
	Row row = (Row)itemToRowMap.get(i);
	JComboBox actionChoice = row.actionChoice;
	String description = actionChoiceDescr(h);
	Debug.noteln("Item " + i + " handled by " + description);
	// Disable before setting selected item.
	// See makeActionChoiceListener.
	actionChoice.setEnabled(false);
	actionChoice.setSelectedItem(description);
    }


    /*
     * New bindings
     */

    public void newBindings(AgendaEvent event, Map bindings) {
	// We need to change the description of any item that contains
	// one of the newly bound Variables in its pattern.
	// This should work but is far from ideal.  /\/
	Set vars = bindings.keySet();
	for (Iterator i = itemToRowMap.entrySet().iterator();
	     i.hasNext();) {
	    Map.Entry m = (Map.Entry)i.next();
	    AgendaItem item = (AgendaItem)m.getKey();
	    JTextField text = ((Row)m.getValue()).descriptionText;
	    // Ought to have a proper check for whether any change is
	    // needed, but this will do for now.  /\/
	    if (!item.getPatternVars().isEmpty()) {
		// The item's short description should have adjusted
		// automatically.
		text.setText(Strings.repeat(item.getLevel(), "     ")
			     + item.getShortDescription());
	    }
	}
    }


    /**
     * A row of the table - corresponds to one item.
     */
    class Row implements AgendaItemListener {
	AgendaItem item;
	boolean open = true;
	JTextField descriptionText;
	JTextField commentsText;
	PriorityComboBox priorityChoice;
	JComboBox actionChoice;
	
	Row(AgendaItem item) {
	    this.item = item;
	    setupRow();
	}

	void setupRow() {
	    setupDescriptionText();
	    setupCommentsText();
	    setupPriorityChoice();
	    setupActionChoice();

	    // Listen to any changes to the isssue
	    item.addItemListener(this);
	}

	void setupDescriptionText() {
	    descriptionText = 
		new JTextField(Strings.repeat(item.getLevel(), "     ")
			       + item.getShortDescription(), 
			       descriptionWidth);
	    descriptionText.setCaretPosition(0); // show left edge
	    descriptionText.setEditable(false);
	    descriptionText.setBackground(Color.white);
	    descriptionText.setBorder(BorderFactory.createEtchedBorder());
	    descriptionText
		.addMouseListener(makeMouseListener(item));
	}

	void setupCommentsText() {
	    commentsText = new JTextField("", commentsWidth);
	    if (!item.getComments().equals(""))
		commentsText.setText
		    (Strings.firstLine(item.getComments()).trim());
	    commentsText.setEditable(false);
	    commentsText.setBackground(Color.white);
	    commentsText.setBorder(BorderFactory.createEtchedBorder());
	    commentsText.addMouseListener(makeMouseListener(item));
	}

	void setupPriorityChoice() {
	    priorityChoice = new PriorityComboBox();
	    priorityChoice
		.addActionListener
		    (CatchingActionListener
		         .listener(makePriorityChoiceListener(item)));
	    priorityChoice
		.setPriority(item.getPriority());
	}

	void setupActionChoice() {
	    actionChoice = new JComboBox();
	    for (Iterator i = item.getActions().iterator();
		 i.hasNext();) {
		HandlerAction act = (HandlerAction)i.next();
		actionChoice.addItem(actionChoiceDescr(act));
	    }
	    actionChoice
		.addActionListener
		    (CatchingActionListener
		         .listener(makeActionChoiceListener(item)));
	    actionChoice
		.setRenderer(new ActionCellRenderer(item));
	    actionChoice
		.setBackground(item.getStatus().getColor());
	}

	int getRowIndex() {
	    // After the Row has been added to the table, we may
	    // need to know its numeric index so that we can insert
	    // rows for child items after it.  For this, we need
	    // to ask one of the columns, but it doesn't matter 
	    // which one.
	    return descriptionCol.getRowIndex(descriptionText);
	}

	void simplyAddToTable() {
	    descriptionCol.add(descriptionText);
	    commentsCol.add(commentsText);
	    priorityCol.add(priorityChoice);
	    actionCol.add(actionChoice);
	}

	void addToTable() {
	    if (item.getParent() == null) {
		simplyAddToTable();
	    }
	    else {
		Row parentRow = (Row)itemToRowMap.get(item.getParent());

		// Make parent description bold if it isn't already
		JTextField parentText = parentRow.descriptionText;
		Font parentFont = parentText.getFont();
		if (!parentFont.isBold()) {
		    parentText.setFont(parentFont.deriveFont(Font.BOLD));
		}

		// Add item below parent's row and after any siblings
		// already present.  We can assume that none of the siblings
		// have yet been expanded.  /\/
		int insertRow = parentRow.getRowIndex();
		for (Iterator ci = item.getParent().getChildren().iterator();
		     ci.hasNext();) {
		    AgendaItem sibling = (AgendaItem)ci.next();
		    Row siblingRow = (Row)itemToRowMap.get(sibling);
		    if (siblingRow != null) {
			// Sibling is already in the table
			int siblingIndex = siblingRow.getRowIndex();
			if (siblingIndex > insertRow)
			    insertRow = siblingIndex;
		    }
		}
		insertRow++;
		descriptionCol.add(descriptionText, insertRow);
		commentsCol.add(commentsText, insertRow);
		priorityCol.add(priorityChoice, insertRow);
		actionCol.add(actionChoice, insertRow);
	    }
	    // Don't put the item in the map until it's in the
	    // table.  This is needed by the sibling-row code above.
	    itemToRowMap.put(item, this);
	    rows.add(this);
	}

	/*
	 * ItemListener methods
	 */

	public void statusChanged(AgendaItemEvent e) {
	    Status status = item.getStatus();
	    actionChoice.setBackground(status.getColor());
	    if (status == Status.COMPLETE 
		  || status == Status.EXECUTING
		  || status == Status.IMPOSSIBLE) {
		// /\/: Maybe remove all but selected item rather
		// than disable - so the colour will be seen.
		priorityChoice.setEnabled(false);
		actionChoice.setEnabled(false);
	    }
	}

	public void priorityChanged(AgendaItemEvent e) {
	    Priority priority = item.getPriority();
	    priorityChoice.setBackground(priority.getColor());
	}

	public void newReport(AgendaItemEvent e, Report report) {
	    commentsText.setText(Strings.firstLine(report.getText()).trim());
	}

	public void agendaItemEdited(AgendaItemEvent e) {
	    // Let's see ... what might have changed that we
	    // care about?
	    if (item.getReports().isEmpty()
		&& (!item.getComments().equals("")
		    || !commentsText.getText().equals("")))
		commentsText
		    .setText(Strings.firstLine(item.getComments()).trim());
	}

	public void handlerActionsChanged(AgendaItemEvent e) {
	    // /\/: What should we do about shadowing?
	    // /\/: Some duplicated code from setupActionChoice()
	    boolean saved = actionChoice.isEnabled();
	    actionChoice.setEnabled(false); // hack /\/
	    actionChoice.removeAllItems();
	    for (Iterator i = item.getActions().iterator();
		 i.hasNext();) {
		HandlerAction act = (HandlerAction)i.next();
		actionChoice.addItem(actionChoiceDescr(act));
	    }
	    actionChoice.setEnabled(saved);
	}

	public void newHandlerAction(AgendaItemEvent e, HandlerAction act) {
	    // Ignore the new action if the item staus indicates
	    // that it cannot be used.
	    Status status = item.getStatus();
	    if (status == Status.COMPLETE 
		  || status == Status.EXECUTING
		  || status == Status.IMPOSSIBLE) {
		return;
	    }

	    // Make sure the user can see which item we're
	    // talking about.
	    Color savedBackground = descriptionText.getBackground();
	    descriptionText.setBackground(Color.blue);

	    // See if we already know about an action with
	    // the same description as the new one.
	    String target = act.getActionDescription();
	    String exists = null;
	    for (int i = 0, len = actionChoice.getItemCount();
		 i < len; i++) {
		Object item = actionChoice.getItemAt(i);
		if (item.equals(target)) {
		    exists = (String)item;
		    break;
		}
	    }

	    if (exists != null) {
		// There is already an action with the same description.
		// See if the user wants the new action or the old one.
		// We assume that if we do nothing we will continue to
		// find the same action as before.
		if (shouldReplaceAction(exists)) {
		    // The user wants the new action, so shadow the old.
		    setShadowingAction(item, act.getActionDescription(), act);
		    Debug.expect(findHandlerAction(item, exists) == act);
		}
		else {
		    // Check that lookup doesn't get the new action.
		    Debug.expect(findHandlerAction(item, exists) != act);
		}
	    }
	    else {
		// We have a new action that does not match an
		// existing description, so add its description
		// to the JComboBox so the user can select it.
		// /\/: May get too many of these, so eliminate
		// this dialog for now.
//  		JOptionPane.showMessageDialog(AgendaViewTable.this,
//  		    new Object[] {
//  			"New action " + Util.quote(act.getActionDescription()),
//  			"For item " + Util.quote(item.getShortDescription())
//  		    },
//  		    "New action",
//  		    JOptionPane.INFORMATION_MESSAGE);
		actionChoice.addItem(actionChoiceDescr(act));
		adjustSizes();
	    }

	    // Restore item colour changed to make sure the
	    // user can see which item we're talking about.
	    descriptionText.setBackground(savedBackground);

	}

	boolean shouldReplaceAction(String actDescription) {
	    Object[] answers = {
		"Replace existing action",
		"Ignore new action"};
	    Object selected = JOptionPane.showInputDialog(
		AgendaViewTable.this,
		new Object[] {
		    "Item " + Util.quote(item.getShortDescription()),
		    "Already has an action " + 
			Util.quote(actDescription)
		},
		"New Action",			// title
		JOptionPane.INFORMATION_MESSAGE,
		null,				// icon
		answers,
		answers[0]);
	    return selected.equals("Replace existing action");
	}

    }

    public class ActionCellRenderer extends DefaultListCellRenderer
	   implements ListCellRenderer {

	AgendaItem item;

	ActionCellRenderer(AgendaItem item) {
	    super();
	    this.item = item;
	}

	public Component getListCellRendererComponent
	        (JList list,
		 Object value,
		 int index,
		 boolean isSelected,
		 boolean cellHasFocus) {

	    HandlerAction action = findHandlerAction(item, (String)value);

	    // Result is probably == to this, but it shouldn't matter.
	    Component result = 
		super.getListCellRendererComponent
		    (list, value, index, isSelected, cellHasFocus);

	    if (!action.isReady())
		result.setForeground(Color.gray);

	    return result;

	}

    }

    /*
     * Other listeners
     *
     * The following methods are called from the Row when adding a
     * listener to a single item in the row.
     *
     */

    /**
     * Returns a listener that can be called when the user selects an
     * item action.
     */
    ActionListener makeActionChoiceListener(final AgendaItem item) {
	return new ActionListener() {
	    JComboBox cb;
	    public void actionPerformed(ActionEvent e) {
		cb = (JComboBox)e.getSource();
		String actionName = (String)cb.getSelectedItem();
		Debug.noteln("Selected action", actionName);

		if (!cb.isEnabled()) {
		    // /\/: See itemHandled method
		    // /\/: Now also handlerActionsChanged
		    Debug.noteln("Assuming handled elsewhere");
		    return;
		}

		HandlerAction act = findHandlerAction(item, actionName);
		Debug.expect(act != null, "Can't find action", actionName);

		// The user can mess around with the actions
		// while the status is BLANK but can't get
		// anything to happen unless the status is POSSIBLE.
		// /\/: This now depends on the item and action classes.
		// /\/: Not clear the test should be here in the GUI
		// rather than in the agenda.
		if (item.actionCanBeTakenNow(act))
		    // Test was: (item.status == Status.POSSIBLE)
		    handleItem(act);
		else {
		    // Put selection back to "No Action"
		    Debug.noteln("Ignoring action selection");
		    cb.setSelectedItem
                        (HandlerAction.NoAction.NO_ACTION_DESCRIPTION);
		    JOptionPane.showMessageDialog(AgendaViewTable.this,
		        new Object[] {"Action " +
			    Strings.quote(act.getActionDescription()),
			    "cannot be taken at this time."},
			"Action not ready",
			JOptionPane.INFORMATION_MESSAGE);
		}
	    }
	    void handleItem(HandlerAction act) {
		// Protect the GUI from problems elsewhere.
		// /\/: Need to deal with Errors, such as
		// AssertionFailure too
		try { agendaManager.handleItem(item, act); }
		catch (Throwable t) {
		    Debug.noteln("Exception while handling item", item);
		    Debug.noteException(t);
		    JOptionPane.showMessageDialog(AgendaViewTable.this,
		        new Object[] {"Problem while handling item",
				      Debug.describeException(t)},
			"Problem while handling item",
			JOptionPane.ERROR_MESSAGE);
		    cb.setSelectedItem
                        (HandlerAction.NoAction.NO_ACTION_DESCRIPTION);
		}
	    }
	};
    }

    /**
     * Returns a listener that can be called when the user selects
     * an item priority.
     */
    ActionListener makePriorityChoiceListener(final AgendaItem item) {
	return new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		PriorityComboBox cb = (PriorityComboBox)e.getSource();
		Priority priority = cb.getSelectedPriority();
		Debug.noteln("Selected priority", priority);
		item.setPriority(priority);
	    }
	};
    }

    /**
     * Returns a listener than can be called when the user clicks
     * in the text of an item description.
     */
    MouseListener makeMouseListener(final AgendaItem item) {

	return new MouseAdapter() {

	    ItemPopupMenu popup = new ItemPopupMenu(item);

	    public void mouseClicked(MouseEvent e) {
		JTextField clicked = (JTextField)e.getComponent();
		Debug.noteln("User clicked on item",
			     item.getShortDescription());
		if (SwingUtilities.isLeftMouseButton(e)) {
		    ensureItemEditor();
		    itemEditor.showItem(item);
		}
	    }

	    public void mousePressed(MouseEvent e) {
		JTextField clicked = (JTextField)e.getComponent();
		Debug.noteln("User pressed on item",
			     item.getShortDescription());
		// /\/: Looks like isPopupTrigger() may always be
		// false on some platforms.
		Debug.noteln("event.isPopupTrigger()=" + e.isPopupTrigger());
		if (SwingUtilities.isRightMouseButton(e)) {
		    Debug.noteln("Right press");
		    popup.noticeItemState();
		    popup.show(e.getComponent(), e.getX(), e.getY());
		}

	    }

	};

    }

    /**
     * The popup menu that appears when the user right-clicks on a item.
     */
    class ItemPopupMenu extends AbstractAgendaItemPopupMenu {

	ItemPopupMenu(AgendaItem item) {
	    super(AgendaViewTable.this.ip2, item);
	}

	void showDetails() {
	    ensureItemEditor();
	    itemEditor.showItem(item);
	}

	boolean isOpen() {
	    Row r = (Row)itemToRowMap.get(item);
	    return r.open;
	}

	void fold() {
	    Row r = (Row)itemToRowMap.get(item);
	    r.open = false;
	    redisplay();
	}

	void unfold() {
	    Row r = (Row)itemToRowMap.get(item);
	    r.open = true;
	    redisplay();
	}

    }

}

// Issues:
// * What should be done when removing an item that has children?
//   Don't allow it?  Remove descendents as well?

