/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 16 18:33:30 2006 by Jeff Dalton
 * Copyright: (c) 2002 - 2003, 2006, AIAI, University of Edinburgh
 */

package ix.iface.util;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.TableModelEvent;

import java.awt.event.*;

import java.util.*;

import ix.util.*;
import ix.util.lisp.*;

/**
 * Packages a JTable together with a table model suitable for viewing
 * a mapping from keys to values.
 */
public abstract class KeyValueTable {

    protected Map viewedMap = new HashMap();
    protected List keyList = new LinkedList(); // List because we need indexOf
    protected Map keyTimestamps = new HashMap();
    protected int timestamp = 0;

    /**
     * Determines the order in which keys are listed.  The initial
     * value is a Comparator that puts the newest key last by
     * comparing timestamps; but it can be replaced by a Comparator
     * that imposes a different order.
     *
     * @see #keyTimestamp(Object)
     * @see #setKeySortComparator(Comparator)
     */
    protected Comparator keySortComparator = new MostRecentLastComparator();

    protected ViewJTable table;
    protected ViewTableModel model;
    protected ViewTableMouseListener mouseListener;

    public KeyValueTable(String keyColName, String valColName) {
	this(null, keyColName, valColName);
    }

    public KeyValueTable(Map initialMap,
			 String keyColName,
			 String valColName) {
	super();
	model = new ViewTableModel(keyColName, valColName);
	table = new ViewJTable(model);
	if (initialMap != null)
	    recordNewValues(initialMap);
	mouseListener = new ViewTableMouseListener();
	table
	    .addMouseListener(mouseListener);
    }

    public JTable getJTable() {
	return table;
    }

    public Object getValue(Object key) {
	return viewedMap.get(key);
    }

    public Map getViewedMap() {
	return Collections.unmodifiableMap(viewedMap);
    }

    protected String keyToString(Object key) {
	return Lisp.printToString(key);
    }

    protected String valueToString(Object value) {
	return Lisp.printToString(value);
    }

    public Comparator getKeySortComparator() {
	return keySortComparator;
    }

    /**
     * Sets the Comparator used to sort keys.
     *
     * @see #keySortComparator
     * @see MostRecentLastComparator
     * @see MostRecentFirstComparator
     * @see LexicographicComparator
     * @see PatternObjectComparator
     */
    public void setKeySortComparator(Comparator c) {
	keySortComparator = c;
	Collections.sort(keyList, keySortComparator);
	model.fireTableDataChanged();
    }

    /**
     * Empties the table.
     */
    public void reset() {
	viewedMap.clear();
	keyList.clear();
	keyTimestamps.clear();
	// Tell the JTable that all row data has changed.
	model.fireTableChanged(new TableModelEvent(model));
	table.invalidate();
    }

    public void recordNewValues(Map map) {
	for (Iterator i = map.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    recordNewValue(e.getKey(), e.getValue());
	}
    }

    public void recordNewValue(Object key, Object value) {
	int where = keyList.indexOf(key);
	if (where == -1) {
	    // Add new row
	    viewedMap.put(key, value);
	    int row = insertKey(key);
	    model.fireTableRowsInserted(row, row);
	}
	else {
	    // Just change value
	    viewedMap.put(key, value);
	    model.fireTableCellUpdated(where, model.VAL_COL);
	}
    }

    protected int insertKey(Object key) {
	Debug.expect(!keyList.contains(key), "Old key", key);
	timestampKey(key);
	int row = 0;
	for (ListIterator i = keyList.listIterator(); i.hasNext();) {
	    Object k = i.next();
	    if (keySortComparator.compare(key, k) < 0) {
		i.previous();
		i.add(key);
		Debug.expect(keyList.indexOf(key) == row);
		return row;
	    }
	    row++;
	}
	// Add at end
	Debug.expect(row == keyList.size());
	keyList.add(key);
	return row;
    }

    protected void timestampKey(Object key) {
	Debug.expect(!keyTimestamps.containsKey(key), "Old key", key);
	keyTimestamps.put(key, new Integer(timestamp++));
    }

    protected int keyTimestamp(Object key) {
	Integer i = (Integer)keyTimestamps.get(key);
	Debug.expect(i != null, "No timestamp for key", key);
	return i.intValue();
    }

    public void deleteEntry(Object key, Object value) {
	Object v = getValue(key);
	Debug.expect(v != null, "No value for key", key);
	if (!v.equals(value)) {
	    throw new IllegalArgumentException
		("Wrong value when deleting key " + key + ":" +
		 " found " + v +
		 " but expected " + value);
	}
	deleteEntry(key);
    }

    public void deleteEntry(Object key) {
	int where = keyList.indexOf(key);
	Debug.expect(where >= 0, "Can't find key to delete", key);
	keyList.remove(key);
	keyTimestamps.remove(key);
	viewedMap.remove(key);
	model.fireTableRowsDeleted(where, where);
    }

    public class MostRecentLastComparator implements Comparator {
	public MostRecentLastComparator() { }
	public int compare(Object k1, Object k2) {
	    return keyTimestamp(k1) - keyTimestamp(k2);
	}
    }

    public class MostRecentFirstComparator implements Comparator {
	public MostRecentFirstComparator() { }
	public int compare(Object k1, Object k2) {
	    return -(keyTimestamp(k1) - keyTimestamp(k2));
	}
    }

    public static class LexicographicComparator extends ObjectComparator {
	public LexicographicComparator() { }
    }

    public static class PatternObjectComparator extends ObjectComparator {
	public PatternObjectComparator() { }
	protected int compareLists(List a, List b) {
	    // Lists too short to have a 2nd elt go first
	    int aLen = a.size();
	    int bLen = b.size();
	    if (aLen < 2) {
		if (bLen < 2)
		    return super.compareLists(a, b);
		else
		    return -1;
	    }
	    else if (bLen < 2) {
		return 1;
	    }
	    else {
		Object aObj = a.get(1);
		Object bObj = b.get(1);
		int c = compare(aObj, bObj);
		if (c != 0)
		    return c;
		else
		    return super.compareLists(a, b);
	    }
	}
    }

    protected class ViewJTable extends JTable {

	public ViewJTable(TableModel model) {
	    super(model);
	    getTableHeader().setReorderingAllowed(false);
	    // setSelectionMode(ListSelectionModel.SINGLE_SELECTION); // ? /\/
	    // Setting cell-selection false seems to stop cells from
	    // turning blue when selected and the user then selects in
	    // a different JTable, or something like that.  \/\
	    setCellSelectionEnabled(false); // ? /\/
	    makeCellsMouseable();
	    setPreferredColumnWidths();
	    setPreferredScrollableViewportSize(getPreferredSize());
	}

	private void makeCellsMouseable() {
	    // This method is based on Jussi's I-DE code.
	    JTextField noEditTF = new JTextField();
	    noEditTF.setEditable(false); // affects background colour
	    noEditTF.setBackground(java.awt.Color.white);

	    DefaultCellEditor mouseableEditor =
		new DefaultCellEditor(noEditTF);
	    // /\/: Not clear why using String.class doesn't work.
	    // setDefaultEditor(String.class, mouseableEditor);
	    setDefaultEditor(Object.class, mouseableEditor);

	    // Make right button pop up menu even when pressed
	    // in a cell being "edited".
	    MouseListener ml = new MouseAdapter() {
		public void mousePressed(MouseEvent me) {
		    if (SwingUtilities.isRightMouseButton(me)) {
			// Pretend it was done in table instead.
			mouseListener.
			mousePressed(new MouseEvent(table,
				      me.getID(), me.getWhen(), 
				      me.getModifiers(),
				      me.getComponent().getX() + me.getX(),
				      me.getComponent().getY() + me.getY(),
				      me.getClickCount(),
				      me.isPopupTrigger()));
		    }
		}
	    };
	    noEditTF.addMouseListener(ml);
	    // mouseableEditor.getComponent().addMouseListener(ml);

	}

	protected void setPreferredColumnWidths() {
	    // /\/: Widths are times assumed number of pixels per char
	    getColumnModel().getColumn(model.KEY_COL)
		.setPreferredWidth(20 * 10);
	    getColumnModel().getColumn(model.VAL_COL)
		.setPreferredWidth(20 * 10);
	}

    }

    /** Mediates between the data and the JTable. */
    protected class ViewTableModel extends AbstractTableModel {

	final int KEY_COL = 0, VAL_COL = 1;
	final String[] columnName;

	public ViewTableModel(String keyColName, String valColName) {
	    super();
	    this.columnName = new String[]{keyColName, valColName};
	}

	public String getColumnName(int col) {
	    return columnName[col];
	}

	public int getColumnCount() {
	    return columnName.length;
	}

	public int getRowCount() {
	    return keyList.size();
	}

	public Object getValueAt(int row, int col) {
	    switch(col) {
	    case KEY_COL:
		return keyToString(keyList.get(row));
	    case VAL_COL:
		return valueToString(viewedMap.get(keyList.get(row)));
	    }
	    throw new ConsistencyException("Bogus column " + col);
	}

	public boolean isCellEditable(int row, int col) {
	    return true;
	}

	public void setValueAt(Object value, int row, int col) {
	    Debug.noteln("Asked to set table value to", value);
	}

    }

    protected class ViewTableMouseListener extends MouseAdapter {

	RowPopupMenu popup = makePopupMenu();

	public ViewTableMouseListener() { }

	public void mousePressed(MouseEvent e) {
	    if (popup != null && SwingUtilities.isRightMouseButton(e)) {
		int row = table.rowAtPoint(e.getPoint());
		Debug.noteln("Table right press in row", row);
		popup.setRow(row);
		popup.show(e.getComponent(), e.getX(), e.getY());
	    }
	    else if (SwingUtilities.isLeftMouseButton(e)) {
		// This invokes the cell editor on the first
		// left-button press.  If we ignored these
		// button presses here, the user would have to
		// double-click or triple-click or click-pause-click
		// or whethever the heck it is.  /\/
		int row = table.rowAtPoint(e.getPoint());
		int col = table.columnAtPoint(e.getPoint());
		// /\/: For now we don't ever want editing, so this
		// can be commented-out; moreover, if we leave it in,
		// changes to the cell being edited won't appear
		// until a different cell is selected, and clearing
		// the selection won't make any difference.
		// /\/: Is there some way to stop editing that's
		// started?
		// table.editCellAt(row, col);
	    }
	}

    }

    /** Factory method. */
    protected abstract RowPopupMenu makePopupMenu();

    /** 
     * Called by the row popup menu to intepret any action command
     * selected from the menu.
     */
    protected abstract void doPopupAction
	                      (ActionEvent event, int row, Object key);

    protected abstract class RowPopupMenu 
              extends JPopupMenu implements ActionListener {

	protected int row = -1;

	public RowPopupMenu() {
	    super();
	}

	protected JMenuItem makeMenuItem(String text) {
	    JMenuItem item = new JMenuItem(text);
	    item.addActionListener(CatchingActionListener.listener(this));
	    return item;
	}

	/**
	 * Tells the menu what row it is about.  This method is called
	 * just before the show method and should be used to do any
	 * row-dependent setup.
	 */
	public void setRow(int row) {
	    this.row = row;
	}

	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("Row popup command", command);
	    Debug.expect(row >= 0, "bad row", new Integer(row));
	    Object key = keyList.get(row);
	    Debug.expect(key != null, "no key for row", new Integer(row));
	    doPopupAction(e, row, key);
	}

    }

}
