/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Jul 30 12:38:18 2007 by Jeff Dalton
 * Copyright: (c) 2002, 2007, AIAI, University of Edinburgh
 */

package ix.iface.domain;

import java.io.*;
import java.util.*;

import ix.icore.domain.*;
import ix.icore.*;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Writes domain descriptions in O-Plan TF syntax.
 */
public class TF_Writer extends LTF_Writer {

    public TF_Writer(File domainName) {
	super(domainName);
    }

    @Override
    void outDomainHeader(Domain domain) {
	// No header in TF
	// /\/: Perhaps output comments?
    }

    @Override
    void outDomainAnnotation(Object key, Object value) {
	// Outputs an annotation of a domain.
	// There isn't a very good way to do this for TF,
	// but we might try outputting them as comments
	// or in a "language lisp".
    }

    @Override
    void outObjectClass(ObjectClass c) {
	// /\/: No way to do this.
    }

    @Override
    void outRefinementHeader(Refinement r) {
	// "schema " name ";"
	out("schema ");
	out(safeName(r.getName()));
	out(";");
	// Variable declarations, if any
	List varDcls = r.getVariableDeclarations();
	if (varDcls != null) {
	    outClauseStart("vars ");
	    for (Iterator i = varDcls.iterator(); i.hasNext();) {
		VariableDeclaration dcl = (VariableDeclaration)i.next();
		out(dcl.getName());
		if (i.hasNext())
		    out(", ");
	    }
	    outClauseFinish();
	}
	// "expands " pattern ";"
	outClauseStart("expands ");
	out(r.getPattern());
	outClauseFinish();
    }

    @Override
    void outRefinementClose() {
	outln("");
	outln("end_schema;");
    }

    @Override
    void outClauseStart(String name) {
	out(2, name);
    }

    @Override
    void outClauseFinish() {
	out(";");
    }

    @Override
    void outBigSeparator() {
	// Can assume a new line will follow, because out(int, whatever)
	// will be called.
	out(",");
    }

    @Override
    void outSmallSeparator() {
	out(", ");
    }

    @Override
    void outVarDcls(List varDcls) {
	// Handled in refinement header
    }

    @Override
    void outNode(NodeSpec spec) {
	out(4, spec.getId());
	out(" action ");
	out(spec.getPattern());
    }

    @Override
    void outOrdering(Ordering ord) {
	NodeEndRef from = ord.getFrom();
	NodeEndRef to = ord.getTo();
	outNodeEndRef(from);
	out(" ---> ");
	outNodeEndRef(to);
    }

    void outNodeEndRef(NodeEndRef ref) {
	out(ref.getNode());	// for now /\/
    }

    @Override
    void outConstraints(List constraints) {
	List conds = new LinkedList();
	List effects = new LinkedList();
	for (Iterator i = constraints.iterator(); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    Debug.expect(c.getType() == S_WORLD_STATE,
			 "Can't handle constraint", c);
	    Symbol relation = c.getRelation();
	    if (relation == S_CONDITION)
		conds.add(c);
	    else if (relation == S_EFFECT)
		effects.add(c);
	    else
		throw new ConsistencyException
		    ("Unknown condition type: world-state", relation);
	}
	if (!conds.isEmpty())
	    outConstraints("conditions", conds);
	if (!effects.isEmpty())
	    outConstraints("effects", effects);
    }

    private static final Symbol
	S_WORLD_STATE = Symbol.intern("world-state"),
	S_CONDITION   = Symbol.intern("condition"),
	S_EFFECT      = Symbol.intern("effect");

    private void outConstraints(String clauseName, List constraints) {
	outClauseStart(clauseName);
	for (Iterator i = constraints.iterator(); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    outConstraint(c);
	    if (i.hasNext()) outBigSeparator();
	}
	outClauseFinish();
    }

    @Override
    void outConstraint(Constraint c) {
	out(4, "");
	if (c.getRelation() == S_CONDITION)
	    out("unsupervised ");
	for (Iterator i = c.getParameters().iterator(); i.hasNext();) {
	    outConstraintParameter(i.next());
	    if (i.hasNext())
		throw new ConsistencyException("Can't handle", c);
	}
    }

    @Override
    void outIssues(List issues) {
 	outClauseStart(";;; issues");
	for (Iterator i = issues.iterator(); i.hasNext();) {
	    Issue issue = (Issue)i.next();
	    outIssue(issue);
	    if (i.hasNext()) outBigSeparator();
	}
	outClauseFinish();
    }

    @Override
    void outIssue(Issue issue) {
	out(2, ";;;   ");
	out(issue.getPattern());
    }

    @Override
    void outComments(String comments) {
	List lines = Strings.breakIntoLines(comments);
	for (Iterator i = lines.iterator(); i.hasNext();) {
	    String line = (String)i.next();
	    out(2, ";;; ");
	    out(line);
	}
    }

    @Override
    void outAnnotations(Annotations annotations) {
	// Outputs the annotations of a refinement.
	// There isn't a very good way to do this for TF,
	// but we might try outputting them as comments
	// or in a "language lisp".
    }

    // Output utilities

    @Override
    String toString(Object o) {
	return o instanceof LList ? patternText((LList)o) : o.toString();
    }

    private String patternText(LList pattern) {
	pattern = pattern.mapTree(new Function1() {
	    public Object funcall(Object elt) {
		if (elt instanceof Symbol
		    && elt.toString().indexOf('-') >= 0)
		    return Symbol.intern(safeName(elt.toString()));
		else
		    return elt;
	    }
	});
	return Strings.replaceChars("()", "{}", pattern.toString());
    }

    private String safeName(String name) {
	return Strings.replaceChars("- ", "__", name);
    }

}
