/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Jun  1 18:52:28 2008 by Jeff Dalton
 * Copyright: (c) 2002, 2005, 2007, 2008, AIAI, University of Edinburgh
 */

package ix.iface.domain;

import java.io.*;
import java.util.*;

import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

public class ITF_Writer {

    public ITF_Writer() {
    }

    public void writeDomain(final Domain dom, File file) throws IOException {
	final Writer out = new BufferedWriter(new FileWriter(file));
	Util.run(new WithCleanup() {
	    public void body() throws IOException {
		writeDomain(dom, out);
		out.flush();
	    }
	    public void cleanup() throws IOException {
		out.close();
	    }
	});
    }

    public void writeDomain(Domain dom, Writer out) throws IOException {
	StringListLtfWriter ltf = new StringListLtfWriter();
	ltf.outDomain(dom);
	for (String line: new LineTrimmer(ltf.getLines()).trimLines()) {
	    out.write(line);
	    out.write("\n");
	}
	out.flush();
    }

    // There are three sorts of lines:
    //   * Blank lines -- leave them as-is
    //   * Indented lines that begin with "(" and end with ")" --
    //     remove the outer parens.
    //   * Lines that begin with "(" but don't end with ")" --
    //     These should be followed by an indented block of lines.
    //     Remove the "(" and remove one extra ")" from the end
    //     of the last line in the indented block.

    // We know that LTF_Writer uses only spaces for indentation.

    private class LineTrimmer {

	private List<String> lines;

	private LinkedList<Integer> indentStack;
	private ListIterator<String> lineIter;
	private LinkedList<String> result;
	int indentLevel;

	private LineTrimmer(List<String> lines) {
	    this.lines = lines;
	}

	private List<String> trimLines() {

	    indentStack = new LinkedList<Integer>();
	    lineIter = lines.listIterator();
	    result = new LinkedList<String>();
	    indentLevel = 0;

	    while (lineIter.hasNext()) {
		String line = trimTrailingSpaces(lineIter.next());
		if (isAllSpaces(line)) {
		    result.add("");
		    continue;
		}
		int indent = indentLevel(line);
		if (indent > indentLevel) {
		    pushIndent(indent, line);
		}
		if (indent < indentLevel) {
		    popIndent(indent, line);
		}
		int parens = countParens(line);
		if (parens == 0)
		    result.add(trimOuterParens(line));
		else if (parens == 1)
		    result.add(trimOuterOpenParen(line));
		else if (parens < 0) {
		    while (parens < 0) {
			line = trimOuterCloseParen(line);
			parens++;
		    }
		    result.add(trimOuterParens(line));
		}
		else
		    throw new ConsistencyException
			("Can't handle line: " + line);
			       
	    }
	    return result;

	}

	int countParens(String line) {
	    int count = 0;
	    for (int i = 0; i < line.length(); i++) {
		char c = line.charAt(i);
		if (c == '(') count++;
		else if (c == ')') count--;
	    }
	    return count;
	}

	void pushIndent(int indent, String line) {
	    indentStack.add(indentLevel);
	    indentLevel = indent;
	}

	void popIndent(int indent, String line) {
	    while (indent < indentLevel) {
		indentLevel = indentStack.removeFirst();
	    }
	}

	// We might not need to trim trailing spaces, but it would be
	// a pain to verify that LTF_Writer never outputs them, and it
	// would be an irritating bug if LTF_Writer changed so that
	// it did.

	String trimTrailingSpaces(String line) {
	    int lastIndex = line.length() - 1;
	    int i = lastIndex;
	    while (i >= 0)
		if (line.charAt(i) != ' ') break; else i--;
	    // i is the index of the last non-space
	    return i == lastIndex ? line : line.substring(0, i + 1);
	}

	String trimOuterParens(String line) {
	    return trimOuterOpenParen(trimOuterCloseParen(line));
	}

	String trimOuterOpenParen(String line) {
	    // The line may have spaces before the first open paren
	    StringBuilder result = new StringBuilder();
	    for (int i = 0; i < line.length(); i++) {
		char c = line.charAt(i);
		if (c == ' ')
		    result.append(c);
		else if (c == '(')
		    return result.append(line.substring(i + 1)).toString();
		else
		    break;
	    }
// 	    throw new IllegalArgumentException
// 		("Missing initial open paren in " + Strings.quote(line));
	    return line;
	}

	String trimOuterCloseParen(String line) {
	    // The close paren must be the last character in the line;
	    // otherwise, the line is returned as-is.
	    return line.endsWith(")")
		? line.substring(0, line.length() - 1)
		: line;
	}
    }

    private int indentLevel(String line) {
	for (int i = 0; i < line.length(); i++)
	    if (line.charAt(i) != ' ')
		return i;
	throw new IllegalArgumentException("blank lines have no indent level");
    }

    private boolean isAllSpaces(String line) {
	for (int i = 0; i < line.length(); i++)
	    if (line.charAt(i) != ' ')
		return false;
	return true;
    }

    private static class StringListLtfWriter extends LTF_Writer {

	private List<String> lines = new LinkedList<String>();
	private StringBuilder line = new StringBuilder();

	List<String> getLines() {
	    return lines;
	}

	@Override
	void out(String s) {
	    line.append(s);
	}

	@Override
	void outln(String s) {
	    out(s);
	    lines.add(line.toString());
	    line.delete(0, line.length()); // start new line
	}

	@Override
	void outOrdering(Ordering ord) {
	    NodeEndRef from = ord.getFrom();
	    NodeEndRef to = ord.getTo();
	    outNodeEndRef(from, End.END);
	    out(" --> ");
	    outNodeEndRef(to, End.BEGIN);
	}

    }

    public static void main(String[] argv) throws Exception {
	Parameters.processCommandLineArguments(argv);
	String domainName = Parameters.requireParameter("domain");
	Domain dom = XML.readObject(Domain.class, XML.toURL(domainName));
	ITF_Writer itf = new ITF_Writer();
	itf.writeDomain(dom, new OutputStreamWriter(System.out));
    }

}
