/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Jun  7 21:48:26 2008 by Jeff Dalton
 * Copyright: (c) 2004, 2006 - 2008, AIAI, University of Edinburgh
 */

package ix.icore.process;

import java.util.*;

import ix.icore.Status;
import ix.icore.HasStatus;
import ix.icore.domain.End;
import ix.icore.domain.NodeEndRef;
import ix.icore.domain.Ordering;

import ix.ip2.Ip2ModelManager;	// /\/

import ix.util.*;
import ix.util.lisp.Lisp;
import ix.util.context.*;

/**
 * One end of a PNode.
 */
public class PNodeEnd extends TimePoint implements HasStatus {

    private End end;
    private PNode node;

    private long[] marks = new long[]{-1, -1};

    /** Context-dependent status. */
    private ContextValue status =
	new TypedContextValue(Status.class, Status.BLANK);

    /** Contex-dependent list of node-ends linked before this one. */
    private LLQueue predecessors = new LLQueue();

    /** Contex-dependent list of node-ends linked after this one. */
    private LLQueue successors = new LLQueue();

    public PNodeEnd(End end, PNode node) {
	this.end = end;
	this.node = node;
    }

    public static PNodeEnd fromRef(Map idToNodeMap, NodeEndRef ref) {
	PNode node = (PNode)idToNodeMap.get(ref.getNode());
	End end = ref.getEnd();
	Debug.expect(node != null, "Can't find node for", ref);
	if (end == End.BEGIN)
	    return node.getBegin();
	else if (end == End.END)
	    return node.getEnd();
	else
	    throw new ConsistencyException("Bad end in", ref);
    }

    public static void addOrdering(Map idToNodeMap, Ordering ord) {
	PNodeEnd from = fromRef(idToNodeMap, ord.getFrom());
	PNodeEnd to = fromRef(idToNodeMap, ord.getTo());
	from.linkBefore(to);
    }

    public PNode getNode() {
	return node;
    }

    public End getEnd() {
	return end;
    }

    public PNodeEnd getOtherNodeEnd() {
	return end == End.BEGIN ? getNode().getEnd() : getNode().getBegin();
    }

    public Status getStatus() {
	return (Status)status.get();
    }

    public void setStatus(Status s) {
	status.set(s);
	PNode.allComputeStatus(getSuccessors());
    }

    public void computeStatus() {
// 	if (!PNode.computeStatus) return;
	if (!((Ip2ModelManager)node.modelManager).computeStatus) return;
	Status stat = getStatus();
	if (stat == Status.BLANK) {
	    if (PNode.allHaveStatus(getPredecessors(), Status.COMPLETE)) {
		setStatus(Status.POSSIBLE);
		if (end == End.BEGIN)
		    node.setStatus(Status.POSSIBLE);
	    }
	}
    }

    public List getPredecessors() {
	return predecessors.contents();
    }

    public List getSuccessors() {
	return successors.contents();
    }

    public void linkBefore(PNodeEnd to) {
	if (to.node != this.node)
	    Debug.noteln("-- " + this + " --> " + to);
	if (!successors.contains(to)) {
	    successors.add(to);
	    to.predecessors.add(this);
	}
    }

    public void linkAfter(PNodeEnd from) {
	if (from.node != this.node)
	    Debug.noteln("-- " + from + " --> " + this);
	if (!from.successors.contains(this)) {
	    from.successors.add(this);
	    predecessors.add(from);
	}
    }

    public void unlink() {
	// Link every predecessor of this node-end before every successor.
	for (Iterator pi = predecessors.iterator(); pi.hasNext();) {
	    PNodeEnd pre = (PNodeEnd)pi.next();
	    for (Iterator si = successors.iterator(); si.hasNext();) {
		PNodeEnd suc = (PNodeEnd)si.next();
		pre.linkBefore(suc);
	    }
	}
	// Tell predecessors and successors to forget their links
	// to this node-end.
	for (Iterator pi = predecessors.iterator(); pi.hasNext();) {
	    PNodeEnd pre = (PNodeEnd)pi.next();
	    pre.successors.remove(this);
	}
	for (Iterator si = successors.iterator(); si.hasNext();) {
	    PNodeEnd suc = (PNodeEnd)si.next();
	    suc.predecessors.remove(this);
	}
    }

    /**
     * Determines whether this node-end is marked by the indicated mark.
     *
     * <p>Each PNodeEnd has, internally, a small array of marks, so
     * that node-ends can be marked for more than one purpose by an
     * angorithm that uses marks.  That's what the 'markIndex'
     * parameter is about.  It is obviously necessary to be very
     * careful with this, and it is best to use marks only "locally"
     * within an algorithm, rather than relying on a markIndex to
     * have a lasting significance.   Marks are not context-layered,
     * which is another sense in which they should be used "locally".</p>
     *
     * <p>Marks are obtained by calling {@link Ip2ModelManager#nextMark()},
     * using this node's model-manager, but marking methods often do that
     * for you automatically and then return the new mark when they're
     * done.</p>
     */
    public boolean isMarked(int markIndex, long mark) {
	return this.marks[markIndex] == mark;
    }

    /**
     * Returns the current mark-value for this node-end.  To check
     * whether this node-end has a particular make value, the
     * {@link #isMarked(int, long)} method should be called instead.
     */
    public long getMark(int markIndex) {
	return this.marks[markIndex];
    }

    /**
     * Marks all node-ends that are forced by ordering contstraints
     * to be after this one, then returns the (new) mark.  This node-end
     * is also marked.  Then the {@link #isMarked(int, long)} method can
     * be used to tell whether a node-end is marked by that mark.
     * See that method for more about marking.
     */
    public long markShadowedEnds(int markIndex) {
	long mark = ((Ip2ModelManager)node.modelManager).nextMark();
	markShadowedEnds(this, markIndex, mark);
	return mark;
    }

    private void markShadowedEnds(PNodeEnd at, int markIndex, long mark) {
	if (at.marks[markIndex] == mark)
	    return;
	else {
	    at.marks[markIndex] = mark;
	    for (Iterator i = at.getSuccessors().iterator(); i.hasNext();) {
		PNodeEnd s = (PNodeEnd)i.next();
		markShadowedEnds(s, markIndex, mark);
	    }
	}
    }

    /**
     * Marks all node-ends that are forced by ordering contstraints
     * to be before this one, then returns the (new) mark.  This node-end
     * is also marked.  Then the {@link #isMarked(int, long)} method can
     * be used to tell whether a node-end is marked by that mark.
     * See that method for more about marking.
     */
    public long markEndsBefore(int markIndex) {
	long mark = ((Ip2ModelManager)node.modelManager).nextMark();
	markEndsBefore(this, markIndex, mark);
	return mark;
    }

    private void markEndsBefore(PNodeEnd at, int markIndex, long mark) {
	if (at.marks[markIndex] == mark)
	    return;
	else {
	    at.marks[markIndex] = mark;
	    for (Iterator i = at.getPredecessors().iterator(); i.hasNext();) {
		PNodeEnd p = (PNodeEnd)i.next();
		markEndsBefore(p, markIndex, mark);
	    }
	}
    }

    /**
     * Returns a list containing all the node-ends that are forced
     * by ordering constraints to be after this one.
     */
    public List getShadowedEnds() {
	// /\/: It would be better to return a Set, since we
	// usually want the result only so we can see whether
	// various node-ends are in it.
	TopologicalSorter tsort = new TopologicalSorter() {
	    protected Collection getChildren(Object ne) {
		return ((PNodeEnd)ne).getSuccessors();
	    }
	};
	return tsort.sort(Lisp.list(this));
    }

    public String toString() {
	return end + "_of " + node;
    }

}
