/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Aug 16 23:27:25 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.icore.plan.build;

import java.util.*;

import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.plan.*;
import ix.util.*;

import ix.test.PlanBuilderTest;

/**
 * Plan-construction utility.
 */
public class ExamplePlanBuilder implements PlanBuilder {

    /** Used by {@link #genId(String)}. */
    protected Gensym.Generator nameGen = new Gensym.Generator();

    // Objects to be included in the top level of the plan.
    protected List issues = new LinkedList();
    protected List activities = new LinkedList();
    protected ListOfConstrainer constraints = new LinkedListOfConstrainer();
    protected Annotations annotations = new Annotations();

    /** Maps a object to a list of its subobjects. */
    protected MultiMap itemToSubitems = new MultiHashMap();

    // Used when building the plan.  See getPlan(). */
    protected Plan plan = new Plan();
    protected ListOfPlanIssue planIssues = new LinkedListOfPlanIssue();
    protected ListOfPlanIssueRefinement issueRefinements = 
	new LinkedListOfPlanIssueRefinement();
    protected ListOfPlanNode planNodes = new LinkedListOfPlanNode();
    protected ListOfPlanRefinement planRefinements =
	new LinkedListOfPlanRefinement();

    public ExamplePlanBuilder() {}

    public void addIssue(Issue issue) {
	issues.add(issue);
    }

    public void addSubissue(Issue issue, Issue subissue) {
	itemToSubitems.addValue(issue, subissue);
    }

    public void addActivity(Activity activity) {
	activities.add(activity);
    }

    public void addSubactivity(Activity activity, Activity subactivity) {
	itemToSubitems.addValue(activity, subactivity);
    }

    public void addConstraint(Constraint c) {
	constraints.add(c);
    }

    public void setAnnotation(Object key, Object value) {
	annotations.put(key, value);
    }

    /**
     * Generates a short id for use within the plan.
     *
     * @see ix.util.Gensym.Generator#nextName(String)
     */
    protected Name genId(String base) {
	return nameGen.nextName(base);
    }

    /**
     * Returns the subissues of an issue, or the subactivities
     * of an activity, that have been added so far.
     */
    protected List getSubitems(Object item) {
	return (List)itemToSubitems.get(item);
    }

    public Plan getPlan() {

	// Issues
	new IssueExpander(planIssues, issueRefinements).walk(issues);

	// Activities
	new ActivityExpander(planNodes, planRefinements).walk(activities);

	// Fill in the plan
	fillInPlan();

	return plan;

    }

    /**
     * Installs any non-empty values in the relevant fields
     * of the {@link Plan}.  Called by {@link #getPlan()}.
     */
    protected void fillInPlan() {
	if (!planIssues.isEmpty())
	    plan.setPlanIssues(planIssues);
	if (!issueRefinements.isEmpty())
	    plan.setPlanIssueRefinements(issueRefinements);
	if (!planNodes.isEmpty())
	    plan.setPlanNodes(planNodes);
	if (!planRefinements.isEmpty())
	    plan.setPlanRefinements(planRefinements);
	if (!constraints.isEmpty())
	    plan.setConstraints(constraints);
	if (!annotations.isEmpty())
	    plan.setAnnotations(annotations);
    }

    /**
     * Constructs the plan-items and refinements used to tie an
     * issue or activity to its subissues or subactivities.
     */
    protected abstract class Expander {

	protected List planItems;
	protected List planRefinements;

	public Expander(List planItems, List planRefinements) {
	    this.planItems = planItems;
	    this.planRefinements = planRefinements;
	}

	public void walk(List items) {
	    walk(items, planItems, null);
	}

	protected void walk(List items, List planItems,
			    AbstractPlanItem parentPI) {
	    for (Iterator i = items.iterator(); i.hasNext();) {
		TaskItem item = (TaskItem)i.next();
		List children = getSubitems(item);
		AbstractPlanItem pi = makePlanItem(item, parentPI);
		planItems.add(pi);
		if (children != null)
		    buildExpansion(pi, children);
	    }
	}

	protected void buildExpansion(AbstractPlanItem pi, List children) {

	    // Construct the refinement that will describe the
	    // expansion of the item.
	    AbstractRefinement ref = makePlanRefinement();
	    planRefinements.add(ref);

	    // Connect the plan-item and the refinement.
	    pi.setExpansion(ref.getId());
	    ref.setExpands(pi.getId());

	    // Visit children
	    List childPlanItems = new LinkedList();
	    ref.setSubitems(childPlanItems);
	    walk(children, childPlanItems, pi);

	    Debug.expect(childPlanItems.size() == children.size());

	}

	protected abstract AbstractPlanItem
	          makePlanItem(TaskItem item, AbstractPlanItem parentPI);

	protected abstract AbstractRefinement makePlanRefinement();

    }

    protected class IssueExpander extends Expander {

	public IssueExpander(List planItems, List planRefinements) {
	    super(planItems, planRefinements);
	}

	/** Returns a {@link PlanIssue}. */
	protected AbstractPlanItem makePlanItem(TaskItem item,
						AbstractPlanItem parentPI) {
	    String baseName = parentPI == null
		? "issue"
		: parentPI.getId().toString();
	    return new PlanIssue(genId(baseName), (Issue)item);
	}

	/** Returns a {@link PlanIssueRefinement}. */
	protected AbstractRefinement makePlanRefinement() {
	    PlanIssueRefinement ref = new PlanIssueRefinement();
	    ref.setId(genId("issue-refinement"));
	    return ref;
	}

    }

    protected class ActivityExpander extends Expander {

	public ActivityExpander(List planItems, List planRefinements) {
	    super(planItems, planRefinements);
	}

	/** Returns a {@link PlanNode}. */
	protected AbstractPlanItem makePlanItem(TaskItem item,
						AbstractPlanItem parentPI) {
	    String baseName = parentPI == null
		? "node"
		: parentPI.getId().toString();
	    return new PlanNode(genId(baseName), (Activity)item);
	}

	/** Returns a {@link PlanRefinement}. */
	protected AbstractRefinement makePlanRefinement() {
	    PlanRefinement ref = new PlanRefinement();
	    ref.setId(genId("refinement"));
	    return ref;
	}

    }

    /**
     * Main program for testing.  Makes a {@link PlanBuilderTest}
     * and calls {@link PlanBuilderTest#test(PlanBuilder)} on
     * an instance of ExamplePlanBuilder.
     */
    public static void main (String[] argv) throws Exception {
	new PlanBuilderTest().test(new ExamplePlanBuilder());
    }

}
