/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Sep 19 02:07:42 2007 by Jeff Dalton
 * Copyright: (c) 2007, AIAI, University of Edinburgh
 */

package ix.icore.plan;

import java.util.*;

import ix.icore.domain.*;
import ix.icore.*;

import ix.ip2.ActivityItem; // for EXPANSION_REFINEMENT_NAME /\/

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

/**
 * A plan constructed from a domain that's meant to represent a plan.
 * The domain must follow some of the conventions used by
 * {@link PlanAsDomain}.
 */
public class DomainAsPlan extends Plan {

    static final Symbol
	EXPANDS = Symbol.intern("expands"),
	WORLD_STATE = Symbol.intern("world-state");

    protected Map<Symbol,PlanVariable> nameToPlanVarMap = 
	new HashMap<Symbol,PlanVariable>();

    protected Map<Name,NodeSpec> nodeIdToNodeSpecMap =
	new HashMap<Name,NodeSpec>();

    protected Map<Name,Refinement> nodeIdToRefinementMap =
	new HashMap<Name,Refinement>();

    protected Gensym.Generator nameGen = new Gensym.Generator();

    public DomainAsPlan() {
    }

    public DomainAsPlan(Domain domain) {
	// Make maps we'll need later.
	new Pass1().walk(domain);
	fillNodeIdToRefinementMap(domain);
	// The top level of the plan is in a refinement rather than
	// directly in the domain.
	Refinement top = domain.getNamedRefinement("plan-top-level");
	// Variable declarations
	setPlanVariableDeclarations
	    (makePlanVarDcls(top, VariableScope.GLOBAL));
	// Issues
	// Nodes
	setPlanNodes(makePlanNodes(top));
	// Plan-refinements
	if (domain.getRefinements() != null) {
	    ListOfPlanRefinement prs = new LinkedListOfPlanRefinement();
	    for (Refinement r: (List<Refinement>)domain.getRefinements())
		if (r != top)
		    prs.add(makePlanRefinement(r));
	    setPlanRefinements(prs);
	}
	// Constraints
	setConstraints(getConstrainers(top));
	// World state
	Map stateMap = (Map)top.getAnnotation(WORLD_STATE);
	if (stateMap != null)
	    setWorldState(PatternAssignment.mapToAssignments(stateMap));
	// We're done.
    }

    void fillNodeIdToRefinementMap(Domain domain) {
	if (domain.getRefinements() == null)
	    return;
	Refinement top = domain.getNamedRefinement("plan-top-level");
	List<String> dontExpand = new LinkedList<String>();
	List<String> expandNonexistentNode = new LinkedList<String>();
	for (Refinement r: (List<Refinement>)domain.getRefinements()) {
	    if (r == top) continue;
	    Object expands = r.getAnnotation(EXPANDS);
	    if (expands == null)
		dontExpand.add(r.getName());
	    else {
		Name ex = Name.valueOf(expands);
		if (nodeIdToNodeSpecMap.get(ex) == null)
		    expandNonexistentNode.add(r.getName());
		else
		    nodeIdToRefinementMap.put(ex, r);
	    }
	}
	// Report any problems.
	if (dontExpand.isEmpty() && expandNonexistentNode.isEmpty())
	    return;
	String message = "Some refinements aren't connected properly: ";
	if (!dontExpand.isEmpty())
	    message += "Refinements that lack an expands annotation: " +
		Strings.conjunction(dontExpand) + ".  ";
	if (!expandNonexistentNode.isEmpty())
	    message += "Refinements that expand a nonexistent node: " +
		Strings.conjunction(expandNonexistentNode) + ".";
	throw new InvalidPlanDomain(message);
    }

    public static class InvalidPlanDomain extends RuntimeException {
	public InvalidPlanDomain(String message) {
	    super(message);
	}
    }

    class Pass1 extends ObjectWalker {
	@Override
	public void walk(Object o) {
	    if (o instanceof ItemVar)
		walkItemVar((ItemVar)o);
	    else if (o instanceof NodeSpec)
		walkNodeSpec((NodeSpec)o);
	    else
		super.walk(o);
	}
	private void walkItemVar(ItemVar v) {
	    // We generate a new Id rather than assume the part
	    // after the last "/" is unique.  PlanAsDomain has that
	    // part unique on its own (because it's taken from a Plan),
	    // but our domain may not have been produced by PlanAsDomain.
	    PlanVariable pv = nameToPlanVarMap.get(v);
	    if (pv == null) {
		pv = new PlanVariable(nameGen.nextName("var"),
				      namePart(v));
		nameToPlanVarMap.put(v, pv);
	    }
	}
	private void walkNodeSpec(NodeSpec spec) {
	    nodeIdToNodeSpecMap.put(spec.getId(), spec);
	}
	@Override
	public void visitElement(Object elt) {
	    if (!(elt instanceof Domain)) // avoid confusion
		walk(elt);
	}
    }

    PlanRefinement makePlanRefinement(Refinement r) {
	PlanRefinement pr = new PlanRefinement();
	pr.setId(Name.valueOf(r.getName()));
	Name expands = Name.valueOf(r.getAnnotation(EXPANDS));
	pr.setExpands(expands);
	// Variable declarations
	pr.setPlanVariableDeclarations
	    (makePlanVarDcls(r, VariableScope.LOCAL));
	// Nodes
	pr.setPlanNodes(makePlanNodes(r));
	// Constraints
	pr.setConstraints(getConstrainers(r));
	// We're done.
	return pr;
    }

    ListOfPlanNode makePlanNodes(Refinement r) {
	if (r.getNodes() == null)
	    return null;
	ListOfPlanNode result = new LinkedListOfPlanNode();
	for (NodeSpec spec: (List<NodeSpec>)r.getNodes()) {
	    result.add(makePlanNode(spec));
	}
	return result;
    }

    PlanNode makePlanNode(NodeSpec spec) {
	LList pattern = patternForPlan(spec.getPattern());
	Activity act = new Activity(pattern);
	PlanNode pn = new PlanNode(spec.getId(), act);
	Refinement expansionRefinement =
	    nodeIdToRefinementMap.get(spec.getId());
	if (expansionRefinement != null) {
	    pn.setExpansion(Name.valueOf(expansionRefinement.getName()));
	    Object originalExpander =
		expansionRefinement
		    .getAnnotation(ActivityItem.EXPANSION_REFINEMENT_NAME);
	    if (originalExpander != null)
		act.setAnnotation(ActivityItem.EXPANSION_REFINEMENT_NAME,
				  originalExpander);
	}
	return pn;
    }

    ListOfPlanVariableDeclaration makePlanVarDcls(Refinement r,
						  VariableScope scope) {
	ListOfVariableDeclaration dcls = r.getVariableDeclarations();
	if (dcls == null)
	    return null;
	ListOfPlanVariableDeclaration result =
	    new LinkedListOfPlanVariableDeclaration();
	for (VariableDeclaration d: (List<VariableDeclaration>)dcls) {
	    result.add(makePlanVarDcl(d.getName(), scope));
	}
	return result;
    }

    ListOfConstrainer getConstrainers(Refinement r) {
	ListOfConstrainer result = new LinkedListOfConstrainer();
	if (r.getOrderings() != null)
	    result.addAll(r.getOrderings());
	if (r.getConstraints() != null)
	    result.addAll(r.getConstraints());
	return result.isEmpty() ? null : result;
    }

    /* * * Representing PlanVariables and PlanVariableDeclarations * * */

    LList patternForPlan(List pattern) {
	LListCollector result = new LListCollector();
	for (Object item: pattern) {
	    if (item instanceof ItemVar)
		result.add(getPlanVar((ItemVar)item));
	    else if (item instanceof List)
		result.add(patternForPlan((List)item));
	    else
		result.add(item);
	}
	return result.contents();
    }

    ItemVar namePart(ItemVar v) {
	return
	    Util.mustBe(ItemVar.class,
			Symbol.intern(Strings.beforeLast(".", v.toString())));
    }

    PlanVariable getPlanVar(ItemVar v) {
	Debug.noteln("MAP -----", nameToPlanVarMap);
	PlanVariable pv = nameToPlanVarMap.get(v);
	Debug.expect(pv != null, "Can't find plan variable for", v);
	return pv;
    }

    PlanVariableDeclaration makePlanVarDcl(ItemVar v, VariableScope s) {
	PlanVariable pv = getPlanVar(v);
	PlanVariableDeclaration dcl = new PlanVariableDeclaration();
	dcl.setId(pv.getId());
	dcl.setName(pv.getName());
	dcl.setScope(s);
	return dcl;
    }

    /* * * Main program for testing * * */

    public static void main(String[] argv) {
	Debug.off();
	Parameters.processCommandLineArguments(argv);
	String domainName = Parameters.getParameter("domain");
	String output = Parameters.getParameter("output");
	if (domainName == null || output == null) {
	    System.out.println("Usage: Must specify domain= and output=");
	    System.exit(1);
	}
	Domain domain = XML.readObject(Domain.class, domainName);
	Plan plan = new DomainAsPlan(domain);
	XML.writeObject(plan, output);
    }

}
