/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Mar  3 17:25:21 2010 by Jeff Dalton
 * Copyright: (c) 2006, 2009, 2010, AIAI, University of Edinburgh
 */

package ix.icore.log;

import org.jdom.Document;
import org.jdom.output.*;

import java.io.*;
import java.util.*;

import ix.icore.IXAgent;
import ix.icore.Sendable;

import ix.util.*;
import ix.util.ipc.IPCListener;
import ix.util.xml.*;

public class EventLogger {

    IXAgent agent;

    XMLTranslator xmlt = XML.config().makeXMLTranslator();

    XMLOutputter outputter = XML.makePrettyXMLOutputter();

    String logDir;

    String logFileName;

    OutputStream logStream = null;
    FileDescriptor logFD = null;

    public EventLogger(IXAgent agent) {
	this.agent = agent;
	// Adjust the outputter's format to omit the XML declaration.
	Format format = outputter.getFormat(); // gets a clone
	format.setOmitDeclaration(true);
	outputter.setFormat(format);
    }

    public void processCommandLineArguments() {
	logDir = Parameters.getParameter("log-directory");
    }

    String makeLogName() {
	String datePart = new ISODateFormat().formatDateTime(new Date());
	datePart = Strings.replaceChars(":", "-", datePart);
        String name = Parameters.getParameter
            ("log-name", // defaults to:
             agent.getAgentSymbolName());
	return name + "-" + datePart + ".log";
    }

    public synchronized void install() {
	IPC.addIPCListener(new LoggingIPCListener());
    }

    public synchronized boolean isLogging() {
	return logStream != null;
    }

    public synchronized void startLogging() {
	try {
	    logFileName = logDir + "/" + makeLogName();
            FileOutputStream out = new FileOutputStream(logFileName);
	    logStream = new BufferedOutputStream(out);
            logFD = out.getFD();
	    log(new HistoryComment("Logging started."));
	}
	catch (Throwable t) {
	    throw new RethrownException("Can't start logging because", t);
	}
    }

    public synchronized void log(HistoryEvent event) {
	if (logStream == null)	// !isLogging()
	    return;
	try {
	    if (event.getDate() == null)
		// Should be set when event made? /\/
		event.setDate(new Date());
	    if (event.getAgent() == null)
		// It's not clear we need to set this now
		// as opposed to if / when we're merging logs.
		event.setAgent(agent.getAgentSymbolName());
	    Document doc = xmlt.objectToDocument(event);
	    outputter.output(doc, logStream);
	    logStream.flush();
            logFD.sync();
	}
	catch (Throwable t) {
	    throw new RethrownException("Problem while logging:", t);
	}
    }

    class LoggingIPCListener extends IPCListener {
	LoggingIPCListener() { }
	public void sendingMessage(Object destination, Object contents) {
	    ; // do nothing
	}
	public void messageSent(Object destination, Object contents) {
	    log(new MessageSent(destination, contents));
	}
	public void messageReceived(IPC.InputMessage message) {
	    Object contents = message.getContents();
	    Object from = contents instanceof Sendable
		? ((Sendable)contents).getSenderId()
		: null;
	    log(new MessageReceived(from, contents));
	}
    }

    /** Simple test program. */
    public static void main(String[] argv) {
	IXAgent ip2 = new ix.test.PlainIp2();
	ip2.mainStartup(argv);
	EventLogger logger = new EventLogger(ip2);
	logger.processCommandLineArguments();
	logger.startLogging();
    }

}
