/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Jun 17 18:09:13 2008 by Jeff Dalton
 * Copyright: (c) 2005-2006, 2008, AIAI, University of Edinburgh
 */

package ix.icore.domain;

import ix.util.*;
import ix.util.lisp.Symbol;
import ix.util.lisp.ItemVar;

import ix.icore.Variable;

/**
 * Represents a duration range.  The min and max values are meant to be
 * {@link Duration}s.  However, they can be other objects, typically
 * variables, at other times.  /\/ ...
 */
public class TimeWindow implements java.io.Serializable {

    protected Object min;	// Duration or a variable
    protected Object max;	// Duration, null, or a variable

    public TimeWindow() { }

    public TimeWindow(Object min, Object max) {
	setMin(min);
	setMax(max);
    }

    public TimeWindow(Duration min, Duration max) {
	this.min = min;
	this.max = max;
	checkRange();
    }

    public Duration getMinDuration() {
	return (Duration)min;
    }

    public Duration getMaxDuration() {
	return (Duration)max;
    }

    public long getMinMilliseconds() {
	return ((Duration)min).asMilliseconds();
    }

    public long getMaxMilliseconds() {
	return max == null ? Inf.INFINITY : ((Duration)max).asMilliseconds();
    }

    public Object getMin() {
	return min;
    }

    public void setMin(Object d) {
	this.min = toDurationIfPossible(d);
	checkRange();
    }

    public Object getMax() {
	return max;
    }

    public void setMax(Object d) {
	if ((d instanceof Symbol || d instanceof String) && Inf.isInfName(d))
	    this.max = null;
	else
	    this.max = toDurationIfPossible(d);
	checkRange();
    }

    public boolean hasUnboundedMax() {
	return max == null;
    }

    private void checkRange() {
	if (min instanceof Duration && max instanceof Duration) {
	    if (((Duration)min).compareTo((Duration)max) > 0)
		throw new ConsistencyException
		    ("Min duration " + min + " is not <= max duration " + max);
	}
    }
		
    private Object toDurationIfPossible(Object d) {
	if (d == null)
	    return d;
	else if (d instanceof Duration)
	    return d;
	else if (d instanceof ItemVar || d instanceof Variable)
	    return d;
	else if (d instanceof String)
	    return new Duration((String)d);
	else if (d instanceof Symbol)
	    return new Duration(d.toString());
	else if (d instanceof Number)
	    return new Duration(((Number)d).longValue());
	else
	    throw new ClassCastException
		("Invalid duration: " + d);
    }

    public String toString() {
	if (min instanceof Duration)
	    return "TimeWindow["
		+     getMinDuration().toISOString()
		+     ".."
		+     (hasUnboundedMax() ? "inf"
		         : getMaxDuration().toISOString())
		+  "]";
	else
	    return "TimeWindow[" + min + ".." + max + "]";
    }

}
