/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jun 18 18:42:51 2008 by Jeff Dalton
 * Copyright: (c) 2002 - 2008, AIAI, University of Edinburgh
 */

package ix.icore.domain;

import java.util.*;

import ix.icore.*;
import ix.iface.domain.SyntaxException; // need better package for it /\/
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

import static ix.iplan.ServiceSymbols.S_INPUT_OBJECTS;  //\/
import static ix.iplan.ServiceSymbols.S_OUTPUT_OBJECTS;	//\/
import static ix.iplan.ServiceSymbols.S_TYPE;           //\/


/**
 * A Refinement describes one way to carry out a possible activity
 * in a process being modelled.
 */

public class Refinement extends AbstractIXObject implements Named, Cloneable {
    protected String name;
    protected LList pattern;
    protected ListOfVariableDeclaration variableDeclarations;
    protected ListOfNodeSpec nodes;
    protected ListOfOrdering orderings;
    protected ListOfConstraint constraints;
    protected ListOfIssue issues;
    // annotations
    // protected String comments = "";

    // Need a better place for these, here and in the Ip2 Model Manager /\/
    public static final Symbol
	S_WORLD_STATE     = Symbol.intern("world-state"),
	S_CONDITION       = Symbol.intern("condition"),
	S_EFFECT          = Symbol.intern("effect"),
	S_COMPUTE         = Symbol.intern("compute"),
	S_MULTIPLE_ANSWER = Symbol.intern("multiple-answer"),
	S_SELF            = Symbol.intern("self");

    public Refinement() {
    }

    public Refinement (String name, LList pattern) {
	this.name = name;
	this.pattern = pattern;
	Debug.noteln("New blank refinement " + Util.quote(name) + " for",
		     pattern);
    }

    public String getName() { return name; }
    public void setName(String name) { this.name = name; }

    public LList getPattern() {	return pattern; }
    public void setPattern(LList pattern) { this.pattern = pattern; }

    public ListOfVariableDeclaration getVariableDeclarations() {
	return variableDeclarations;
    }
    public void setVariableDeclarations
	           (ListOfVariableDeclaration variableDeclarations) {
        this.variableDeclarations = variableDeclarations;
    }

    public ListOfNodeSpec getNodes() { return nodes; }
    public void setNodes(ListOfNodeSpec nodes) { this.nodes = nodes; }

    public ListOfOrdering getOrderings() { return orderings; }
    public void setOrderings(ListOfOrdering orderings) {
	this.orderings = orderings;
    }

    public ListOfConstraint getConstraints() { return constraints; }
    public void setConstraints(ListOfConstraint cs) { this.constraints = cs; }

    public ListOfIssue getIssues() { return issues; }
    public void setIssues(ListOfIssue issues) { this.issues = issues; }

    // public String getComments() { return comments; }
    // public void setComments(String comments) { this.comments = comments; }

    public String getComments() {
	// Backwards compatibility: never returns null. /\/
	String c = super.getComments(); // get from annotation
	return c == null ? "" : c;      // make sure result is not null
    }

    // /\/: Backwards-compatibility for I-DE.
    public void setVariableDeclarations(List dcls) {
	setVariableDeclarations(dcls == null ? null :
				new LinkedListOfVariableDeclaration(dcls));
    }
    public void setNodes(List nodes) {
	setNodes(nodes == null ? null : new LinkedListOfNodeSpec(nodes));
    }
    public void setOrderings(List orderings) {
	setOrderings(orderings == null ? null :
		     new LinkedListOfOrdering(orderings));
    }
    public void setConstraints(List constraints) {
	setConstraints(constraints == null ? null :
		       new LinkedListOfConstraint(constraints));
    }
    public void setIssues(List issues) {
	setIssues(issues == null ? null : new LinkedListOfIssue(issues));
    }

    /*
    public ListOfPatternAssignment getFilterConditions() {
	// N.B. for now returns a list of PatternAssignments. /\/
	// Also returns all world-state conditions, not just filters. /\/
	ListOfPatternAssignment result = new LinkedListOfPatternAssignment();
	for (Iterator i = Collect.iterator(constraints); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    if (c.getType() == S_WORLD_STATE
		    && c.getRelation() == S_CONDITION) {
		PatternAssignment pv = (PatternAssignment)c.getParameter(0);
		result.add(pv);
	    }
	}
	return result;
    }
    */

    public ListOfConstraint getFilterConstraints() {
	// Returns all world-state conditions, not just filters. /\/
	ListOfConstraint result = new LinkedListOfConstraint();
	for (Iterator i = Collect.iterator(constraints); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    if (c.getType() == S_COMPUTE
		  || (c.getType() == S_WORLD_STATE
		      && c.getRelation() == S_CONDITION)) {
		result.add(c);
	    }
	}
	return result;
    }

    public ListOfPatternAssignment getConditions() {
	// N.B. for now returns a list of PatternAssignments /\/
	ListOfPatternAssignment result = new LinkedListOfPatternAssignment();
	for (Iterator i = Collect.iterator(constraints); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    if (c.getType() == S_WORLD_STATE
		    && c.getRelation() == S_CONDITION) {
		PatternAssignment pv = (PatternAssignment)c.getParameter(0);
		result.add(pv);
	    }
	}
	return result;
    }

    public ListOfPatternAssignment getEffects() {
	// N.B. for now returns a list of PatternAssignments /\/
	ListOfPatternAssignment result = new LinkedListOfPatternAssignment();
	for (Iterator i = Collect.iterator(constraints); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    if (c.getType() == S_WORLD_STATE
		    && c.getRelation() == S_EFFECT) {
		PatternAssignment pv = (PatternAssignment)c.getParameter(0);
		result.add(pv);
	    }
	}
	return result;
    }

    public ListOfConstraint getConstraints(Symbol type, Symbol relation) {
	ListOfConstraint result = new LinkedListOfConstraint();
	if (constraints != null) {
	    for (Constraint c: constraints) {
		if (c.getType() == type && c.getRelation() == relation)
		    result.add(c);
	    }
	}
	return result;
    }

    public ListOfConstraint getConditionConstraints() {
	return getConstraints(S_WORLD_STATE, S_CONDITION);
    }

    public ListOfConstraint getEffectConstraints() {
	return getConstraints(S_WORLD_STATE, S_EFFECT);
    }

    public ListOfConstraint getOtherConstraints() {
	ListOfConstraint result = new LinkedListOfConstraint();
	for (Iterator i = Collect.iterator(constraints); i.hasNext();) {
	    Constraint c = (Constraint)i.next();
	    if (c.getType() != S_WORLD_STATE
		    && c.getType() != S_COMPUTE
		    && c.getType() != Ordering.S_TEMPORAL)
		result.add(c);
	}
	return result;
    }

    /**
     * Checks the consistency of this refinement.
     *
     * @throws SyntaxException if it finds an inconsistency
     */
    public void checkConsistency() {
	// Debug.noteln("Checking consistency of", this);

	// Check that all variables have been declared.
//	checkVariableRefs();

	// Check that constraints don't refer to undefined nodes
	checkNodeRefs();

	// Check inputs and outputs against conditions and effects
	checkInputsAndOutputs();

    }

    /**
     * Checks that the set of variables declared in this refinement
     * and the set of variables referred to (used) in this refinement
     * are the same.
     *
     * @throws SyntaxException if the sets are different.
     */
    protected void checkVariableRefs() {
	Set declared = getDeclaredVariables();
	Set used = getVariablesUsed();
	Set undeclared =(Set)Collect.difference(used, declared);
	Set unused = (Set)Collect.difference(declared, used);
	Debug.noteln("Declared", declared);
	Debug.noteln("Used", used);
	Debug.noteln("Undeclared", undeclared);
	Debug.noteln("Unused", unused);
	Debug.expectEquals(used, getVariablesUsed2()); // /\/ REMOVE THIS
	if (!unused.isEmpty() && !undeclared.isEmpty())
	    throw new SyntaxException
		("Incorrect variable declarations in refinement " + name +
		 ".  Declared but not used: " + 
		       Strings.conjunction(unused) +
		 ".  Used but not declared: " +
                       Strings.conjunction(undeclared) + ".");
	if (!unused.isEmpty())
	    throw new SyntaxException
		("Unused variables in refinement " + name + ": "
		 + Strings.conjunction(unused));
	if (!undeclared.isEmpty())
	    throw new SyntaxException
		("Undeclared variables in refinement " + name + ": "
		 + Strings.conjunction(undeclared));
    }

    /**
     * Returns a set containing the {@link ItemVar}s used
     * within this refinement.
     */
    public SortedSet getVariablesUsed() {
	// /\/: We don't have a way to just look at all the ItemVars,
	// but instantiation does it along the way, even though we
	// discard the instantiated objects.  We can't just use
	// ObjectWalker.collectIf, because it would include the
	// var declarations in the walk.
	final SortedSet vars = new TreeSet();
	instantiate(Matcher.emptyEnv, new Function1() {
	    public Object funcall(Object arg) {
		ItemVar var = (ItemVar)arg;
		vars.add(var);
		return arg;	// have to return something ...
	    }
	});
	return vars;
    }

    private SortedSet getVariablesUsed2() {
	ListOfVariableDeclaration saved = variableDeclarations;
	variableDeclarations = null;
	try {
	    return (SortedSet)ObjectWalker.collectIf
		(this, new TreeSet(), Fn.isInstanceOf(ItemVar.class));
	}
	finally {
	    variableDeclarations = saved;
	}
    }

    /**
     * Returns a set containing the {@link ItemVar}s declared
     * with this refinement.
     */
    public SortedSet getDeclaredVariables() {
	return (SortedSet)
	    Collect.map(new TreeSet(), 
			Collect.ensureList(variableDeclarations),
			Fn.accessor(VariableDeclaration.class, "getName"));
    }

    /**
     * Returns true iff this specified variable is declared in
     * this refinement.
     */
    protected boolean isDeclared(ItemVar v) {
	if (variableDeclarations == null)
	    return false;
	for (Iterator i = variableDeclarations.iterator(); i.hasNext();) {
	    VariableDeclaration dcl = (VariableDeclaration)i.next();
	    if (dcl.getName() == v)
		return true;
	}
	return false;
    }

    /**
     * Checks that every node mentioned in a constraint is defined
     * in this refinement.
     *
     * @throws SyntaxException if there is a reference to an undefined node.
     */
    protected void checkNodeRefs() {
	Set defined = new HashSet();
	Set undefined = new TreeSet();
	for (Iterator i = Collect.iterator(getNodes()); i.hasNext();) {
	    defined.add(((NodeSpec)i.next()).getId());
	}
	collectUndefinedNodeRefs(getOrderings(), defined, undefined);
	collectUndefinedNodeRefs(getConstraints(), defined, undefined);
	if (!undefined.isEmpty())
	    throw new SyntaxException
		("There are references to undefined nodes in" +
		 " refinement " + name + ": " +
		 Strings.conjunction(undefined));
    }

    private void collectUndefinedNodeRefs(Object where,
					  final Set defined,
					  final Set undefined) {
	if (where == null) return;
	ObjectWalker walker = new ObjectWalker() {
	    public void walk(Object o) {
		if (o instanceof NodeEndRef) {
		    Name node = ((NodeEndRef)o).getNode();
		    if (!"self".equals(node.toString()))
			if (!defined.contains(node))
			    undefined.add(node);
		}
		else
		    super.walk(o);
	    }
	};
	walker.walk(where);
    }

    protected void checkInputsAndOutputs() {
	checkParametersAgainstConstraints(S_INPUT_OBJECTS, getConditions());
	checkParametersAgainstConstraints(S_OUTPUT_OBJECTS, getEffects());
    }

    protected void checkParametersAgainstConstraints(Symbol key,
						     List constraints) {
	List params = (List)getAnnotation(key);
	if (params == null)
	    return;
	// See what suitable var -> type mappings are in the constraints.
	Map<ItemVar,Symbol> varToType = new HashMap<ItemVar,Symbol>();
	for (Iterator i = constraints.iterator(); i.hasNext();) {
	    PatternAssignment pv = (PatternAssignment)i.next();
	    List pat = pv.getPattern();
	    if (pat.size() == 2 && pat.get(0) == S_TYPE) {
		Object maybeVar = pat.get(1);
		if (!(maybeVar instanceof ItemVar))
		    continue;
		ItemVar var = (ItemVar)maybeVar;
		if (!(pv.getValue() instanceof Symbol))
		    continue;
		varToType.put(var, (Symbol)pv.getValue());
	    }
	}
	// Checl that all params correspond to a mapping from the constraints.
	Map<ItemVar,Symbol> wrong = new LinkedHashMap<ItemVar,Symbol>();
	for (Iterator i = params.iterator(); i.hasNext();) {
	    List pair = (List)i.next();
	    ItemVar var = Util.mustBe(ItemVar.class, pair.get(0));
	    Symbol type = Util.mustBe(Symbol.class, pair.get(1));
	    Symbol typeFromConstraints = varToType.get(var);
	    if (type != typeFromConstraints)
		wrong.put(var, typeFromConstraints);
	}
	if (!wrong.isEmpty())
	    throw new SyntaxException
		("The " + key + " annotation did not match the constraints." +
		 "  Mismatches: " + wrong);
    }

    /**
     * Returns a copy of this refinement in which each {@link ItemVar}
     * that has a value in the {@link MatchEnv} has been replaced by
     * that value.
     *
     * @throws MissingValuesException if it encounters any variable that
     *    is not given a value.
     */
    public Refinement instantiate(MatchEnv env) {
	final Set unbound = new TreeSet();
	Refinement r = instantiate(env, new Function1() {
	    public Object funcall(Object name) {
		ItemVar v = (ItemVar)name;
		unbound.add(v);
		return v;
	    }
	});
	if (unbound.isEmpty())
	    return r;
	else
	    throw new MissingValuesException(unbound);
    }

    /**
     * Returns a copy of this refinement in which each {@link ItemVar}
     * that has a value in the {@link MatchEnv} has been replaced by
     * that value.  The <i>ifUnbound</i> function is called on any
     * variable that does not have a value in the environment.
     */
    public Refinement instantiate(MatchEnv env, Function1 ifUnbound) {
	try {
	    Refinement r = (Refinement)clone();
	    Debug.expect(new StructuralEquality().equal(this, r));
	    // Pattern
	    r.pattern = (LList)env.instantiateTree(r.pattern, ifUnbound);
	    // Nodes
	    if (r.nodes != null) {
		r.nodes = new LinkedListOfNodeSpec(r.nodes);	// copy list
		for (ListIterator i = r.nodes.listIterator(); i.hasNext();) {
		    NodeSpec spec = (NodeSpec)i.next();
		    i.set(spec.instantiate(env, ifUnbound));
		}
	    }
	    // Issues
	    if (r.issues != null) {
		r.issues = new LinkedListOfIssue(r.issues);
		for (ListIterator i = r.issues.listIterator(); i.hasNext();) {
		    Issue issue = (Issue)i.next();
		    i.set(issue.instantiate(env, ifUnbound));
		}
	    }
	    // Constraints
	    if (r.constraints != null) {
		r.constraints = new LinkedListOfConstraint(r.constraints);
		for (ListIterator i = r.constraints.listIterator();
		     i.hasNext();) {
		    Constraint c = (Constraint)i.next();
		    i.set(c.instantiate(env, ifUnbound));
		}
	    }
	    // Annotations that are really constraints /\/
	    if (r.getAnnotations() != null) {
		r.setAnnotations((Annotations)r.getAnnotations().clone());
		Object inputs = r.getAnnotation(S_INPUT_OBJECTS);
		if (inputs != null)
		    r.setAnnotation(S_INPUT_OBJECTS,
				    env.instantiate(inputs, ifUnbound));
		Object outputs = r.getAnnotation(S_OUTPUT_OBJECTS);
		if (outputs != null)
		    r.setAnnotation(S_OUTPUT_OBJECTS,
				    env.instantiate(outputs, ifUnbound));
	    }
	    // Return the instantiated refinement.
	    return r;
	}
	catch (CloneNotSupportedException e) {
	    throw new RethrownException(e);
	}
    }

    public Object clone() throws CloneNotSupportedException {
	return super.clone();
    }

    public String toString() {
	return "Refinement[" + name + " expands " + pattern + "]";
    }

    /**
     * A comparator that puts refinements in alphabetical order by name.
     * <p>Note: this comparator imposes orderings that are inconsistent
     * with equals.</p>
     */
    public static class NameComparator implements Comparator {
	public NameComparator() { }
	public int compare(Object o1, Object o2) {
	    String name1 = ((Refinement)o1).getName();
	    String name2 = ((Refinement)o2).getName();
	    return name1.compareTo(name2);
	}
    }

}

// Issues:
// * Should we define an Instantiable interface for objects that
//   provide an instantiate(MatchEnv, Function1) method?
// * When a field F has declared type List, you can't call clone() without
//   casting it to a concrete class (and we don't want to depend on which
//   concrete class it is), and to make an LList of a List that might be
//   empty, you can't use a 1-argument constructor; so we use the idiom:
//      field = LList.newLList(field)
//   N.B. "newLList", no space.
