/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 22 18:25:18 2006 by Jeff Dalton
 * Copyright: (c) 2001, 2004, 2006, AIAI, University of Edinburgh
 */

package ix.ichat;

import java.awt.Container;
import java.awt.Dimension;
import java.awt.Color;
import java.awt.event.*;

import javax.swing.*;

import java.util.*;

import ix.icore.*;

import ix.itest.SendPanel;
import ix.itest.ReceivePanel;

import ix.ip2.WaitForReportsActivity;

import ix.iface.util.ToolFrame;
// import ix.iface.util.TranscriptTextArea; // /\/
import ix.iface.util.TranscriptTextPane;
import ix.iface.util.Reporting;
import ix.iface.util.ComboChoice;
import ix.iface.util.IconImage;
import ix.iface.util.CatchingActionListener;
import ix.iface.util.IFUtil;

import ix.util.*;
import ix.util.lisp.*;

/**
 * A frame for chat dialogues.
 */
public class ChatFrame extends ToolFrame implements ActionListener {

    IXAgent agent;

    protected Container contentPane = getContentPane();
    ReceivePanel receivePanel;
    SendPanel sendPanel;

    JMenu styleMenu;

    public ChatFrame(IXAgent agent, String title) {
	super(title);
	this.agent = agent;

	setIconImage(IconImage.getIconImage(this));

	receivePanel = new ReceivePanel(agent);
	sendPanel = new ChatSendPanel(agent);

	setJMenuBar(makeMenuBar());

//  	contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));
//  	contentPane.add(receivePanel);
//  	contentPane.add(sendPanel);

	JSplitPane split = new JSplitPane
	    (JSplitPane.VERTICAL_SPLIT, receivePanel, sendPanel);
	split.setResizeWeight(1.00);

	contentPane.add(split);

	pack();
    }

    /**
     * Creates the menu bar
     */
    protected JMenuBar makeMenuBar() {
	JMenuBar bar = new JMenuBar();

	JMenu fileMenu = new JMenu("File");
	bar.add(fileMenu);
	if (agent instanceof IChat)
	    // Standalone I-Chat
	    fileMenu.add(IFUtil.makeMenuItem("Exit", this));
	else
	    // Part of another agent
	    fileMenu.add(IFUtil.makeMenuItem("Close", this));

        // Make "Style" menu
	styleMenu = new JMenu("Style");
	bar.add(styleMenu);
	styleMenu.add(IFUtil.makeMenuItem("Message Color", this));

	// /\/: Maybe should have Edit -> Preferences

	return bar;
    }

    /**
     * Action interpreter
     */
    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("ChatFrame action:", command);
	if (command.equals("Close")) {
	    setVisible(false);
	}
	else if (command.equals("Exit")) {
	    if (Util.dialogConfirms(this,
				    "Are you sure you want to exit I-Chat?"))
		System.exit(0);
	}
	else if (command.equals("Message Color")) {
	    sendPanel.editMessageColor();
            styleMenu.setForeground
                (agent.getContactManager()
                          .getAgentColor(agent.getAgentSymbolName()));
	}
	else
	    Debug.noteln("Nothing to do for", command);
    }

    public void newMessage(ChatMessage message) {
	receivePanel.newMessage(message);
	// sendPanel.noteSender(message);
    }

    public void initReportBackReport(TaskItem item) {
	// Called to set up to report back about an item.
	Name ref = item.getRef();
	Name reportTo = item.getSenderId();
	Status status = item.getStatus();
	ReportType type =
	    (status == Status.BLANK || status == Status.EXECUTING
    	                            || status == Status.POSSIBLE
	     ? ReportType.PROGRESS
	     : ReportType.INFORMATION);
	sendPanel.initReport(item, reportTo, ref, type);
    }

    public void initForwardReport(TaskItem item) {
	Name ref = item.getId();
	Name reportTo = item.getForwardedTo();
	ReportType type = ReportType.INFORMATION;
	sendPanel.initReport(item, reportTo, ref, type);
    }

    public SendPanel getSendPanel() {
	return sendPanel;
    }

    /**
     * A panel for sending issues, activities, reports, and chat messages.
     */
    class ChatSendPanel extends SendPanel {

	ChatSendPanel(IXAgent agent) {
	    super(agent);
	}

	public void send(String destination, Sendable contents) {
	    // N.B. Can modify contents.
	    if (contents instanceof Issue || contents instanceof Activity) {
		TaskItem item = (TaskItem)contents;
		if (!destination.equalsIgnoreCase("me")
		      && item.getReportBack() == YesNo.YES
		      && item.getRef() == null) {
		    // Create an activity that will record any reports-back.
		    Activity activity =
			// /\/: might want a more convenient constructor
			new WaitForReportsActivity
			    (destination, item.getPattern());
		    activity.ensureId();
		    item.setRef(activity.getId());
		    activity.setPriority(item.getPriority());
		    agent.handleNewActivity(activity);
		}
	    }
	    super.send(destination, contents);
	}

	public void sending(String destination, Sendable contents) {
	    // Record outgoing chat-messages in the transcript.
	    if (contents instanceof ChatMessage)
		receivePanel.newMessage((ChatMessage)contents);
	}

    }

}
