/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Nov 24 16:54:44 2000 by Jeff Dalton
 * Copyright: (c) 2000, AIAI, University of Edinburgh
 */

package ix.examples;

import java.util.*;

import ix.util.*;
import ix.util.lisp.*;


/**
 * Simple resource constraints for use with PicoISim.
 */

public class SimResource {


    static class ISim extends PicoISim {}	// abbreviation

    static final Symbol
        S_RESOURCE = Symbol.intern("resource"),
	S_PRODUCE  = Symbol.intern("produce"),
	S_CONSUME  = Symbol.intern("consume");


    /**
     * Simple "produce" or "consume" resource constraints.
     */
    static class Constraint extends ISim.Constraint {

	Symbol subtype;
	Object pattern;
	long amount;

	Constraint(Symbol subtype, long amount, Object pattern) {
	    super(S_RESOURCE);			// type
	    this.subtype = subtype;
	    this.pattern = pattern;
	    this.amount = amount;
	    this.args = Lisp.list(new Long(amount), pattern);
	}

    }


    /**
     * A constraint manager for simple "produce" or "consume"
     * resource constraints.
     */
    static class CM extends ISim.ConstraintManager {

	Hashtable resourceTable = new Hashtable();

	CM() {
	    super(S_RESOURCE);			// type
	}

	Object addConstraint(ISim.Constraint c) {
	    Constraint rc = (Constraint)c;
	    if (rc.subtype == S_CONSUME) {
		long amount = getAmount(rc.pattern);
		if (amount >= rc.amount) {
		    setAmount(rc.pattern, amount - rc.amount);
		    noteSuccess(rc);
		    return Boolean.TRUE;
		}
		else {
		    noteFailure(rc);
		    return Boolean.FALSE;
		}
	    }
	    else {
		Debug.expect(rc.subtype == S_PRODUCE,
			     "invalid subtype in", rc);
		setAmount(rc.pattern, getAmount(rc.pattern) + rc.amount);
		noteSuccess(rc);
		return Boolean.TRUE;
	    }
	}

	Object tryConstraint(ISim.Constraint c) {
	    Constraint rc = (Constraint)c;
	    if (rc.subtype == S_CONSUME) {
		return new Boolean(getAmount(rc.pattern) >= rc.amount);
	    }
	    else {
		Debug.expect(rc.subtype == S_PRODUCE,
			     "invalid subtype in", rc);
		return Boolean.TRUE;
	    }
	}

	long getAmount(Object pattern) {
	    Long value = (Long)resourceTable.get(pattern);
	    return value == null ? 0 : value.longValue();
	}

	void setAmount(Object pattern, long amount) {
	    resourceTable.put(pattern, new Long(amount));
	}

	void noteSuccess(Constraint rc) {
	    Debug.noteln(rc.subtype + " " +
			 rc.amount + " " + rc.pattern);
	}

	void noteFailure(Constraint rc) {
	    Debug.noteln("Could not " + rc.subtype + " " +
			 rc.amount + " " + rc.pattern);
	}

	void printStatusReport() {
	    System.out.println("");
	    System.out.println("Resource state");
	    System.out.println("-------------");
	    for(Enumeration ke = resourceTable.keys();
		ke.hasMoreElements();) {
		Object key = ke.nextElement();
		System.out.println(key + " = " + resourceTable.get(key));
	    }
	    System.out.println("");
	}

    }

}
