/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Mar 19 20:24:13 2005 by Jeff Dalton
 * Copyright: (c) 2000, 2005, AIAI, University of Edinburgh
 */

package ix.examples;

import java.util.*;

import ix.util.*;
import ix.util.lisp.*;


/**
 * A simple example using an I-X framework.  This example shows how
 * to extend PicoISim to create a simulator that uses a simple 
 * resource CM to keep track of the current "state of the resource
 * world".
 */

public class PicoISimTest3 extends PicoISim {

    /**
     * Main program.
     */
    public static void main(String[] argv) {

	// Make a simulator with the desired activity handlers and
	// constraint managers.

	final IX_Simulator sim = new IX_Simulator();

	sim.setListener(new IX_SystemListener());

	sim.addActivityHandlers(new Object [] {
	    new ActivityHandler("reportResourceState") {
		void handleActivity(Activity i) {
		    // Find the resource CM and tell it to print
		    // a status report.
		    Hashtable CMTable = system.model.CMTable;
		    ((SimResource.CM)CMTable.get(SimResource.S_RESOURCE))
			.printStatusReport();
		}
	    }
	});

	sim.addConstraintManagers(new Object [] {
	    new SimResource.CM()
	});

	sim.addSimEventHandlers(new Object [] {
	    new ResourceEventHandler(SimResource.S_PRODUCE),
	    new ResourceEventHandler(SimResource.S_CONSUME)
	});

	// Schedule some initial events.

	SimResource.Constraint
	    c1 = new SimResource.Constraint(SimResource.S_CONSUME, 2, "items"),
	    p1 = new SimResource.Constraint(SimResource.S_PRODUCE, 4, "items");

	sim.schedule(new ResourceEvent(c1,  5)); //  5 tick delay between
	sim.schedule(new ResourceEvent(p1, 20)); // 20 tick delay between

	// Start the simulation.

	sim.start();

	// While the simulation is running, the user can ask for a
	// report of the resource levels by hitting return.

	new Thread() {
            public void run() {
		while (true) {
		    Util.readLine(System.in);
		    askForResourceReport(sim);
		    System.out.println("OK");
		}
	    }
	}.start();

    }

    static void askForResourceReport(IX_Simulator sim) {
	sim.newEvent(new Activity("reportResourceState"));
    }


    // The simulation contains events that produce or consume
    // resources and then reschedule themselves.

    static class ResourceEvent extends SimEvent {

	SimResource.Constraint rc;
	long timeBetween;

	ResourceEvent(SimResource.Constraint rc, long timeBetween) {
	    super(rc.subtype,	// verb - produce or comsume
		  rc.pattern,	// pattern - identifies the resource
		  0);		// run asap
	    this.rc = rc;
	    this.timeBetween = timeBetween;
	}

	void execute(IX_Simulator sim) {
	    // Each event sleeps for a bit just to make it easier
	    // for someone who's watching to see what happens.
	    try { Thread.sleep(2 * 1000); }	// sleep 2 seconds
	    catch (InterruptedException e) {}
	}

    }


    static class ResourceEventHandler extends SimEventHandler {

	ResourceEventHandler(Symbol rcSubtype) {
	    super(rcSubtype);	// sets this.verb
	}

	void handleActivity(Activity i) {
	    ResourceEvent re = (ResourceEvent)i;
	    SimResource.Constraint rc = re.rc;

	    // Add the constraint
	    Debug.expect(addConstraint(rc).equals(Boolean.TRUE),
			 "Could not add", rc);

	    // Let something happen
	    re.execute((IX_Simulator)system);

	    // Reschedule self
	    long next = getSimTime() + re.timeBetween;
	    re.est = next;
	    postSimEvent(re);
	}

	boolean isAble(SimEvent event) {
	    ResourceEvent re = (ResourceEvent)event;
	    SimResource.Constraint rc = re.rc;
	    return tryConstraint(rc).equals(Boolean.TRUE);
	}

    }

}
