/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Aug 23 18:07:42 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.applet;

import java.net.*;
import java.io.*;
import java.util.*;
import java.applet.*;
import javax.swing.SwingUtilities;

import ix.icore.IXAgent;
import ix.ip2.Ip2Applet;	// /\/
import ix.ip2.*;		// /\/
import ix.iface.util.Reporting;
import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.*;

/**
 * Lets I-P2 applets talk with each other.  It can also be used with
 * non-applet instances of {@link Ip2} if they specify a "base-url"
 * parameter.
 *
 * @see Parameters
 */
public class AppletCommunicationStrategy
       implements IPC.CommunicationStrategy {

    AppletCommunicationTool tool; // a user interface.

    Ip2 agent;
    URL docBase;

    IPC.MessageListener messageListener;

    boolean ableToSend = false;

    // One higher than the highest we've received so far.
    // Volatile because registerAs refers to it and is called
    // in the GUI event thread.  /\/
    volatile int expectedSeqNo = 0;

    // We need to know which messages we've received, but all
    // we really need to keep is a record of their sequence numbers --
    // the keys in the messageMap. /\/ For now, we are keeping the
    // values as well.
    MessageMemory messageMap = new MessageMemory();

    public AppletCommunicationStrategy() {
    }

    void setTool(AppletCommunicationTool tool) {
	// This method exists because the tool doesn't exist until
	// the user asks for it.
	this.tool = tool;
    }

    String getAgentName() {
	return agent.getAgentSymbolName();
    }

    synchronized boolean isAbleToSend() {
	return ableToSend;
    }
    synchronized void setIsAbleToSend(boolean v) {
	ableToSend = v;
    }

    synchronized URL getDocumentBase() {
	return docBase;
    }

    public synchronized void appletStart() {
	Debug.noteln(this + "appletStart() called");
    }

    public synchronized void appletStop() {
	Debug.noteln(this + "appletStop() called");
    }

    public synchronized void appletDestroy() {
	Debug.noteln(this + "appletDestroy() called");
    }

    public synchronized void setupServer(Object destination,
					 IPC.MessageListener listener) {

	Debug.noteln("Setting up AppletCommunicationStrategy server.");
	messageListener = listener;
	agent = (Ip2)IXAgent.getAgent();
	if (agent instanceof Ip2Applet.AppletIp2) {
	    Applet applet = ((Ip2Applet.AppletIp2)agent).getApplet();
	    docBase = applet.getDocumentBase();
	}
	else if (Parameters.haveParameter("base-url")) {
	    try {
		docBase = new URL(Parameters.getParameter("base-url"));
	    }
	    catch (MalformedURLException e) {
		Debug.displayException
		    ("Invalid base-url for applet communication strategy", e);
		return;
	    }
	}
	else {
	    Util.displayAndWait(null,
	      "No base-url was given for the applet communication strategy");
	    return;
	}

	agent.addTool(new AppletCommunicationTool.Controller(this));

    }

    void registerAs(String name) {
	// Called only from the communication tool.
	Debug.expect(SwingUtilities.isEventDispatchThread(), "not in Swing");
	Debug.expect(!isAbleToSend(), "Registering when already registered");
	Debug.noteln("Trying to register as", name);

	AppletMessage request =
	    new AppletMessage(getAgentName(), "register-as", name);

	// Let the message-server know the highest-numbered message
	// we've received (in case the server exited and was restarted).
	// N.B. If we haven't yet received anything, the value will be -1.
	request.setSeqNo(expectedSeqNo - 1);

	Object reply = requestObject(request);
	if (!reply.equals("ok"))
	    throw new IPC.IPCException(reply.toString());

	transcript("Registered as " + name);
	agent.setAgentSymbolName(name);
	setIsAbleToSend(true);
	new ReceiveThread().start();
    }

    void serverUnavailable() {
	Util.swingAndWait(new Runnable() {
	    public void run() {
		tool.needToReregister();
	    }
	});
    }

    class ReceiveThread extends Thread {

	String ourName = getAgentName();

	public void run() {
	    AppletMessage message = null;
	messageLoop:
	    while(true) {
		// Get a message
		try {
		    message = nextMessage(message);
		}
		catch (IPC.BrokenConnectionException b) {
		    transcript("Connection problem: " +
			       Debug.describeException(b));
		    Debug.displayException
			("Exception while waiting for input.  " +
			 "Probably means the message server has exited.  " +
			 "Another exception may follow.  " +
			 "This exception was",
			 b);
		    // No message to handle.
		    continue messageLoop;
		}
		catch (ConnectFailure f) {
		    setIsAbleToSend(false);
		    transcript("Connection problem: " +
			       Debug.describeException(f));
		    Debug.displayException(f);
		    serverUnavailable();
		    return;	// exit this thread
		}
		catch (Throwable t) {
		    transcript("Receive failure: " +
			       Debug.describeException(t));
		    Debug.displayException
			("Exception while waiting for input.  " +
			 "Probably a timeout that can be ignored.  " +
			 "The exception was",
			 t);
		    // No message to handle.
		    continue messageLoop;
		}
		// Handle the message
		try {
		    handleMessage(message);
		}
		catch (Throwable t) {
		    transcript(Debug.describeException(t));
		    Debug.displayException(t);
		}
	    }
	}

	AppletMessage nextMessage(AppletMessage previous) {
	    transcript("Waiting for next message ...");

	    AppletMessage request = 
		new AppletMessage(getAgentName(), "get-message");

	    // Let the message-server know we received the previous message.
	    if (previous != null)
		request.setSeqNo(previous.getSeqNo());

	    Object reply = requestObject(request);
	    if (!(reply instanceof AppletMessage))
		throw new ClassCastException
		    ("Received an instance of " + reply.getClass().getName() +
		     " instead of an AppletMessage.");

	    // The message should have the form
	    //   AppletMessage(fromName, "send-to", [ourName, message])
	    AppletMessage next = (AppletMessage)reply;
	    Debug.expect(next.getCommand().equals("send-to"));
	    Debug.expect(next.getArg(0).equals(ourName));
	    return next;
	}

	void handleMessage(AppletMessage message) {
	    // Perhaps we've already received this message, but
	    // the server didn't know we had and so sent it again.
	    // Or we may have missed a message while the server
	    // thought it had been sent successfully.
	    Integer mSequence = message.getSequenceNumber();
	    int mSeq = message.getSeqNo();
	    Debug.expect(mSeq == mSequence.intValue());
	    if (mSeq == expectedSeqNo) {
		Debug.expect(!messageMap.containsKey(mSequence));
		expectedSeqNo++;
	    }
	    else if (mSeq > expectedSeqNo) {
		Debug.noteln("Missing messages between " + expectedSeqNo +
			     " and " + mSeq);
		expectedSeqNo = mSeq + 1;
	    }
	    else {
		Debug.expect(mSeq < expectedSeqNo);
		if (messageMap.containsKey(mSequence)) {
		    // We've already received this message.
		    transcript("Received " + mSeq + " again");
		    // So ignore it.
		    return;
		}
		else {
		    // This message fills a gap in the previous messages.
		    Debug.noteln("Received gap message", mSeq);
		}
	    }
	    // We have a new message.
	    Object contents = message.getArg(1);
	    transcript("Received " + mSeq + ": " +
		       Reporting.description(contents));
	    messageMap.remember(message); 	//\/ won't GC message
	    messageListener.messageReceived
		(new IPC.BasicInputMessage(contents));
	}

    }

    protected void transcript(final String line) {
	Debug.noteln("Transcript:", line);
	Util.swingAndWait(new Runnable() {
	    public void run() {
		do_transcript(Reporting.dateString() + " " + line);
	    }
	});
    }

    protected void do_transcript(String line) {
	tool.transcript(line);
    }

    public synchronized void sendObject(Object destination, Object contents) {
	if (!isAbleToSend())
	    throw new UnsupportedOperationException
		("Cannot send to " + destination +
		 " until you have registered with the message server.");
	transcript("Sending to " + destination + ": " +
		   Reporting.description(contents));
	AppletMessage command =
	    new AppletMessage
	        (getAgentName(), "send-to", destination, contents);
	command.setSendDate(new Date());
	try {
	    Object reply = requestObject(command);
	    Debug.noteln("Received", reply);
	    if (!reply.equals("ok"))
		throw new IPC.IPCException
		    ("Unexpected reply from message-server: " + reply);
	}
	catch (RuntimeException e) {
	    Debug.noteln("sendObject rethrowing", Debug.describeException(e));
	    throw e;
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new RethrownException(e);
	}
    }

    /**
     * Sends a request to the message server and returns the server's reply.
     * A string reply that reports an exception is turned into an appropriate
     * exception and then thrown, rather than being returned.
     */
    protected Object requestObject(AppletMessage m) {
	String replyText = "no reply";
	try {
	    replyText = sendText(encodeForSend(m));
	}
	catch (IOException e) {
	    // Includes MalformedURLException
	    Debug.noteException(e);
	    throw new IPC.IPCException(e);
	}
	if (replyText.startsWith("<")) {
	    Object reply = decodeReply(replyText);
	    if (reply instanceof String) {
		String text = (String)reply;
		if (text.startsWith("Server exception: "))
		    throw interpretServerException(text);
	    }
	    return reply;
	}
	else if (replyText.startsWith("Exception: "))
	    throw interpretRelayException(replyText);
	else
	    throw new IPC.IPCException
		("Request failed because " + replyText);
    }

    RuntimeException interpretServerException(String text) {
	return new IPC.IPCException(text);
    }

    RuntimeException interpretRelayException(String text) {
	Debug.expect(text.startsWith("Exception: "));
	String e = Strings.afterFirst(": ", text);
	if (e.startsWith("BrokenConnectionException: ")) {
	    String reason = Strings.afterFirst(": ", e);
	    return new IPC.BrokenConnectionException(reason);
	}
	else if (e.startsWith
		 ("IPCException: Cannot connect to message-server")) {
	    // Continues as " caused by ConnectException: Connection Refused",
	    // at least on Linux/BSD.
	    String reason = Strings.afterFirst("caused by ", e);
	    return new ConnectFailure(reason);
	}
	return new IPC.IPCException
	    ("Request failed because " + text);
    }

    static class ConnectFailure extends IPC.IPCException {
	ConnectFailure(String message) {
	    super(message);
	}
    }

    protected String encodeForSend(AppletMessage m) {
	return XML.objectToXMLString(m);
    }

    protected Object decodeReply(String text) {
	return XML.objectFromXML(text);
    }

    protected String sendText(String text)
              throws MalformedURLException, IOException {
	Debug.noteln("Applet sending text", text);
	URL to = new URL(docBase, "message.cgi");
	URLConnection conn = to.openConnection();
	conn.setDoInput(true);
	conn.setDoOutput(true);
	conn.setUseCaches(false);
	conn.setRequestProperty("Content-Type",
				"text/plain; charset=utf-8");
	Writer send = new OutputStreamWriter(conn.getOutputStream(), "utf-8");
	send.write(text);
	send.write("\r\n");
	send.close();
	// Read the reply
	BufferedReader read = new BufferedReader(makeReader(conn));
	String line;
	List lines = new LinkedList();
	while ((line = read.readLine()) != null) {
	    lines.add(line);
	}
	read.close();
	String reply = Strings.joinLines(lines).trim();	// trim needed /\/
	Debug.noteln("Applet received text", reply);
	return reply;
    }

    protected InputStreamReader makeReader(URLConnection conn)
              throws IOException, UnsupportedEncodingException {
	InputStream stream = conn.getInputStream();
	String encoding = conn.getContentEncoding();
	if (encoding == null)
	    return new InputStreamReader(stream);
	else 
	    return new InputStreamReader(stream, encoding);
    }

}

