/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Sep 30 17:17:46 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util.rdf;

import java.util.*;

import com.hp.hpl.jena.rdf.model.*;

import ix.util.*;

/**
 * A vocabulary of resources and properties constructed as-needed.
 */
class Vocabulary {

    Model model;

    String base;		// base URI
    String hash;		// base URI + "#"

    Morphology morph = new Morphology();

    // When a Jena model creates a resource or property, it may
    // create a new object or return an existing one that has
    // the correct values.  We always want an existing one,
    // if one exists.
    Map resourceCache = new HashMap();
    Map propertyCache = new HashMap();

    Vocabulary(String baseURI) {
	this(baseURI, ModelFactory.createDefaultModel());
    }

    Vocabulary(String baseURI, Model model) {
	// The ability to specify the model is primarily for
	// ontology models, because then we typically want the
	// resources to be in the model we're building rather
	// than in a separate one that represets a vocabulary
	// we're using.  /\/: But, in fact, this isn't used.
	this.base = baseURI;
	this.hash = base + "#";
	this.model = model;
    }

    String getBaseURI() {
	return base;
    }

    String getHashURI() {
	return hash;
    }

    Resource resource(String name) {
	Resource r = (Resource)resourceCache.get(name);
	if (r == null) {
	    r = model.createResource(hash + name);
	    resourceCache.put(name, r);
	}
	// Debug.expect(contains(r));
	return r;
    }

    Property property(String name) {
	Property p = (Property)propertyCache.get(name);
	if (p == null) {
	    p = model.createProperty(hash, "has-" + name);
	    propertyCache.put(name, p);
	}
	// Debug.expect(contains(p));
	return p;
    }

    Property singular(String pluralName) {
	return property(morph.singular(pluralName));
    }

    boolean contains(Resource r) {
	return r.getNameSpace().equals(hash);
    }

    boolean contains(Property p) {
	return p.getNameSpace().equals(hash);
    }

    String name(Resource r) {
	return r.getLocalName();
    }

    String name(Property p) {
	String locName = p.getLocalName();
	Debug.expect(locName.startsWith("has-"));
	return Strings.afterFirst("has-", locName);
    }

    private String pluralName(Property singularProp) {
	throw new UnsupportedOperationException
	    ("Plurals are too hard");
    }

}

// Issues:
// * Since Property is a subinterface of Resource, we don't really
//   need separate methods that have a Property as a parameter when
//   we have the same method for a Resource.
