/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Jan 17 02:53:10 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util.rdf;

import java.lang.reflect.*;
import java.util.*;

import com.hp.hpl.jena.rdf.model.*;
import com.hp.hpl.jena.vocabulary.*;

import ix.util.*;
import ix.util.xml.*;
import ix.util.reflect.*;
import ix.util.lisp.*;

import ix.test.xml.Mapping;	// /\/
import ix.test.xml.MapEntry;	// /\/

/**
 * Converts a Jena RDF model to one or more objects.
 */
class ModelParser {

    Vocabulary ix = new Vocabulary(Rdf.baseURI);

    ClassSyntax classSyntax = Rdf.classSyntax();

    Map resourceToObjectMap = new HashMap();
    Set rootObjects = new HashSet();
    boolean processedModel = false;

    RdfTranslator translator;
    Model model;

    ModelParser(RdfTranslator translator, Model model) {
	this.translator = translator;
	this.model = model;
    }

    Object getRootObject() {
	processModel();
	if (rootObjects.size() > 1)
	    throw new RuntimeException("More than one root object");
	else if (rootObjects.size() < 1)
	    throw new RuntimeException("No root object");
	else
	    return rootObjects.toArray()[0];
    }

    void processModel() {
	if (processedModel)
	    return;
	processedModel = true;
	if (model.containsResource(RDF.nil))
	    resourceToObjectMap.put(RDF.nil, Lisp.NIL); // avoid repeated check
	for (ResIterator i = model.listSubjects(); i.hasNext();) {
	    Resource sentence_subj = i.nextResource();
	    processResource(sentence_subj);
	}
	// Find root objects
	// Start with resources that appear as sentence subjects
	// but not as sentence objects.
	Set candidates = new HashSet(resourceToObjectMap.keySet());
	for (NodeIterator i = model.listObjects(); i.hasNext();) {
	    RDFNode sentence_obj = i.nextNode();
	    candidates.remove(sentence_obj);
	}
	// Go from the root resources to the corresponding Java objects
//  	Collect.map(rootObjects, candidates, 
//  		    Collect.toFunction(resourceToObjectMap));
	for (Iterator i = candidates.iterator(); i.hasNext();) {
	    Resource rootResource = (Resource)i.next();
	    // /\/: Make a "real ref" to the resource.
	    rootObjects.add(resourceToObject(rootResource));
	}
	// Debug.noteln("Object Map", resourceToObjectMap);
	// Debug.noteln("roots", rootObjects);
    }

    Object processResource(Resource r) {
	Object obj = resourceToObjectMap.get(r);
	if (obj == null) {
	    obj = buildObject(r);
	    resourceToObjectMap.put(r, obj);
	}
	return obj;
    }

    Object resourceToObject(Resource r) { // objectFromResource ? /\/
	// Called for "real refs" to the resource
	Object obj = processResource(r);
//  	if (obj instanceof Cons) {
//  	    Resource type = rdfType(r);
//  	    if (type != null && ix.name(type).equals("map")) {
//  		// Convert to Map.
//  		// /\/: If the type isn't null, it shouldn't be
//  		// anything other than map.
//  		Map m = new HashMap();
//  		for (Iterator i = ((List)obj).iterator(); i.hasNext();) {
//  		    MapEntry e = (MapEntry)i.next();
//  		    m.put(e.getKey(), e.getValue());
//  		}
//  		resourceToObjectMap.put(r, m);
//  		return m;
//  	    }
//  	}
	return obj;
    }

    Resource rdfType(Resource r) {
	Statement s = r.getProperty(RDF.type);
	if (s == null)
	    return null;
	Resource type = (Resource)s.getObject();
	return type;
    }

    Object buildObject(Resource r) {
	// Get the RDF type.  If no statement gives the type,
	// the resource may be an RDFList.
	Resource type = rdfType(r);
	if (type == null)
	    return tryAsList(r);
	if (!ix.contains(type))
	    throw new IllegalArgumentException
		("Unknown type " + type + " for " + r);
	// Work out the corresponding Java class.
	// Null is a special case.
	String className = ix.name(type);
	if (className.equals("null")) // irritating special case /\/
	    return null;
	Class c = classSyntax.classForExternalName(className);
	if (c == null)
	    throw new XMLException("Cannot find class " + className);
	else
	    return buildObject(c, r);
    }

    Object buildObject(Class c, Resource r) {
	// Here the class is just the class that directly corresponds
	// to the RDF type of the resource.  This is unlike the XML
	// translator, where the class is "desired" class.  There,
	// the desired class might be a field-value class rather than
	// the class that corresponds directly to the element name.
	Object result = null;
	ClassDescr cd = classSyntax.getClassDescr(c);
	if (c == String.class)	// String is final
	    result = stringFromResource(c, r);
	else if (cd.isPrimitive())
	    result = primitiveObjectFromResource(c, r);
//  	else if (cd.isList())
//  	    result = listFromResource(c, r);
//  	else if (cd.isList() && cd.getEltType().theClass == MapEntry.class)
//  	    result = tryAsList(r); // questionable trick /\/
//  	else if (cd.isSet())
//  	    result = setFromResource(c, r);
//  	else if (cd.isMap())
//  	    result = mapFromResource(c, r);
	else if (c == Mapping.class)
	    result = ((Mapping)structFromResource(c, r)).toMap();
	else if (cd.isStruct())
	    result = structFromResource(c, r);
//  	else if (cd.isXML())
//  	    result = embeddedXMLFromResource(c, r);
	else
	    throw new ConsistencyException("Unknown object type " + cd);
	return result;
    }

    LList tryAsList(Resource r) {
	// The type of list nodes is supposedly implicit,
	// so we look for RDF.first instead of RDF.type.
	Statement s1 = r.getProperty(RDF.first);
	Debug.expect(s1 != null, "Invalid map - missing first -", r);
	Statement s2 = r.getProperty(RDF.rest);
	Debug.expect(s2 != null, "Invalid list - missing rest -", r);
	Resource first = (Resource)s1.getObject();
	Resource rest = (Resource)s2.getObject();
	return new Cons(resourceToObject(first),
			(LList)resourceToObject(rest));
    }

    String stringFromResource(Class c, Resource r) {
	Statement s = r.getProperty(ix.property("litvalue"));
	Debug.expect(s != null, "No value property", r);
	Literal lit = (Literal)s.getObject();
	// Check against expected XSD type.
	String expected = classSyntax.xmlSchemaDatatype(c);
	String found = Strings.afterFirst("#", lit.getDatatypeURI());
	Debug.expectEquals(expected, found);
	// Return the textual representation
	return lit.getLexicalForm();
    }

    Object primitiveObjectFromResource(Class c, Resource r) {
	ClassDescr cd = classSyntax.getClassDescr(c);
	if (cd.isEnumeration()) {
	    return primitiveObjectFromString
		       (c, r.getLocalName());
	}
	else {
	    return primitiveObjectFromString
		       (c, stringFromResource(c, r));
	}
    }

    Object primitiveObjectFromString(Class c, String s) {
	// Special cases
	Stringer stringer = classSyntax.getStringer(c);
	if (stringer != null)
	    return stringer.fromString(s);
	// General case
	try {
	    Method meth = c.getMethod("valueOf", new Class[]{String.class});
	    return meth.invoke(null, new Object[]{s});
	}
	catch (InvocationTargetException e) {
	    Debug.noteException(e);
	    throw new XMLException
		("Can't convert to " + c, e.getTargetException());
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException
		("Can't convert to " + c, e);
	}
    }

    Object mapFromRDFList(Class c, Resource r) {		// unused /\/
	// /\/: We leave map tails as lists and want to turn
	// only full maps into Maps.  We assume a map tail is
	// referred to only once, by the earlier part of the
	// map.
	Statement s1 = r.getProperty(RDF.first);
	Debug.expect(s1 != null, "Invalid map - missing first -", r);
	Statement s2 = r.getProperty(RDF.rest);
	Debug.expect(s2 != null, "Invalid map - missing rest -", r);
	Resource first = (Resource)s1.getObject();
	Resource rest = (Resource)s2.getObject();
	return new Cons(resourceToObject(first),
			// N.B. not a "real ref" to rest
			(LList)processResource(rest));
    }

    Object structFromResource(Class c, Resource r) {
	// One way to do this is to iterate over the properties
	// of the resource, finding the corresponding fields as
	// we go.  That's what the XML translator does.  Here,
	// that's a bit tricky, because list-valued fields can
	// correspond to more than one property, making it a pain
	// to build the lists, and because we'd have to try
	// property names first as-is, then as plurals, in case
	// they were for a list field.  So we iterate over the
	// fields instead.  That might be less efficient,
	// depending on how fast the Resource.listProperties(Property)
	// method is.  /\/
	Object obj = Util.makeInstance(c);
	ClassDescr cd = classSyntax.getClassDescr(c);
	for (Iterator fi = cd.getFieldDescrs().iterator(); fi.hasNext();) {
	    FieldDescr fd = (FieldDescr)fi.next();
	    Object fieldValue = fieldValue(obj, fd, r);
	    // Install the field value.
	    try {
		if (fieldValue != null)
		    setFieldValue(obj, fd, fieldValue);
	    }
	    catch (Exception e) {
		Debug.noteException(e);
		throw new XMLException
		    ("Cannot set field " + fd.getName(), e);
	    }
	}
	return obj;
    }

    void setFieldValue(Object obj, FieldDescr fd, Object fieldValue)
           throws IllegalAccessException,
                  InvocationTargetException {
	Class valueClass = fd.getType(); 	// desired value class
	Class valueObjClass = valueClass.isPrimitive()
	    ? XMLTranslator.wrapperClass(valueClass)
	    : valueClass;
	if (valueObjClass.isInstance(fieldValue)) {
	    fd.setValue(obj, fieldValue);
	    return;
	}
	// Uh oh, we're in trouble; something's come along,
	// and it's burst our bubble.
	ClassDescr valueDescr = fd.getTypeDescr();
	if (valueDescr.isInterface())
	    ; // Fall through to lose.
	else if (fieldValue instanceof Collection 
	        && valueDescr.isCollection()) {
	    Collection col = (Collection)Util.makeInstance(valueClass);
	    col.addAll((Collection)fieldValue);
	    fd.setValue(obj, col);
	    return;
	}
	else if (fieldValue instanceof Map
		     && valueDescr.isMap()) {
	    Map map = (Map)Util.makeInstance(valueClass);
	    map.putAll((Map)fieldValue);
	    fd.setValue(obj, map);
	    return;
	}
	// We lose
	throw new IllegalArgumentException
	    ("Field cannot accept a value of class " + fieldValue.getClass());
    }

    Object fieldValue(Object obj, FieldDescr fd, Resource r) {
	ClassDescr valueDescr = fd.getTypeDescr();
	String fieldName = fd.getExternalName();
	/*
	if (valueDescr.isPrimitive()
                && !(valueDescr.theClass == String.class)) {
	    return getLiteralValue(valueDescr, r, ix.property(fieldName));
	}
	else */ if (valueDescr.isTypedList()) {
	    Property fieldProp = ix.singular(fieldName);
	    return getListValue(valueDescr, r, fieldProp);
	}
	else {
	    return getResourceValue(valueDescr, r, ix.property(fieldName));
	}
    }

    /*
    Object getLiteralValue(ClassDescr valueDescr, Resource r, Property p) {
	Literal lit = (Literal)onePropertyValue(r, p);
	Class fieldClass = valueDescr.theClass;
	Class fieldObjClass = fieldClass.isPrimitive()
	    ? XMLTranslator.wrapperClass(fieldClass)
	    : fieldClass;
	return lit == null
	    ? null
	    : primitiveObjectFromString(fieldObjClass, lit.getString());
    }
    */

    Object getResourceValue(ClassDescr valueDescr, Resource r, Property p) {
	Resource res = (Resource)onlyPropertyValue(r, p);
	return res == null
	    ? null
	    : resourceToObject(res);
    }

    RDFNode onlyPropertyValue(Resource r, Property p) {
	StmtIterator si = r.listProperties(p);
	if (si.hasNext()) {
	    Statement s = si.nextStatement();
	    if (si.hasNext())
		throw new IllegalArgumentException
		    ("More than one value for resource " + r +
		     " property " + p);
	    else
		return s.getObject();
	}
	else
	    return null;
    }

    List getListValue(ClassDescr valueDescr, Resource r, Property p) {
	StmtIterator si = r.listProperties(p);
	if (!si.hasNext())
	    return null;
	Class c = valueDescr.theClass;
	Debug.expect(TypedList.class.isAssignableFrom(c));
	Class implClass =
	    ListOf.findImplementationClass(c, LinkedList.class);
	if (implClass == null)
	    throw new XMLException
		("Cannot find a linked implementation for " + c);
	TypedList list = (TypedList)Util.makeInstance(implClass);
	while (si.hasNext()) {
	    Statement s = si.nextStatement();
	    list.add(resourceToObject((Resource)s.getObject()));
	}
	return list;
    }

}
