/*
 * Created on Dec 27, 2003
 *
 */
// package org.mindswap.owls.io.impl;

package ix.util.owls;
import org.mindswap.owls.io.impl.*;

import java.io.Writer;
import java.net.URI;
import java.util.Iterator;

import org.mindswap.owl.OWLResource;
import org.mindswap.owl.Util;
import org.mindswap.owl.vocabulary.RDF;
import org.mindswap.owl.vocabulary.RDFS;
import org.mindswap.owls.OWLSFactory;
import org.mindswap.owls.grounding.AtomicGrounding;
import org.mindswap.owls.grounding.Grounding;
import org.mindswap.owls.grounding.MessageMap;
import org.mindswap.owls.grounding.MessageMapList;
import org.mindswap.owls.grounding.UPnPAtomicGrounding;
import org.mindswap.owls.grounding.WSDLAtomicGrounding;
import org.mindswap.owls.process.AtomicProcess;
import org.mindswap.owls.process.CompositeProcess;
import org.mindswap.owls.process.ControlConstruct;
import org.mindswap.owls.process.DataFlow;
import org.mindswap.owls.process.DataFlowElement;
import org.mindswap.owls.process.Input;
import org.mindswap.owls.process.Parameter;
import org.mindswap.owls.process.ParameterList;
import org.mindswap.owls.process.Process;
import org.mindswap.owls.process.ProcessComponent;
import org.mindswap.owls.process.ProcessComponentList;
import org.mindswap.owls.process.ProcessModel;
import org.mindswap.owls.process.Sequence;
import org.mindswap.owls.profile.Profile;
import org.mindswap.owls.service.Service;
import org.mindswap.owls.vocabulary.FLAServiceOnt;
import org.mindswap.owls.vocabulary.OWLS;
//import org.mindswap.owls.vocabulary.OWLS_0_9;
import org.mindswap.owls.vocabulary.OWLS_1_0;

import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFList;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.RDFWriter;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.ResourceFactory;
import com.hp.hpl.jena.vocabulary.RDFSyntax;

import com.hp.hpl.jena.vocabulary.OWL;


/**
 * @author Evren Sirin
 *
 */
public class ProcessWriter extends OWLSWriterImpl {
	public static boolean DEBUG = false;
	
	private Model model = null;
	private URI base = null;

	private OWLS_1_0 OWLS = OWLS_1_0.instance;
	
	/**
	 * 
	 */
	public ProcessWriter() {
		version = "1.0";
	}

	/* (non-Javadoc)
	 * @see org.mindswap.owls.io.OWLSWriter#write(org.mindswap.owls.service.Service, java.io.Writer)
	 */
	public void write(Service service, Writer out) {
		model = ModelFactory.createDefaultModel();
		model.setNsPrefix("service", OWLS.Service.URI);
		model.setNsPrefix("profile", OWLS.Profile.URI);
		model.setNsPrefix("process", OWLS.Process.URI);
		model.setNsPrefix("grounding", OWLS.Grounding.URI);
		
		base = service.getFileURI();
		
		RDFWriter writer = model.getWriter("RDF/XML-ABBREV");
		// Jena does not let to use relative URI's if we don't disable this check
		writer.setProperty("allowBadURIs", "true");
		// we don't want literals to be used as attributes
		writer.setProperty("blockRules", new Resource[] {RDFSyntax.propertyAttr});
		// show xml header <?xml version="..." ...?>
		writer.setProperty("showXmlDeclaration", "true");

		// /\/: Avoid having <process:AtomicProcess ../>
		// being duplicated outside the Sequence.
		writer.setProperty("prettyTypes",
	          new Resource[] {
		    OWL.Ontology,
		    makeResource(OWLS.Service.Service)
		    // makeResource(OWLS.Process.Sequence)
		 });
		
		writeService(service);
		
		writer.write(model, out, "");
	}

        // /\/: Convenience method.
        private Resource makeResource(URI uri) {
	    return ResourceFactory.createResource(uri.toString());
	}

	private Resource toRDF(URI u) {
		if(base != null) u = base.relativize(u);
		return Util.toResource(u);
	}
	
	private Resource toRDF(OWLResource r) {		
		return r.getJenaResource();//toRDF(r.getURI());
	}
	
	private RDFNode toRDF(String o) {
		return toRDF(o, false);
	}

	private RDFNode toRDF(String o, boolean wellFormed) {
		return model.createLiteral(o, wellFormed);
	}

	
	private void addStatement(Object s, Object p, Object o) {
		Resource subj = null;
		Property prop = null;
		RDFNode  obj  = null;
		
		if(s instanceof Resource)
			subj = (Resource) s;
		else if(s instanceof OWLResource)
			subj = toRDF((OWLResource) s);
		else if(s instanceof URI)
			subj = toRDF((URI) s);
		else
			throw new RuntimeException("Invalid subject " + s + " " + s.getClass());
		
		if(p instanceof Property)
			prop = (Property) p;
		else if(p instanceof URI)
			prop = Util.toProperty((URI) p);
		else
			throw new RuntimeException("Invalid property " + p + " " + p.getClass());
		
		if(o instanceof RDFNode)
			obj = (RDFNode) o;
		else if(o instanceof OWLResource)
			obj = toRDF((OWLResource) o);
		else if(o instanceof URI)
			obj = toRDF((URI) o);
		else if(o instanceof String)
			obj = toRDF((String) o);
		else
			throw new RuntimeException("Invalid object " + o + " " + o.getClass());
		
		model.add(subj, prop, obj);
	}
	
	private void writeService(Service service) {
	        addStatement(service, RDF.type, OWLS.Service.Service);
		if (service.getProfile() != null)
		    addStatement(service, OWLS.Service.presents, service.getProfile());
		if (service.getProcessModel() != null)
		    addStatement(service, OWLS.Service.describedBy, service.getProcessModel());
		if (service.getGrounding() != null)
		    addStatement(service, OWLS.Service.supports, service.getGrounding());
		
		if (service.getProfile() != null) writeProfile(service.getProfile());
		if (service.getProcessModel() != null) writeProcessModel(service.getProcessModel());
		if (service.getGrounding() != null) writeGrounding(service.getGrounding());
	}

	private void writeProfile(Profile profile) {
		URI profileType = profile.getType();
		
		for(int i = 0; i < OWLSFactory.supportedVersions.length; i++) {
			String version = OWLSFactory.supportedVersions[i];
			OWLS vocabulary = OWLSFactory.getVocabulary(version);
			if(profileType.equals(vocabulary.getProfile().Profile)) {
				profileType = OWLS.Profile.Profile;
				break;
			}
		}
			
		addStatement(profile, RDF.type, profileType);
		addStatement(profile, OWLS.Service.presentedBy, profile.getService());
		
		if(profile.getLabel() != null) {
			addStatement(profile, OWLS.Profile.serviceName, profile.getLabel());
			addStatement(profile, RDFS.label, profile.getLabel());
		}
		if(profile.getTextDescription() != null)
			addStatement(profile, OWLS.Profile.textDescription, profile.getTextDescription());
		
		writeProfileParams(profile, profile.getInputs());
		writeProfileParams(profile, profile.getOutputs());
	}

	private void writeProfileParams(Profile profile, ParameterList params) {
		for(int i = 0; i < params.size(); i++) {
			Parameter param = params.parameterAt(i);

			if(param instanceof Input)
				addStatement(profile, OWLS.Profile.hasInput, param);			
			else
				addStatement(profile, OWLS.Profile.hasOutput, param);
		}
	}

	private void writeProcessModel(ProcessModel processModel) {
		addStatement(processModel, RDF.type, OWLS.Process.ProcessModel);
		// /\/: Why did it make the model describe AtomicProcess???
		// addStatement(processModel, OWLS.Service.describes, OWLS.Process.AtomicProcess);
		addStatement(processModel, OWLS.Service.describes, processModel.getService());
		addStatement(processModel, OWLS.Process.hasProcess, processModel.getProcess());
		
		writeProcess(processModel.getProcess());
	}
	
	private void writeProcess(Process process) {
		if(process instanceof AtomicProcess) 
			writeAtomicProcess((AtomicProcess) process);
		else if(process instanceof CompositeProcess) 
			writeCompositeProcess((CompositeProcess) process);

		writeProcessParams(process, process.getInputs());
		writeProcessParams(process, process.getOutputs());
		
//		TODO
		writeDefaultValues(process);
		writeDataFlow(process);			
	}
	
	private void writeAtomicProcess(AtomicProcess process) {			
		addStatement(process, RDF.type, OWLS.Process.AtomicProcess);
		
		writeProcessParams(process, process.getInputs());
		writeProcessParams(process, process.getOutputs());
	}
	
	private void writeProcessParams(Process process, ParameterList params) {
		for(int i = 0; i < params.size(); i++) {
			Parameter param = params.parameterAt(i);

			if(param instanceof Input) {
				addStatement(process, OWLS.Process.hasInput, param);
				addStatement(param, RDF.type, OWLS.Process.Input);
			}
			else {
				addStatement(process, OWLS.Process.hasOutput, param);
				addStatement(param, RDF.type, OWLS.Process.Output);
			}
			
			addStatement(param, OWLS.Process.parameterType, param.getType());
			
			if(param.getLabel() != null)
				addStatement(param, RDFS.label, param.getLabel());			
		}
	}

	private void writeCompositeProcess(CompositeProcess process) {			
		addStatement(process, RDF.type, OWLS.Process.CompositeProcess);	
		
		RDFNode cc = writeControlConstruct(process.getComposedOf());
		
		addStatement(process, OWLS.Process.composedOf, cc);		
	}

	private RDFNode writeProcessComponent(ProcessComponent c) {
		if(c instanceof Process) {
			writeProcess((Process) c);
			return c.getJenaResource();
		}
		else 
			return writeControlConstruct((ControlConstruct) c);
	}	
		
	private RDFNode writeControlConstruct(ControlConstruct construct) {			
		RDFNode cc = null;
		if(construct instanceof Sequence)
			cc = writeSequence((Sequence) construct);
	
		return cc;
	}
	
	private RDFNode writeSequence(Sequence sequence) {
		RDFNode components = writeComponents(sequence.getComponents());
		
		addStatement(sequence, RDF.type, OWLS.Process.Sequence);
		addStatement(sequence, OWLS.Process.components, components);
		
		return sequence.getJenaResource();
	}
	
	private RDFNode writeComponents(ProcessComponentList list) {
		RDFNode[] elems = new RDFNode[list.size()];
		for(int i = 0; i < list.size(); i++) {
			ProcessComponent c = list.processComponentAt(i);
			
			elems[i] = writeProcessComponent(c);
		}
		
		return model.createList(elems);
	}

        // /\/: Fix bug in data-flow processing: it was using "theProperty" instead of "theParameter".
        private Property OWLS_Process_theParameter =
	    ResourceFactory.createProperty("http://www.daml.org/services/owl-s/1.0/Process.owl#theParameter");
        // /\/: and "atClass" instead of "atProcess".
        private Property OWLS_Process_atProcess =
	    ResourceFactory.createProperty("http://www.daml.org/services/owl-s/1.0/Process.owl#atProcess");

	private void writeDataFlow(Process process) {
		DataFlow df = process.getDataFlow();
		for(int i = 0; i < df.size(); i++) {
			DataFlowElement dfe = df.dfeAt(i);
						
			RDFNode[] elems = new RDFNode[dfe.size()];
			for(int j = 0; j < dfe.size(); j++) {
				Parameter p = dfe.parameterAt(j);
				elems[j] = ResourceFactory.createResource();
				addStatement(elems[j], RDF.type, OWLS.Process.ValueOf);
				// addStatement(elems[j], OWLS.Process.theProperty, p);
				addStatement(elems[j], OWLS_Process_theParameter, p);
				if(p.getProcess() != null)
				    // addStatement(elems[j], OWLS.Process.atClass, p.getProcess());
				    addStatement(elems[j], OWLS_Process_atProcess, p.getProcess());
			}			
			addStatement(process, OWLS.Process.sameValues, model.createList(elems));
		}
		
	}

	private void writeDefaultValues(Process process) {
//		ValueMap values = process.getDefaultValues();
//		Iterator i = values.entrySet().iterator();
//		while(i.hasNext()) {
//			Map.Entry entry = (Map.Entry) i.next();
//			URI p = (URI) entry.getKey();
//			String value = entry.getValue().toString();
//
//			Resource restr = ResourceFactory.createResource();
//			addStatement(restr, RDF.type, OWL.Restriction);
//			addStatement(restr, OWL.onProperty, p);
//			addStatement(restr, OWL.hasValue, value);
//			
//			addStatement(process, RDFS.subClassOf, restr);			
//		}
	}
	
	private void writeGrounding(Grounding grounding) {
		boolean isWSDL = false;
		boolean isUPnP = false;
		
		addStatement(grounding, OWLS.Service.supportedBy, grounding.getService());
		
		Iterator i = grounding.getAtomicGroundings().iterator();
		while(i.hasNext()) {
			Object n = i.next();
			AtomicGrounding ag = (AtomicGrounding) n;
			isWSDL = isWSDL || (ag instanceof WSDLAtomicGrounding);
			isUPnP = isUPnP && (ag instanceof UPnPAtomicGrounding);
			
			if(ag instanceof WSDLAtomicGrounding) {
				addStatement(grounding, OWLS.Grounding.hasAtomicProcessGrounding, ag);
				writeWSDLGrounding((WSDLAtomicGrounding) ag);
			}
			else if(ag instanceof UPnPAtomicGrounding) {
				addStatement(grounding, OWLS.Grounding.hasAtomicProcessGrounding, ag);
				writeUPnPGrounding((UPnPAtomicGrounding) ag);
			}			
		}
		
		addStatement(grounding, OWLS.Service.supportedBy, grounding.getService());
		if(isWSDL)
			addStatement(grounding, RDF.type, OWLS.Grounding.WsdlGrounding);
		else if(isUPnP)
			addStatement(grounding, RDF.type, FLAServiceOnt.UPnPGrounding);
	}

	private void writeWSDLGrounding(WSDLAtomicGrounding grounding) {
		addStatement(grounding, RDF.type, OWLS.Grounding.WsdlAtomicProcessGrounding);
		addStatement(grounding, OWLS.Grounding.wsdlDocument, grounding.getWSDL());
		addStatement(grounding, OWLS.Grounding.owlsProcess, grounding.getProcess());
		
		
		try {
			addStatement(grounding, OWLS.Grounding.wsdlInputMessage, grounding.getInputMessage());
			addStatement(grounding, OWLS.Grounding.wsdlOutputMessage, grounding.getOutputMessage());
		} catch (Exception e) {
		}
		
		Resource opDesc = ResourceFactory.createResource();
		addStatement(grounding, OWLS.Grounding.wsdlOperation, opDesc);
		addStatement(opDesc, RDF.type, OWLS.Grounding.WsdlOperationRef);
		if(grounding.getPortType() != null)
			addStatement(opDesc, OWLS.Grounding.portType, grounding.getPortType());
		addStatement(opDesc, OWLS.Grounding.operation, grounding.getOperation());
		
		writeWSDLGroundingParams(grounding, true);
		writeWSDLGroundingParams(grounding, false);
	}

	private void writeWSDLGroundingParams(WSDLAtomicGrounding grounding, boolean writeInputs) {
		MessageMapList messageMapList;
		if(writeInputs)
			messageMapList = grounding.getInputMap();
		else
			messageMapList = grounding.getOutputMap();

		if(messageMapList.size() == 0)
			return;
		
		RDFNode[] elements = new RDFNode[messageMapList.size()];
		for(int i = 0; i < messageMapList.size(); i++) {
			MessageMap map = messageMapList.messageMapAt(i);

			Resource r = ResourceFactory.createResource();
			addStatement(r, RDF.type, OWLS.Grounding.wsdlMessageMap);
			addStatement(r, OWLS.Grounding.owlsParameter, map.getOWLSParameter());
			addStatement(r, OWLS.Grounding.wsdlMessagePart, map.getGroundingParameter());
			if(map.getTransformation() != null)
				addStatement(r, OWLS.Grounding.xsltTransformation, toRDF(map.getTransformation(), true));
			elements[i] = r;
		}	

		RDFList list = model.createList(elements);
		if(writeInputs)
			addStatement(grounding, OWLS.Grounding.wsdlInputMessageParts, list);
		else 
			addStatement(grounding, OWLS.Grounding.wsdlOutputMessageParts, list);			
		
	}
	
	private void writeUPnPGrounding(UPnPAtomicGrounding grounding) {
		addStatement(grounding, RDF.type, FLAServiceOnt.UPnPAtomicProcessGrounding);
		addStatement(grounding, OWLS.Grounding.owlsProcess, grounding.getProcess());
		
		addStatement(grounding, FLAServiceOnt.upnpCommand, grounding.getUPnPAction());
		addStatement(grounding, FLAServiceOnt.upnpDeviceURL, grounding.getUPnPDescription());
		addStatement(grounding, FLAServiceOnt.upnpServiceID, grounding.getUPnPService());
				
		writeUPnPGroundingParams(grounding, true);
		writeUPnPGroundingParams(grounding, false);	
	}

	private void writeUPnPGroundingParams(UPnPAtomicGrounding grounding, boolean writeInputs) {
		MessageMapList messageMapList;
		if(writeInputs)
			messageMapList = grounding.getInputMap();
		else
			messageMapList = grounding.getOutputMap();

		if(messageMapList.size() == 0)
			return;
				
		RDFNode[] elements = new RDFNode[messageMapList.size()];
		for(int i = 0; i < messageMapList.size(); i++) {
			MessageMap map = messageMapList.messageMapAt(i);

			Resource r = ResourceFactory.createResource();
			addStatement(r, RDF.type, FLAServiceOnt.UPnPMap);
			addStatement(r, OWLS.Grounding.owlsParameter, map.getOWLSParameter());
			addStatement(r, FLAServiceOnt.upnpParameter, map.getGroundingParameter());
			if(map.getTransformation() != null)
				addStatement(r, OWLS.Grounding.xsltTransformation, toRDF(map.getTransformation(), true));
			elements[i] = r;
		}	

		RDFList list = model.createList(elements);
		if(writeInputs)
			addStatement(grounding, FLAServiceOnt.UPnPInputMapping, list);
		else 
			addStatement(grounding, FLAServiceOnt.UPnPOutputMapping, list);			
		
	}
	
}
