
/****************************************************************************
 * A table model for showing and editing (world-state) constraints on object
 * properties in refinements.
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Thu Feb  1 14:41:50 2007
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview.table;

import javax.swing.*;       
import javax.swing.event.*;       
import javax.swing.table.*;       
import java.util.*;
import java.awt.Dimension;
import java.awt.event.FocusListener;
import java.awt.event.FocusEvent;
/*
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
*/
import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.icore.domain.*;
import ix.iview.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iview.tree.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.table.*;
import ix.iface.domain.*;

/****************************************************************************
 * A table model for showing and editing (world-state) constraints on object
 * properties in refinements.
 *
 *****************************************************************************
 */

public class PropertyTableModel extends AbstractIXTableModel
					implements FocusListener
{
  public static Symbol REL_DEFAULT = Refinement.S_EFFECT;

  //int mode = ALL;
  String[] columns = {"Property","Value","Value Range","At Node","Relation"};
  Class[] classes = {ObjectProperty.class, String.class, 
		     ObjectProperty.Syntax.class, Symbol.class, Symbol.class};
  
  private HashMap propertyConstraints = new HashMap();
  JComboBox propertyBox;
  UIObjectClass objectClass; //given and noted - cannot be changed from here
  ItemVar variable;
  int mode;


  public PropertyTableModel() {
    super(false);
    setColumnNames(columns);
    setColumnClasses(classes);
  }

  public JComboBox setupPropertyBox() {
    propertyBox = new JComboBox();
    propertyBox.addItem(" ");
    propertyBox.setRenderer(new NamedRenderer(false));
    propertyBox.setEditable(true);
    return propertyBox;
  }

  /**
   * Adjust the given table (which should be the table whose model
   * this is). Adjusts the column sizes of the table and sets good
   * renderers.
   */
  public void adjustTable(IXTable table) {
    table.setPreferredScrollableViewportSize(new Dimension(500, 120));
    table.setWidth(Boolean.class, new Integer(35));

    //renderers for NODE column and RELATION colummn
    TableColumn column = table.getColumnModel().getColumn(this.NODE_COL);
    column.setCellRenderer(new ix.iview.util.NodeNameRenderer(false));

    column = table.getColumnModel().getColumn(this.EFF_COL);
    JComboBox condEffBox = new JComboBox();
    condEffBox.addItem(Refinement.S_CONDITION);
    condEffBox.addItem(Refinement.S_EFFECT);
    column.setCellEditor(new DefaultCellEditor(condEffBox));
    condEffBox.setSelectedItem(this.REL_DEFAULT);

    column = table.getColumnModel().getColumn(this.PROP_COL);
    column.setPreferredWidth(100);
    JComboBox propertyBox = this.setupPropertyBox();
    column.setCellRenderer(new NamedRenderer(false));
    DefaultCellEditor propCellEditor = new DefaultCellEditor(propertyBox);
    column.setCellEditor(propCellEditor);
  }

  public void clear() {
    variable = null;
    setObjectClass(null);
  }

  public void setConstraints(ItemVar var, UIObjectClass oClass, 
			     HashMap propConstrMap) {
    //Debug.noteln("PropTM: setting constraints for var", var);
    //Debug.noteln(" map", propConstrMap);
   
    if ((var != null) && !var.equals(variable)) { //new var
      noteOldConstraints();
      variable = var;
    }

    //if ((oClass != null) && !oClass.equals(objectClass)) { //new object class
    if (oClass != null) {
      objectClass = oClass;
      updatePropertyBox(); //get new properties into the box
    }

    if (propConstrMap != null) 
      //new constraints; throw away old ones and put these in
      propertyConstraints = new HashMap(propConstrMap);
    else propertyConstraints = new HashMap();

    refreshProperties(); //filter for current properties and set table data
  }
    
  public void setObjectClass(UIObjectClass oClass) {
    objectClass = oClass;
    updatePropertyBox();
    refreshProperties();  //filter for current properties and set table data
  }


  public void setConstraints(HashMap propConstraints) {
    if (propConstraints != null) 
      //new constraints; throw away old ones and put these in
      propertyConstraints = new HashMap(propConstraints);
    else propertyConstraints = new HashMap();
    refreshProperties();    
  }
  public void setMode(int propMode) {
    mode = propMode;
    refreshProperties();
  }

  private void clearPropertyBox() {
    propertyBox.removeAllItems();
  }

  //remove table data, then fill from hashMap according to object class
  //and display-mode. set table data
  protected List refreshProperties() {
    List constraints = getPropertyConstraints();
    setData(constraints.toArray());
    //updatePropertyBox();
    return constraints;
  }
  public void updatePropertyBox() { //fill propertyBox with class props
    //Debug.noteln("PropTM: Updating property box");
    propertyBox.removeAllItems();
    if (objectClass != null) {
      Collection objectProperties = objectClass.getAllProperties();
      //Debug.noteln(" properties are", UIUtil.show(objectProperties));
      if (objectProperties != null) {
	//update the properties that can be selected in the table
	for (Iterator i = objectProperties.iterator(); i.hasNext(); ) 
	  propertyBox.addItem(i.next());
      }
    }
  }

  private HashMap filterPropertyConstraints(Map oldMap, Collection properties){
    //Debug.noteln("OCE: filtering cs", UIUtil.show(oldMap.entrySet()));
    //Debug.noteln(" for properties", UIUtil.show(properties));
    HashMap newConstrs = new HashMap();
    HashMap oldConstrs = new HashMap(oldMap);
    if (properties != null) {
      //find constraints for current properties, keep them (not old!)
      for (Iterator i = properties.iterator(); i.hasNext(); ) {
	Object prop = i.next();
	Object cSet = oldMap.get((ObjectProperty)prop);
	//Debug.noteln("OCE: constraints for prop", prop);
	//Debug.noteln(" in map", UIUtil.show(oldMap.entrySet()));
	//Debug.noteln(" are", cSet);
	if (cSet != null) {
	  newConstrs.put(prop, cSet);
	  oldConstrs.remove(prop); //still in use, so not old
	}
      }
    }
    //oldConstrs now only contain props that no longer fit.
    //***warn about losing these old ones?
    return newConstrs;
  }
  
  public int PROP_COL = 0;
  public int VALUE_COL = 1;
  public int SYNTAX_COL = 2;
  public int NODE_COL = 3;
  public int EFF_COL = 4;
  public Object getCellValueAt(Object baseObject, int columnIndex) {
    Constraint constraint = (Constraint)baseObject;
    if (IVUtil.isEmptyConstraint(constraint))	return null;
    if (columnIndex == PROP_COL) {
      return IVUtil.propertyFromPattern(constraint.getPattern(), objectClass);
    }
    else if (columnIndex == SYNTAX_COL) { 
      ObjectProperty property = 
	IVUtil.propertyFromPattern(constraint.getPattern(), objectClass);
      if (property == null) return "";
      return property.getSyntax();
    }
    else if (columnIndex == VALUE_COL) {
      return constraint.getValue();
    }
    else if (columnIndex == NODE_COL) {
      return Symbol.intern("self");
    }
    else if (columnIndex == EFF_COL) {
      if (constraint.getRelation() == null) 
	constraint.setRelation(REL_DEFAULT); //default
      return constraint.getRelation();
    }
    else return "";
  }
  
  public boolean isCellEditable(int row, int col) {
    if ((col == VALUE_COL) || (col == EFF_COL) || (col == NODE_COL)) 
      return true; 
    if ((row == getRowCount()-1) 
	&& ((col == PROP_COL) || (col == SYNTAX_COL))) return true;
    else {
      //Debug.noteln("OCE: cannot edit row - col", row + " " + col);
      return false;
    }
  }
  
  public void setValueAt(Object val, int row, int col) {
    //Debug.noteln("OCE: settingValue " + val + " of class", val.getClass());
    //Debug.noteln(" OCE: setting value for object", dataObject);
    if (col == SYNTAX_COL) return;
    
    Object dataObject = getRowObject(row);
    try {
      //clone the constraint so updates are fired in UIR
      Constraint c = (Constraint)((Constraint)dataObject).clone();
      
      if (col == EFF_COL) {
	if (IVUtil.isEmptyConstraint(c)) {
	  complainEmpty();
	  return;
	}
	else {
	  //Debug.noteln("PropTabMod: setting relation");
	  c.setRelation((Symbol)val);
	  //Debug.noteln(" constraint now", IVUtil.printConstraint(c));
	  //refreshProperties();
	  //return;
	}
	
      }
      else if (col == VALUE_COL) {
	if (IVUtil.isEmptyConstraint(c)) {//last row, ask for prop first
	  complainEmpty();
	  return; //do not update ------------->
	}
	else {
	  IVUtil.setConstraintValue(objectClass, c, val);
	}
      }
      else if (col == PROP_COL) { //same for old and new; clears value
	//no property given, nothing done
	if ((val == null) || val.equals("") || val.equals(" ")) return; 
	
	if (variable == null) {
	  String[] message = {"Please specify a variable name first",
			      "(Nothing done)"};
	  JOptionPane.showMessageDialog(null, message);
	  return; //do not update ------------->
	}
	else {//new constraint
	  if (c.getRelation() == null) c.setRelation(REL_DEFAULT);
	  if (c.getType() == null) c.setType(Refinement.S_WORLD_STATE);
	  ObjectProperty prop;
	  //get the property
	  if (val instanceof ObjectProperty)
	    prop = ((ObjectProperty)val);
	  else {
	    //Debug.noteln("OCE: prop selection of class", val.getClass());
	    List patt = new ArrayList();
	    patt.add(val);
	    prop = IVUtil.propertyFromPattern(patt, objectClass);
	    //Debug.noteln(" prop is", prop);
	  }

	  if (prop == null) prop = defineNewProperty(val);
	  //make "empty" constraint with property pattern, null value
	  if (prop != null) {
	    LList pat = Lisp.list(prop.getName(), variable);
	    PatternAssignment pa = new PatternAssignment(pat, "");
	    c.setParameters(Lisp.list(pa));
	    if (c == null) {
	      String bits = prop.getName().toString() + " " + 
		variable.toString();
	      throw new SyntaxException("Invalid constraint: " 
					+ Strings.quote(bits));
	      //throw - do not update ------------->
	    }
	    else {
	      //Debug.noteln("PropTM: new constraint parameters are", 
		//UIUtil.show(c.getParameters()));
	      //Debug.noteln(" variable is", variable);
	      addConstraint(prop, c);
	      refreshProperties(); 
	      return; //already updated ------------>
	    }	    
	  }
	  else //editing null property - nothing to do
	    return; //do not update ------------->
	}
      }
      replaceConstraint((Constraint)dataObject, c);
    }
    catch (Exception e) { Debug.noteException(e); }
    refreshProperties(); //filter for current properties and set table data
  }
  
  private List getPropertyConstraints() {
    List constraints = new LinkedList();
    if (objectClass != null) { //collect previous constraints 
      Collection properties = objectClass.getAllProperties(); //all props
      //get constraints
      HashMap propConsts = 
	filterPropertyConstraints(propertyConstraints, properties);
      //Debug.noteln("OCE: filtered cs", UIUtil.show(propConsts.entrySet()));
      if (propConsts != null) {
	LinkedList values = new LinkedList(propConsts.values());
	if (values != null) { //values is a list of lists; flatten it
	  for (Iterator i = values.iterator(); i.hasNext(); ) {
	    Collection vals = (Collection)i.next();
	    constraints.addAll(vals);
	  }
	}
      }
    }
    //add empty line at end to put more constraints
    constraints.add(new Constraint());
    return constraints;
  }
  private ObjectProperty defineNewProperty(Object val) {
    if (objectClass == null) {

      JOptionPane.showMessageDialog(null, 
				    "The current object class is not valid");
      return null;
    }
    List syns = ObjectProperty.Syntax.values();
    String[] message = {"Please set the value range for the new property",
			"Cancel if you did not intend to define",
			"a new property for type" + objectClass};
    Object answer = 
	JOptionPane.showInputDialog(null, message, 
				    "Value Range Input", 
				    JOptionPane.INFORMATION_MESSAGE,
				    null, syns.toArray(),
				    ObjectProperty.Syntax.DEFAULT);
    if (answer == null) return null; //cancelled ------------>
	    
    ObjectProperty prop = new ObjectProperty();
    prop.setName(Symbol.intern(val.toString()));
    prop.setSyntax(ObjectProperty.Syntax.valueOf(answer.toString()));
    objectClass.addProperty(prop);
    updatePropertyBox(); //reflect change in prop list
    //make sure the property we use is exactly the one in the OC
    prop = IVUtil.propertyFromString(val.toString(), objectClass);
    return prop;
  }
    
  private void complainEmpty() {
      String[] message = {"Please use column " + (PROP_COL+1), 
			  "to enter a property to constrain first.",
			  "(Nothing done)"};
      JOptionPane.showMessageDialog(null, message);
  }

  private void addConstraint(ObjectProperty prop, Constraint c) {
    propertyConstraints = 
	noteProperty(propertyConstraints, objectClass, prop, c); 
    //Debug.noteln("PropTM: prop constraints now", propertyConstraints);
  }

  private void replaceConstraintData(Constraint oldC, Constraint newC) {
    replaceData(oldC, newC);
  }
    
  private void replaceConstraint(Constraint oldC, Constraint newC) {
    //Debug.noteln("PTM: Replacing constraint", oldC);
    if ((oldC != null) && oldC.equals(newC)) return; //-----> no change

    //Debug.noteln("  with", newC);    
    ObjectProperty prop = 
	IVUtil.propertyFromPattern(oldC.getPattern(), objectClass);
    Object oldCs = propertyConstraints.get(prop);
    if ((oldCs != null) && (oldCs instanceof Set)) {
      Set oldConstraints = (Set)oldCs;  
      //Debug.noteln("  in", UIUtil.show(oldConstraints));    
      /*
      if (!oldConstraints.contains(oldC)) 
	//Debug.noteln("Old set does not contain old constraint", oldC);
      
      if (!oldConstraints.remove(oldC))
	Debug.noteln(" did not find old constraint in", 
		     UIUtil.show(oldConstraints));
      */
      Set newConstraints = IVUtil.removeConstraint(oldConstraints, oldC);
      newConstraints.add(newC);
      //oldConstraints.add(newC);
      propertyConstraints.put(prop, newConstraints);
      //noteConstraint(oldConstraints, prop, newC);
      //replaceData(oldC, newC);
    }
  }


  
  /**
   * Note the given constraint in the given propMap under the given property.
   * the given propMap should be a property-constraintSet map.
   * Handles overwrites of old constraints.
   * @return the given HashMap with the constraint added.
   */
  public static HashMap noteProperty(HashMap propMap, UIObjectClass oClass, 
				     ObjectProperty prop, Constraint c) {
    Object oldCs = propMap.get(prop); //property already constrained?
    Set propSet;
    if (oldCs == null) { //no constraint on property yet, so set it up
      propSet = new HashSet();
      propSet.add(c);
      propMap.put(prop, propSet);
      return propMap;
    }
    else {
      propSet = (Set)oldCs;
      return noteConstraint(propMap, oClass, propSet, prop, c);
    }
  }

  private static HashMap noteConstraint(HashMap propMap, UIObjectClass oClass, 
					Set cSet, ObjectProperty prop, 
					Constraint c) {
    //check whether the consraint needs to replace a previous one.
    if ((cSet != null) && !cSet.isEmpty()) {
      Set inSet = new HashSet(cSet); //go through a copy so we can remove olds
      //remove any constraints that should be replaced by the new one
      for (Iterator i = inSet.iterator(); i.hasNext(); ) {
	Constraint oldC = (Constraint)i.next();
	if (IVUtil.sameObjectConstraint(c, oldC)) {
	  //Debug.noteln("OCE: replacing constraint");
	  String oldParams = prop.getName() + " = " + oldC.getValue();
	  String newParams = prop.getName() + " = " + c.getValue();
	  if (oldParams.equals(newParams)) {
	      //replacing old with same, so forget about new.
            return propMap;
	  }
	  Symbol otherEff = Refinement.S_CONDITION;
	  String an = "a";
	  if (c.getRelation() == Refinement.S_CONDITION)
	    otherEff = Refinement.S_EFFECT;
	  else an = "an";
	  String[] options = {"overwrite old constraint (" + oldParams + ")", 
			      "change new constraint (" + newParams + ") to " 
			      + otherEff,
			      "change old constraint (" + oldParams + ") to " 
			      + otherEff,
			      "forget new constraint (" + newParams 
			      + ") i.e. Cancel"};
	  String[] message = {"There already is " + an + " " + c.getRelation()
			      + " for property " + prop.getName() + ":",
			      prop.getName() + " = " + oldC.getValue(),
			      "Do you want to..."};
	  Object o = 
	    JOptionPane.showInputDialog(null, message, "Options",
					JOptionPane.INFORMATION_MESSAGE, 
					null, options, options[0]);
	  if ((o == null) || o.equals(options[3])) return propMap;
	  else if (o.equals(options[0])) { //edit old constraint, forget new
	    //cSet.remove(oldC);
	    //replaceConstraintData(oldC, c);
	    IVUtil.setConstraintValue(oClass, oldC, c.getValue());
	    //propMap.put(prop, cSet); //already changed 
	    return propMap;
	  }
	  else if (o.equals(options[1])) { //change new, try again
	    c.setRelation(otherEff);
	    return noteConstraint(propMap, oClass, cSet, prop, c);
	  }
	  else if (o.equals(options[2])) { //add new, change old, try adding that
	    cSet.remove(oldC);
	    cSet.add(c);
	    propMap.put(prop, cSet);
	    //replaceConstraintData(oldC, c);
	    oldC.setRelation(otherEff);
	    return noteProperty(propMap, oClass, prop, oldC);    
	  }
	}
      }
      //if it gets here, there is no replacing to do
    }
    else cSet = new HashSet(); //null or empty = make sure not null
    cSet.add(c); //no replacing to do, just add it
    propMap.put(prop, cSet);
    return propMap;
  }

  public void noteOldConstraints() {
    if (variable != null)
      fireConstraintsChanged(variable, objectClass, propertyConstraints);

  }

  public void focusGained(FocusEvent e) {
  }
  public void focusLost(FocusEvent e) {
    noteOldConstraints();
  }

  HashSet propConstrsListeners = new HashSet();
  public void addPropertyConstraintsListener(PropertyConstraintsListener p) {
    propConstrsListeners.add(p);
  }
  private void fireConstraintsChanged(ItemVar v, UIObjectClass oClass, 
					     HashMap propConstraints) {
    //Debug.noteln("PTM: firing constraints change for var", v);
    if (propConstrsListeners != null) {
      for (Iterator i = propConstrsListeners.iterator(); i.hasNext(); ) {
	PropertyConstraintsListener p = (PropertyConstraintsListener)i.next();
	p.variableConstraintsChanged(v, propConstraints);
      }
    }
  
  }

}
