/****************************************************************************
 * Mirrors the Domain for safe editing and adds UI things and change management
 *
 * @author Jussi Stader
 * @version 4.1
 * Updated: Mon Mar 26 13:09:32 2007
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview.domain;

import java.util.*;
import java.lang.reflect.Field;
import java.awt.Component;
import java.awt.Cursor;
import javax.swing.JOptionPane;
import javax.swing.JFileChooser;
import java.io.*;

import ix.iview.*;
import ix.iview.util.*;
import ix.iview.event.*;
import ix.iview.domain.event.*;
import ix.iview.tree.*;
import ix.icore.domain.event.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.domain.*;
import ix.iface.ui.AbstractSetEditor;
import ix.iface.ui.util.*;
import ix.iface.ui.tree.IXTrees;
import ix.iface.ui.event.*;
import ix.util.*;
import ix.util.match.*;
import ix.util.lisp.*;

/**
 * Mirrors the Domain for safe editing and adds UI things and change
 * management.  A UIDomain maintains the link to a base domain. It can
 * publish changes in the base domain. UIDomain also contains a link
 * to the DomainEditor which is the one that keeps track of which
 * uiDomain is current.
 */

public class UIDomain extends AbstractAnnotatedObject
  implements UIObject, Named, DomainListener, DomainObjectManager
{

  public String name = "Undefined";
  public String comments = ""; //NOTE: always update these after annotations
  public Annotations annotations;
  protected String baseReference = ""; /** name of the original domain */


  /** the public domain - always kept as published (has external listeners) */
  private Domain publicDomain;
  /** 
   * A transient copy of the published domain, used for file work (saving and 
   * loading without publishing) and for checking consistency before saving
   * and publishing. 
   */
  private Domain draftDomain = new Domain();
  private File domainFile;

  /** state toggles used to adjust user interactions */
  private Domain saving = null;
  private boolean loading = false;


  private DomainEditor editor;
 
  protected UIGrammar grammar;

  //the set editors that look after editing refinements and objectClasses
  private UIRefinementSetEditor refSetEditor;
  private UIObjectClassSetEditor classSetEditor;
  private HashMap properties = new HashMap();


  /** list of graphs that apply to Refinements; should be UIRefinement slot */
  public HashMap graphs = new HashMap();

  //--------------------Constructors and Setup---------------------------------

  public UIDomain(Domain baseDomain, DomainEditor editor) {
    super();
    this.editor = editor;
    setCurrentDomain(baseDomain);
    
    grammar = new UIGrammar(this);
  }

  public UIDomain(DomainEditor editor) {
    this(new Domain(), editor);
  }

  public UIDomain(String name, DomainEditor editor) {
    this(new Domain(), editor);
    this.name = name;
  }

  //Note: fires nothing
  public void setCurrentDomain(Domain domain) {
    clearQuiet();
    //Debug.noteln("UID: setting current domain");
    publicDomain = domain;
    draftDomain = cloneDomain(publicDomain, draftDomain);

    //Debug.noteln("UID: all domain refinements", domain.getRefinements());
    refSetEditor = 
      new UIRefinementSetEditor((DomainObjectManager)this, 
				(LinkedList)domain.getRefinements());
    //Debug.noteln(" setEd refinements:",refSetEditor.getAllConstructs());
    collectProperties(domain);
    classSetEditor = 
      new UIObjectClassSetEditor((DomainObjectManager)this, 
				 (LinkedList)domain.getObjectClasses());

    setBaseReference(publicDomain.getName());
    publicDomain.addDomainListener(this);
    loadFromDomain(); //fill in own fields
  }
  
  public void noteSetEditor(AbstractSetEditor setEd) {
    if (setEd instanceof UIRefinementSetEditor) 
      refSetEditor = (UIRefinementSetEditor)setEd;
    else if (setEd instanceof UIObjectClassSetEditor) 
      classSetEditor = (UIObjectClassSetEditor)setEd;
    else Debug.noteln("Cannot note set editor of class", setEd.getClass());
  }


    public UIGrammar getGrammar() {return grammar;}    
    public void setGrammar(UIGrammar grammar) {this.grammar = grammar;}    
	
  //--------------------------Relays to UISetEditors--------------------------

  public void ensureConstruct(UIObject object) {
    if (object instanceof UIRefinement) 
      refSetEditor.ensureConstruct(object);
    else if (object instanceof UIObjectClass) 
      classSetEditor.ensureConstruct(object);
    else Debug.noteln("Cannot ensure construct of class", object.getClass());
  }
  public boolean addConstruct(UIObject object) {
    //Debug.noteln("UID: adding construct", object);
    if (object instanceof UIRefinement) 
      return refSetEditor.addConstruct(object);
    else if (object instanceof UIObjectClass) 
      return classSetEditor.addConstruct(object);
    else {
      Debug.noteln("Cannot add construct of class", object.getClass());
      return false;
    }
  }
  public void updateConstruct(UIObject object) {
    if (this.equals(object)) //noting the domain changes only
      saveToDomain();
    else if (object instanceof UIRefinement) 
      refSetEditor.updateConstruct(object);
    else if (object instanceof UIObjectClass) 
      classSetEditor.updateConstruct(object);
    else Debug.noteln("Cannot update construct of class", object.getClass());
  }

  public void removeConstruct(Named object) {
    if (object instanceof UIRefinement) 
      refSetEditor.removeConstruct(object);
    else if (object instanceof UIObjectClass) 
      classSetEditor.removeConstruct(object);
    else Debug.noteln("Cannot remove construct of class", object.getClass());
  }
  public void changeName(Object object, String newName) {
    //Debug.noteln("UID: name change to: " +newName+ ".");

    if ((newName == null) || (newName.trim().equals(""))) { 
      //Debug.noteln("  empty name");//covered by handleOverwrite
      if (object instanceof UIObjectClass)
	((UIObjectClass)object).setLegalName(newName); 
      else if (object instanceof UIRefinement)
	((UIRefinement)object).setLegalName(newName); 
    }
    else if (object instanceof UIRefinement) {
      //Debug.noteln(" for refinement", object);
      Object nr = getNamedRefinement(newName);
      //Debug.noteln(" got clasher", nr);
      if ((nr != null) && !nr.equals(object) && 
	  !nr.equals(((UIObject)object).getBaseObject())) //clash
	refSetEditor.handleNameClash((UIObject)object,newName,nr);
      else ((UIRefinement)object).setLegalName(newName); 
    }
    else if (object instanceof UIObjectClass) {
      //Debug.noteln(" for UIOC", object);
      Object uio = getNamedObjectClass(newName);
      if ((uio != null) && !uio.equals(object)) //clash
	classSetEditor.handleNameClash((UIObject)object,newName,(UIObject)uio);
      else ((UIObjectClass)object).setLegalName(newName); 
      //Debug.noteln(" done for", object);
    }
    else Debug.noteln("Cannot change name for construct of class", 
		      object.getClass());
  }

  //----------------------------UIObject stuff---------------------------------

  public Domain getDomain() {
    return publicDomain;
  }
  public UIDomain getUIDomain() {
    return this;
  }

  public boolean hasOriginal() {
    return (publicDomain != null);
  }
  public IXObject getOriginal() {
    return publicDomain;
  }
  
  public IXObject getBaseObject() {
    return draftDomain;
  }
  public IXObject makeBaseObject() {
    return new Domain();
  }
  public void setBaseObject(IXObject object) {
    publicDomain = (Domain)object;
  }
  public Object getBaseReference() {
    return baseReference;
  }
  public void setBaseReference(Object o) {
    baseReference = (String)o;
  }
  public Class getBaseClass() {
    return Domain.class;
  }

  /**
   * A refinement is undefined if all its data fields are empty except
   * the pattern which may have data in it.  
   */
  public boolean isUndefined() {
    if ((name == null || name.equals("") || 
	 name.equals("Undefined") || name.equals("undefined"))
	&& (refSetEditor == null || refSetEditor.isEmpty())
	&& (classSetEditor == null || classSetEditor.isEmpty())
	&& (getComments() == null || (getComments().equals("")))
	&& (getAnnotations() == null || (getAnnotations().isEmpty())
	    || UIUtil.isEmptyThing(getAnnotations().values()))) {
      //Debug.noteln("UIDomain is empty");
      return true;
    }
    //Debug.noteln("refinement is not empty");
    return false;
  }
  public boolean isEmpty() {
    if (refSetEditor.isEmpty() && refSetEditor.isEmpty() && 
	((publicDomain == null) || (publicDomain.isEmpty())))
      return true;
    else 
      return false;
  }

  public String getName() {
    return name;
  }
  public void setName(String name) {
    this.name = name;
  }
  public String getComments() {
    return comments;
  }
  public void setComments(String comments) {
    this.comments = comments;
  }
  public Annotations getAnnotations() {
    return annotations;
  }
  public void setAnnotations(Annotations annotations) {
    this.annotations = annotations;
  }

  /**
   * Collects the differences between this object and the base one (draft).
   * @return a list of strings describing changes
   */
  public List collectChanges() {
    return collectChanges((Domain)getBaseObject(), "draft");
  }

  /**
   * Collects the differences between this object and the published one.
   * @return a list of strings describing changes
   */
  public List collectChangesFromOriginal() {
    return collectChanges(publicDomain, "published");
  }

  public List collectChanges(Domain domain) {
    return collectChanges(domain, "old");
  }
  public List collectChanges(Domain domain, String which) {
    List changes = new ArrayList();
    if (isUndefined()
	//((name == "") || (name == null)) && (annotations == null) 
	&& ((domain == null) || domain.isEmpty())) 
      //was and is empty - no change
      return changes;
    if (domain == null) {
	//Debug.noteln("UID has changed: was empty, not now");
	changes.add("The " + which + "domain is empty, this is panel not:");
	changes.add("  " + print());
	return changes;
    }      
    if (nameHasChanged(domain.getName()))
	changes.add("Name has changed from " + domain.getName() +
	    " to " + getName());
    if (!IVUtil.sameString(domain.getComments(), getComments()))
	changes.add("Comments have changed from " + domain.getComments() +
	    " to " + getComments());
    if (!IVUtil.sameMap(getAnnotations(),domain.getAnnotations()))
	changes.add("Annotations have changed from " + domain.getAnnotations()
		    + " to " + getAnnotations());
    return changes;
  }

  /** 
   * Finds out whether the domain has changed from the given one.
   * The domain has not changed if the domain's own fields (name) have not
   * changed.
   */
  public boolean hasChanged(Domain domain) {
    if (isUndefined() 
	//((name == "") || (name == null)) && (annotations == null) 
	&& ((domain == null) || domain.isEmpty())) 
      //was and is empty
      return false;
    else if (domain == null) {
      Debug.noteln("UID has changed: was empty, not now");
      return true;
    }      
    else if (!nameHasChanged(domain.getName())) {
      if (IVUtil.sameMap(annotations,domain.getAnnotations())) {
	if (IVUtil.sameString(domain.getComments(),getComments())) 
	  return false;
	else 
	  Debug.noteln("UID has changed comments from "+domain.getComments()+
		       " to", getComments());
      }
      else 
	Debug.noteln("UID has changed annotations from " 
		     +domain.getAnnotations()+" to", getAnnotations());
    }
    else 
      Debug.noteln("UID has changed name from "+domain.getName()+" to", name);
    return true;
  }
  /** 
   * Finds out whether the domain has changed from the draft.
   */
  public boolean hasChanged() {
    return hasChanged(draftDomain);
  }
  /** 
   * Finds out whether the domain has changed from the published/original one.
   */
  public boolean hasChangedFromOriginal() {
    return hasChanged(publicDomain);
  }


  public boolean nameHasChanged(Object nameThing) {
    return !IVUtil.sameString(name, (String) nameThing);
    /*
    String original = (String) nameThing;
    if ((((name == null) || name.equals("")) &&
	 ((original == null) || (original.equals("")))) ||
	((name != null) && name.equals(original)))
      return false;
    else return true;
    */
  }

  public boolean setValue(String field, Object value) {
    if (field == null) return false;
    try {
      if (field.equals("name")) setName((String)value);
      else if (field.equals("comments")) setComments((String)value);
      else if (field.equals("annotations")) setAnnotations((Annotations)value);
      else {
	Debug.noteln("UIDomain cannot set value for field", field);
	if (value != null) Debug.noteln(" value class is", value.getClass());
	return false;
      }
      return true;
    }
    catch (Exception e) {
      Debug.describeException(e);
      return false;
    }
  }
  public boolean sameValue(String field, Object value, Object otherValue) {
    if (field == null) return false;
    if (value == null) return UIUtil.isEmptyThing(otherValue);
    if (otherValue == null) return false;
    if (field.equals("name") || field.equals("comments"))
      return value.equals(otherValue);
    if (field.equals("annotations")) 
      return IVUtil.sameMap((Map)value, (Map)otherValue);
    return value.equals(otherValue);
  }
  

  /** Save the object's own info into its base object */
  public void saveToDomain() {
    if (publicDomain == null) publicDomain = (Domain)makeBaseObject();
    if (draftDomain == null) {
      draftDomain = cloneDomain((Domain)publicDomain, draftDomain);
    }
    saveToDomain(draftDomain);
  }
  public void saveToDomain(Object domainObject) {
    Domain domain = (Domain)domainObject;
    domain.setName(name);
    if (annotations == null) domain.setAnnotations(null);
    else domain.setAnnotations(new Annotations(annotations));
    domain.setComments(comments);
  }

  /** load the object's own information (name) from the original domain */
  public void loadFromDomain() {
    loadFromDomain(draftDomain);
  }
  public void loadFromDomain(Object domainObject) {
    if (domainObject == null) {
      clearThis();
    }
    else {
      name = ((Domain)domainObject).getName();
      annotations = ((Domain)domainObject).getAnnotations();
      //use a copy, not a pointer
      if (annotations != null) annotations = new Annotations(annotations); 
      //Debug.noteln("UID: got annotations", annotations);
      comments = ((Domain)domainObject).getComments();
    }
    fireDomainEdited();
  }
  public void loadFromOriginal() { 
    loadFromDomain(publicDomain);
  }

  public void insertFromDomain(Object domainObject) {
    if (domainObject != null) {
      if (name == null || name.equals("")) 
	name = ((Domain)domainObject).getName();
      else name = name+"+"+((Domain)domainObject).getName();
      //use a copy, not a pointer
      Annotations newAnnotations = ((Domain)domainObject).getAnnotations();
      if (newAnnotations != null)
	takeAnnotations(((Domain)domainObject).getAnnotations()); 
      //Debug.noteln("UID: got annotations", annotations);
      if (comments == null || comments.equals("")) 
	comments = ((Domain)domainObject).getComments();
      else comments = comments+" + "+((Domain)domainObject).getComments();
    }
    fireDomainEdited();
  }


  //these do not make sense for the domain intself.
  public void addToDomain(Domain domain) {}
  public void updateInDomain(Domain domain) {}
  public void removeFromDomain(Domain domain) {}

  public void moveToUIDomain(UIDomain uiDomain) {}
  public void noteChange() {}


  //-------------------------Public Services----------------------------

  /** 
   * Finds out whether the domain and its constructs have changed from the
   * draft.
   */
  public boolean hasChangedAll() {
    return hasChangedAll(draftDomain);
  }
  /** 
   * Finds out whether the domain and its constructs have changed from the
   * original.
   */
  public boolean hasChangedAllFromOriginal() {
    return hasChangedAll(publicDomain);
  }

  /** 
   * Finds out whether the domain and its constructs have changed from the
   * original.
   */
  public boolean hasChangedConstructsFromOriginal() {
    return (refSetEditor.hasChangedConstructs() ||   // refinements changed
	    classSetEditor.hasChangedConstructs()) ; // classes changed
  }


  private void setSaving(Domain domain) {
    saving = domain;
  }
   private void unsetSaving() {
    saving = null;
  }
  public boolean isSaving(Domain domain) {
    return domain.equals(saving);
  }
  public boolean isSaving() {
    return (saving != null);
  }



  /** 
   * Finds out whether anything in the domain (inc. constructs) have changed
   * from the given one.
   * The domain has not changed if there are no added/edited/removed notes
   * and the domain's own fields have not changed.
   */
  public boolean hasChangedAll(Domain domain) {
    return (hasChanged(domain) ||  //domain itself changed
	    refSetEditor.hasChangedConstructs() ||   // refinements changed
	    classSetEditor.hasChangedConstructs()) ; // classes changed
  }

  /**
   * Collects the differences between this object and the published one.
   * @return a list of strings describing changes
   */
  public List collectAllChanges() {
    return collectAllChanges(publicDomain);
  }
  public List collectAllChanges(Domain domain) {
    List changes = new ArrayList();
    if (hasChanged(domain)) {
      changes.add("Changes in the domain's own specifications:");
      changes.addAll(collectChanges(domain));
    }
    List more = refSetEditor.collectConstructChanges();
    if ((more != null) && !more.isEmpty()) {
      if (changes.size() > 0) changes.add(UIUtil.lineSeparator);
      changes.addAll(more);
    }
    more = classSetEditor.collectConstructChanges();
    if ((more != null) && !more.isEmpty()) {
      if (changes.size() > 0) changes.add(UIUtil.lineSeparator);
      changes.addAll(more);
    }
    return changes;
  }


  public UIObject newUIObject(IXObject object) {
    if (object == null) return null;
    if (object instanceof Domain) {
      UIObject uiObject = new UIDomain((Domain)object, editor);
      addedObject(uiObject, object);
      return uiObject;
    }
    else if (object instanceof Refinement)
      return (UIObject)refSetEditor.newOwnObject(object);
    else if (object instanceof ObjectClass)
      return (UIObject)classSetEditor.newOwnObject(object);
    else return null;
  }

  //--------------------------Getting information

  public List getAllRefinements() {
    return refSetEditor.getAllConstructs();
  }
  public List getAllObjectClasses() {
    //Debug.noteln("UID: all ocs:", classSetEditor.getAllConstructs());
    return classSetEditor.getAllConstructs();
  }


  public boolean hasObjectSystem() {
    List os = getAllObjectClasses();
    if (os == null) return false;
    if (os.isEmpty()) return false;
    if (os.size() == 1) return false;
    return true;
  }
  public boolean hasRefinements() {
    List os = getAllRefinements();
    if (os == null) return false;
    if (os.isEmpty()) return false;
    if (os.size() == 1) return false;
    return true;
  }


  /**
   * Find all object classes in the domain
   */
  public List getObjectClassTree() {
    List allClasses = getAllObjectClasses();
    LinkedList classes = new LinkedList();
    if (allClasses == null) return classes;

    List roots = findRoots(allClasses);
    if (roots == null) return allClasses;

    //for each root, go depth first
    for (Iterator i = roots.iterator(); i.hasNext(); ) {
      Object next = i.next();
      UIObjectClass root = null;
      if (next == null) {}
      else if (next instanceof UIObjectClass)
	root = (UIObjectClass)next;
      else if (next instanceof ObjectClass)
	root = (UIObjectClass)getUIObject((ObjectClass)next);
      else 
	Debug.noteln("UIDomain: root class of "+next+" not suitable", 
		     next.getClass());
      getObjectClassTree(root, classes); //updates classes
    }
    return classes;
  }

  private void getObjectClassTree(UIObjectClass root, LinkedList sofar) {
    //collect the tree starting at root, exclude sofar
    if ((sofar == null) || !sofar.contains(root)) {
      List children = root.getChildren();
      sofar.add(root);
      for (Iterator i = children.iterator(); i.hasNext(); ) {
	try {
	  UIObjectClass child = (UIObjectClass)i.next();
	  getObjectClassTree(child, sofar); //updates classes
	}
	catch (Exception e) {Debug.noteException(e);}
      }
    }
  }

  private List findRoots(List classes) {
    if ((classes == null) || classes.isEmpty()) return null;
    List roots = new ArrayList();
    for (Iterator i = classes.iterator(); i.hasNext(); ) {
      try {
	UIObjectClass c = (UIObjectClass)i.next();
	if (c != null) {  //should not happen, but does
	  List parents = c.getParents();
	  if ((parents == null) || parents.isEmpty()) roots.add(c);
	}
      }
      catch (Exception e) {Debug.noteException(e);}
    }
    return roots;
  }


  public UIObjectClass getObjectClassRoot() {
    List roots = findRoots(getAllObjectClasses());
    if ((roots == null) || roots.isEmpty()) return null;
    else if (roots.size() == 1) 
      return (UIObjectClass)roots.get(0);
    else if (roots.size() > 1) {
      UIObjectClass newRoot = new UIObjectClass(this, IXTrees.ROOTNAME);
      for (Iterator i = roots.iterator(); i.hasNext(); ) {
	Object orphan = i.next();
	String child = IVUtil.getRelativeName(orphan);
	if (child != null) {
	  newRoot.addChild(child);
	  UIObject uiOrphan = findUIObject(orphan);
	  if (uiOrphan != null) ((UIObjectClass)uiOrphan).addParent(newRoot);
	  else Debug.noteln("UID: cannot find UIOC for", child);
	}
      }
      return newRoot;
    }
    return null;
  }

  public List getObjectClassTreeOld() {
    LinkedList classes = new LinkedList();
    //Debug.noteln("UID: getting object class tree from domain", print());
    ObjectClassTreeNode dummyRoot = 
      new ObjectClassTreeNode(new UIObjectClass(this, "Dummy Root"));
    //Debug.noteln(" made dummy root");
    ObjectClassTreeModel classTreeModel = 
      new ObjectClassTreeModel(this, dummyRoot);
    classTreeModel.reloadData();
    Object oRoot = classTreeModel.getRoot();
    if ((oRoot != null) && (oRoot instanceof ObjectClassTreeNode)) {
      ObjectClassTreeNode root = (ObjectClassTreeNode)oRoot;
      //Debug.noteln("UID: getting tree of object classes");
      for (Enumeration e = root.depthFirstEnumeration(); e.hasMoreElements();) {
	Object node = e.nextElement(); //tree node
	//list wants root first
	classes.add(0,((ObjectClassTreeNode)node).getUserObject()); 
      }
    }
    //Debug.noteln("UID:got object classes:", UIUtil.show(classes));
    return classes;
  }

  public UIObjectClass getObjectClassRootOld() {
    LinkedList classes = new LinkedList();
    //Debug.noteln("UID: getting object class tree from domain", print());
    ObjectClassTreeNode dummyRoot = 
      new ObjectClassTreeNode(new UIObjectClass(this, "Dummy Root"));
    //Debug.noteln(" made dummy root");
    ObjectClassTreeModel classTreeModel = 
      new ObjectClassTreeModel(this, dummyRoot);
    classTreeModel.reloadData();
    Object oRoot = classTreeModel.getRoot(); 
    return rootToUIOC(oRoot);

  }
  private UIObjectClass rootToUIOC(Object oRoot) {
    if (oRoot == null) return null;
    if (oRoot instanceof UIObjectClass) return (UIObjectClass)oRoot;
    if (oRoot instanceof ObjectClassTreeNode) 
      return rootToUIOC(((ObjectClassTreeNode)oRoot).getUserObject());
    else {
      Debug.noteln("UID: found root of odd class", oRoot.getClass());
      return null;
    }
  }
  /**
   * Finds a refinement whose name matches the given string.
   * @return a UIRefinement, a Refinement, or null - depending on which exists.
   */
  public Object getNamedRefinement(String name) {
    return refSetEditor.getNamedObject(name);
    /*
    if (name == "") return null;
    Named found = refSetEditor.getNamedObject(name);
    if (name.equals(found.getName())) return found;
    else return null;
    */
  }
  /**
   * Finds an objectClass whose name matches the given string.
   * @return an UIObjectClass, ObjectClass, or null - depending on which exists
   */
  public Object getNamedObjectClass(String name) {
    return classSetEditor.getNamedObject(name);
  }

  public UIObject getUIObject(IXObject object) {
    if (object == null) return null;
    else if (object.equals(publicDomain)) return this;
    else if (object instanceof Refinement)
      return (UIObject)refSetEditor.getOwnObject(object);
    else if (object instanceof ObjectClass)
      return (UIObject)classSetEditor.getOwnObject(object);
    return null;
  }

  public UIObject findUIObject(Object object) {
    if (object == null) return null;
    else if (object.equals(publicDomain) ||  //looking for UIDomain
	     (object.equals(name))) return this;
    else if (object instanceof UIObject) return (UIObject)object;
    else if (object instanceof Refinement)
      return (UIObject)refSetEditor.findOwnObject(object);
    else if (object instanceof ObjectClass)
      return (UIObject)classSetEditor.findOwnObject(object);
    return null;
  }

  public boolean isCurrentObject(Object o) {
    if (refSetEditor.isCurrentOwnObject(o)) return true;
    else return classSetEditor.isCurrentOwnObject(o);
  }


  /**
   * Finds refinements whose names match the given string. 
   * @return the matches sorted by how early the string appears in the name
   */
  public Set getMatchingRefinements(String pattern) {
    return IVUtil.findNameMatches(pattern, getAllRefinements());
  }

  public Set getMatchingObjectClasses(String pattern) {
    return IVUtil.findNameMatches(pattern, getAllObjectClasses());
  }



  public Set getMatchingRefinements(LList pattern) {
    //List dMatches = publicDomain.getRefinements(pattern); //WHAT??
    //List dMatches = publicDomain.getMatchingRefinements();
    HashSet uiMatches = 
      refSetEditor.getMatchingRefinements(pattern, getAllRefinements());
    return uiMatches;
  }
  /**
   * Checks whether the given object is a refinement that matches the pattern.
   * If it has been removed, it never matches.
   * If it is an original or has been added or edited, it matches if the 
   * patterns match.
   */
  public boolean isMatchingRefinement(LList pattern, Object o) {
    if (o instanceof UIRefinement) {
      return ((UIRefinement)o).matchesPattern(pattern);
    }
    else if (o instanceof Refinement) {
      Refinement r = (Refinement)o;
      //Debug.noteln("UID: check "+pattern.toString()+" in", r.getPattern());
      if (((pattern == null) && (r.getPattern() == null)) ||
	  (pattern.equal(r.getPattern())) || 
	  (pattern.equals(r.getPattern())))
	return true;
      else return (null != SimpleMatcher.match(pattern, r.getPattern()));
    }
    else return false;
  }

  
  //---------------------Listening and Firing---------------------------------

  /** DomainListener */
  public void refinementAdded(RefinementEvent e) {
    Domain source = (Domain)e.getSource();
    //Debug.noteln("UID: Refinement added", e.getRefinement().getName());
    //Debug.noteln(" domain is", source);
    // ignore own updates
    if (!isSaving(source)) fireConstructAdded(e);
  }

  /** DomainListener */
  public void objectClassAdded(DomainEvent e) {
    Domain source = (Domain)e.getSource();
    // ignore own updates
    if (!isSaving(source)) fireConstructAdded(e);
  }

  /** from UIRefinement and UIObjectClass when they are created */
  public void addedObject(UIObject uio, IXObject original) {
    if (uio == null) return;
    else if (uio instanceof UIRefinement)
      refSetEditor.addedObject(uio, original);
    else if (uio instanceof UIObjectClass)
      classSetEditor.addedObject(uio, original);
  }


  //--------------------------Managing domain---------------------------------

  /**
   * Collect all the properties of the given domain.
   */
  private void collectProperties(Domain domain) {
    collectProperties(domain.getObjectClasses());
  }
  /**
   * Collect all the properties of the list of object classes. Works
   * for ObjectClass lists and for UIObjectClass lists. Mixed too.
   */
  private void collectProperties(List objectClasses) {
    properties.clear();
    if (objectClasses == null) return;
    for (Iterator i = objectClasses.iterator(); i.hasNext(); ) {
      Object o = i.next();
      List props = null;
      if (o instanceof ObjectClass) 
	props = ((ObjectClass)o).getObjectProperties();
      else if (o instanceof UIObjectClass) 
	props = ((UIObjectClass)o).getObjectProperties();
      if (props == null) return;
      for (Iterator j = props.iterator(); j.hasNext(); ) {
	ObjectProperty prop = (ObjectProperty)j.next();
	properties.put(prop.getName(), prop);
      }
    }
    //Debug.noteln("UID: Collected props", UIUtil.show(properties.keySet()));
  }

  /**
   * Declare the variables of all refinements as they are currently used.
   * Will declare typos as they are, so user should tidy up at some point!
   * Will also declare "none" if no variables are currently used.
   * @return false if user changed their mind, true otherwise.
   */
  public boolean declareAllVariables() {
    List refs = this.getAllRefinements();
    if (refs == null) {
      Debug.noteln("UID: declaration - domain has no refinements",
		   "Nothing done.");
      return true;
    }
    String[] m = {"This will overwrite all declarations of all refinements",
		  "in the domain.",
		  "For each refinement the currently used variables will be",
		  "declared as they are. If you have mis-typed a variable,",
		  "this mistake will appear in the variable declarations.",
		  "Are you sure you want to do this?"};
    int yes = JOptionPane.showConfirmDialog(editor.deFrame, m, "Warning", 
					    JOptionPane.YES_NO_OPTION, 
					    JOptionPane.WARNING_MESSAGE);
    if (yes != JOptionPane.YES_OPTION) return false; //--->

    for (Iterator i = refs.iterator(); i.hasNext(); ) {
      try {
	Object o = i.next();
	if (o instanceof Refinement) { //unchanged original, copy if needed
	  SortedSet varsNow = ((Refinement)o).getDeclaredVariables();
	  SortedSet vars = ((Refinement)o).getVariablesUsed();
	  if ((varsNow == null) //declare this as none, so make a change!
	      || (!IVUtil.sameSet(vars, varsNow))) { //only update if needed
	    UIObject uio = this.newUIObject((Refinement)o);
	    List decs = IVUtil.makeVariableDeclarations(vars);
	    ((UIRefinement)uio).setVariableDeclarations(decs);
	  }	    
	}
	else if (o instanceof UIRefinement) { //just set the declarations
	  SortedSet vars = ((UIRefinement)o).getVariablesUsed();
	  List decs = IVUtil.makeVariableDeclarations(vars);
	  ((UIRefinement)o).setVariableDeclarations(decs);
	}
      }
      catch (Exception e) {
	Debug.noteException(e);
      }
    }
    return true;
  }

  /**
   * Finds all refinements that use the variable with the given name. Does not
   * look at whether the refinement declares it (used during consistency 
   * checks!)
   * @return a list of the refinements that use the variable
   */
  public Collection findVariableUsers(ItemVar var) {
    if (var == null) return null;
    fullSaveToDomain(draftDomain);
    List refs = draftDomain.getRefinements();
    Collection users = new ArrayList();
    for (Iterator i = refs.iterator(); i.hasNext(); ) {
      try {
	Object o = i.next();
	SortedSet vars = ((Refinement)o).getVariablesUsed();
	if ((vars != null) && (vars.contains(var)))
	  users.add(o);
      }
      catch (Exception e) { Debug.noteException(e); }
    }
    return users;
  }
  /**
   * Finds all refinements that use the variable with the given name. Does not
   * look at whether the refinement declares it (used during consistency 
   * checks!)
   * @return a list of the refinements that use the variable
   */
  public Collection findVariableUsers(String name) {
    String varName = IVUtil.ensureVarName(name);
    if (varName != "") {
      Symbol var = Symbol.intern(varName);
      return findVariableUsers((ItemVar)var);
    }
    else return null;
  }
  /**
   * Finds all variables that are used within the domain's refinements.
   * Does not look at whether the refinement declares it (used during  
   * consistency checks!)
   * @return a set of variables used 
   */
  public Set findVariablesUsed() {
    fullSaveToDomain(draftDomain);
    List refs = draftDomain.getRefinements();
    Set allVars = new HashSet();
    for (Iterator i = refs.iterator(); i.hasNext(); ) {
      try {
	Object o = i.next();
	allVars.addAll(((Refinement)o).getVariablesUsed());
      }
      catch (Exception e) { Debug.noteException(e); }
    }
    return allVars;
  }




  /** Save the object and all its components into its base object */
  public void fullSaveToDomain() {
    if (publicDomain == null) publicDomain = (Domain)makeBaseObject();
    if (draftDomain == null) {
      draftDomain = cloneDomain((Domain)publicDomain, draftDomain);
    }
    fullSaveToDomain(draftDomain);
  }

  public void fullSaveToDomain(Object domainObject) {
    //Debug.noteln("");
    Domain domain = (Domain)domainObject;
    setSaving(domain);
    if (domain != publicDomain) {
      //if updating any other but public, set to public first
      cloneDomain(publicDomain, domain);      
      //Debug.noteln("UID full save to draft");
    }
    //else  Debug.noteln("UID full save to public");

    //domain.setName(name);
    saveToDomain(domain); //save local information (name, comments,...)

    //Debug.noteln("UID: base domain is", publicDomain);
    //Debug.noteln("UID: saving to domain", domain);
    //Debug.noteln("UID: this domain", this.print());
    
    
    refSetEditor.saveToDomain(domain);
    classSetEditor.saveToDomain(domain);

    //Debug.noteln("UID: base domain is", publicDomain);
    //Debug.noteln("UID: saving to domain", domain);
    //Debug.noteln("UID: this domain", this.print());
    //Debug.noteln("UID - finished saveToDomain(domain)");
    //Debug.noteln("");

    unsetSaving();
  }


  public void publishToDomain() {
    if (publicDomain == null) publicDomain = new Domain();
    fullSaveToDomain(draftDomain); //use draft domain for checking
    try {
      //Debug.noteln("UID: draft domain is now", draftDomain);
      draftDomain.checkConsistency();
    }
    catch (SyntaxException se) {
      String[] m = {se.getMessage(),
		    "Listening applications may not be able to handle this.",
		    "Are you sure you want to publish it?"};
      int yes = JOptionPane.showConfirmDialog(editor.deFrame, m, "Warning", 
					      JOptionPane.YES_NO_OPTION, 
					      JOptionPane.WARNING_MESSAGE);
      if (yes != JOptionPane.YES_OPTION) return; //--->
    }
    fullSaveToDomain(publicDomain);
    cloneDomain((Domain)publicDomain, draftDomain);

    refSetEditor.publishedEdits(); //forget previous edits
    classSetEditor.publishedEdits();
    

    //Debug.noteln("Public now:", publicDomain.toString());
    //Debug.noteln("Draft now:", draftDomain.toString());
    //Debug.noteln("UIDomain now:", print());
  }


  /** 
   * Reverts the domain and all its construct to the last published domain.
   * If the domain has not been published, reverts to the original (usually
   * an empty domain).
   */
  public void revertToOriginal() {
    refSetEditor.clear();
    classSetEditor.clear();
    graphs.clear();
    cloneDomain(publicDomain, draftDomain);
    loadFromDomain(publicDomain);
    fireDomainEdited();
  }

  //--------------------------Notifying listeners-----------------------------


  private HashSet listeners = new HashSet();

  public void addUIDomainListener(UIDomainListener listener) {
    listeners.add(listener);
  }
  public HashSet getUIDomainListeners() {
    return listeners;
  }
  public void setUIDomainListeners(HashSet listeners) {
    this.listeners = listeners;
  }
  public void removeUIDomainListener(UIDomainListener listener) {
    listeners.remove(listener);
  }

  public void fireConstructAdded(DomainEvent e) {
    //Debug.noteln("UID added construct");
    //(new Throwable()).printStackTrace();
    for (Iterator i = listeners.iterator(); i.hasNext();) {
      UIDomainListener listener = (UIDomainListener)i.next();
      listener.constructAdded(e);
    }
  }
 public void fireConstructAdded(AbstractSetEditor se, Object object) {
   //Debug.noteln("UID added UIconstruct", object.getName());
    //(new Throwable()).printStackTrace();
    UIDomainEvent event = new UIDomainEvent(this, (UIObject)object);
    for (Iterator i = listeners.iterator(); i.hasNext();) {
      UIDomainListener listener = (UIDomainListener)i.next();
      listener.constructAdded(event);
    }
  }
  public void fireConstructEdited(AbstractSetEditor se, Object object) {
    //Debug.noteln("UID: edited UIConstruct");
    UIDomainEvent event = new UIDomainEvent(this, (UIObject)object);
    for (Iterator i = listeners.iterator(); i.hasNext();) {
      UIDomainListener listener = (UIDomainListener)i.next();
      listener.constructEdited(event);
    }
  }
  public void fireConstructRemoved(AbstractSetEditor se, Object object) {
    //Debug.noteln("UID: removed UIConstruct", object.getName());
    UIDomainEvent event = new UIDomainEvent(this, (UIObject)object);
    for (Iterator i = listeners.iterator(); i.hasNext();) {
      UIDomainListener listener = (UIDomainListener)i.next();
      listener.constructRemoved(event);
    }
  }
  /**
   * The domain's own fields changed (currently only name)
   */
  public void fireDomainEdited() {
    //Debug.noteln("UID: edited domain for listeners", listeners);
    UIDomainEvent event = new UIDomainEvent(this);
    for (Iterator i = listeners.iterator(); i.hasNext();) {
      UIDomainListener listener = (UIDomainListener)i.next();
      listener.domainEdited(event);
    }
  }
  public void fireDomainCleared() {
    //Debug.noteln("UID: cleared domain");
    UIDomainEvent event = new UIDomainEvent(this);
    for (Iterator i = listeners.iterator(); i.hasNext();) {
      UIDomainListener listener = (UIDomainListener)i.next();
      listener.domainCleared(event);
    }
  }
  public void fireDomainSet(Domain domain) {
    //Debug.noteln("UID set domain");
    UIDomainEvent event = new UIDomainEvent(this);
    for (Iterator i = listeners.iterator(); i.hasNext();) {
      UIDomainListener listener = (UIDomainListener)i.next();
      listener.domainSet(event);
    }
  }



  //-----------------------data listeners

  HashSet dataChangeListeners = new HashSet();

  public void addDataChangeListener(DataChangeListener l) {
    dataChangeListeners.add(l); //add if not there
  }
  public void removeDataChangeListener(DataChangeListener l) {
    dataChangeListeners.remove(l);
  }

  private void fireDataChange(String field, Object oldValue, Object newValue) {
    if (sameValue(field, oldValue, newValue)) return; //no change
    for (Iterator i = dataChangeListeners.iterator(); i.hasNext(); ) {
      DataChangeListener l = (DataChangeListener)i.next();
      l.dataChanged(this, field, oldValue, newValue);
    }
  }



  //--------------------------Domain Management-------------------------------
  /** 
   * clears this domain, then sets it to the new one.
   */
  public void setDomain(Domain domain) {
    if (getDomain().equals(domain)) return;
    else { 
      //clear this domain, then set the new domain, then fire
      setCurrentDomain(domain);
      fireDomainSet(domain);
    }
  }


  //-------------------------- things---------------------------------

  public Object clone() throws CloneNotSupportedException {
    return super.clone();
  }
 
  public void clearThis() {
    name = "";
    comments = "";
    clearAnnotations();
  }

  /**
   * This gets rid of everything, including lists that keep track of changes.
   */
  public void clearQuiet() {
    clearThis();
    if (refSetEditor != null) refSetEditor.clear();
    if (classSetEditor != null) classSetEditor.clear();
    graphs.clear();
    draftDomain.clear();
    domainFile = null;
    saveToDomain(draftDomain);
  }
  public void clear() {
    clearQuiet();
    fireDomainCleared();
  }
  public void clearBase() {
    publicDomain.clear();
    clear();
    saveToDomain(publicDomain);
  }
  /**
   * Clear all constructs in the draft domain. If there is a base-domain, 
   * note that all its constructs are removed.
   */
  public void clearDraft() {
    clearQuiet();
    refSetEditor.rememberClear();
    classSetEditor.rememberClear();
    fireDomainCleared();
  }

  public void setDomainFile(File file) {
    domainFile = file;
  }
  public File getDomainFile(File file) {
    return domainFile;
  }
  /**
   * Save the UIDomain to a local file - does not publish! 
   * If the domain has been saved before, it uses the previous file (warning
   * about overwriting). Otherwise it will ask the user for a file.
   */
  public File saveDomain() {
    return saveDomain(domainFile);
  }
  /**
   * Save the UIDomain to a local file - does not publish! 
   * If the domain has been saved before, it uses the previous file (warning
   * about overwriting). Otherwise it will ask the user for a file.
   */
  public File saveDomainAs() {
    return saveDomain(null);
  }
  /**
   * Save the UIDomain to the given local file - does not publish! 
   * Assumes that this domain is up-to-date!
   * Call this with a null argument for saveAs behaviour.
   */
  public File saveDomain(File file) {
    //save this domain into the draftDomain
    fullSaveToDomain(draftDomain);
    Debug.noteln("UID: Saving domain:", draftDomain);
    Debug.noteln(" domain comments:", draftDomain.getComments());
    Debug.noteln(" own comments:", getComments());
    //Debug.noteln(" published domain:", publicDomain);
    
    try {
      draftDomain.checkConsistency();
    }
    catch (SyntaxException se) {
      String[] message = {se.getMessage(),
			  "Are you sure you want to save it?"};
      int yes = JOptionPane.showConfirmDialog(editor.deFrame, 
					      message, "Warning", 
					      JOptionPane.OK_CANCEL_OPTION, 
					      JOptionPane.WARNING_MESSAGE);
      if (yes != JOptionPane.OK_OPTION) {
	//JOptionPane.showMessageDialog(editor, "not saving the domain");
	Debug.noteln("UID: not saving");
	return null; //--->
      }
    }

    //save the draft domain to a suitable file (if there is a given one,
    // use that otherwise ask the user for one).
    Debug.noteln("UID: draftDomain comments are", draftDomain.getComments());
    if (file != null) 
      file = DomainWriter.saveDomain(editor.deFrame, draftDomain, file); 
    //no file known or user decided not to overwrite existing one
    if (file == null) 
      file = DomainWriter.saveDomain(editor.deFrame, draftDomain);
    //remember the file that was used
    if (file != null) domainFile = file;
    return file;
  }


  public void analyseDomain() {
    //publicDomain.checkRefinementReferences();
  }

  public void checkConsistency() {
    fullSaveToDomain(draftDomain);
    try {
      draftDomain.checkConsistency();
    }
    catch (SyntaxException se) {
      JOptionPane.showMessageDialog(editor.deFrame, se.getMessage());
    }
  }

  public HashMap collectInconsistencies() {
    fullSaveToDomain(draftDomain);
    List refs = draftDomain.getRefinements();
    HashMap problems = new HashMap();
    for (Iterator i = refs.iterator(); i.hasNext();) {
      Refinement r = (Refinement)i.next();
      try { r.checkConsistency(); }
      catch (SyntaxException se) {
	problems.put(se.getMessage(), r);
      }
    }
    return problems;
  }

  /**
   * todo!!!!!!!!!!
   */
  // ask: go ahead? fix problem list? declare variables?
  public boolean checkAndListConsistency() {
    Debug.noteln("Checking collective consistency of", this);
    HashMap problems = collectInconsistencies();
    if (problems.size() == 0) {
      return true;
    }
    else {
      Object[] probs = problems.keySet().toArray();
      String[] p = {"The domain " + getName() + 
		    "contains the inconsistencies listed below",
		    "To go straight to editing a refinement with a listed" +
		    " inconsistency,",
		    "select it in the list and click ok.",
		    "To declare variables as they are currently used" +
		    " in all refinements,",
		    "first click 'Cancel' then use the domain's" +
		    " 'declare variables' option",
		    "in the 'Edit' menu of this panel (domain panel)."};
      Object todo = 
	JOptionPane.showInputDialog(editor.deFrame, p, 
				    "Domain Inconsistencies",
				    JOptionPane.INFORMATION_MESSAGE, null,
				    probs, probs[0]);
      Object or = problems.get(todo);
      if (or != null) {
	try {
	  Refinement refinement = (Refinement)or;
	  editor.setConstruct(refinement);
	}
	catch (Exception e) {
	  Debug.noteException(e);
	}
      }
      return false;
    }
  }

  public void checkAndReportConsistency() {
    Debug.noteln("Checking collective consistency of", this);
    fullSaveToDomain(draftDomain);
    List refs = draftDomain.getRefinements();
    HashMap problems = new HashMap();
    for (Iterator i = refs.iterator(); i.hasNext();) {
      Refinement r = (Refinement)i.next();
      try { r.checkConsistency(); }
      catch (SyntaxException se) {
	problems.put(se.getMessage(), r);
      }
    }
    if (problems.size() == 0) {
      JOptionPane.showMessageDialog(editor.deFrame, 
				    "No inconsistencies found");
    }
    else {
      Object[] probs = problems.keySet().toArray();
      String[] p = {"The domain " + getName() + 
		    "contains the inconsistencies listed below",
		    "To go straight to editing a refinement with a listed" +
		    " inconsistency,",
		    "select it in the list and click ok.",
		    "To declare variables as they are currently used" +
		    " in all refinements,",
		    "first click 'Cancel' then use the domain's" +
		    " 'declare variables' option",
		    "in the 'Edit' menu of this panel (domain panel)."};
      Object todo = 
	JOptionPane.showInputDialog(editor.deFrame, p, 
				    "Domain Inconsistencies",
				    JOptionPane.INFORMATION_MESSAGE, null,
				    probs, probs[0]);
      Object or = problems.get(todo);
      if (or != null) {
	try {
	  Refinement refinement = (Refinement)or;
	  editor.setConstruct(refinement);
	}
	catch (Exception e) {
	  Debug.noteException(e);
	}
      }
    }
  }



  private Domain cloneDomain(Domain original, Domain clone) {
    try {
      //Make a deep copy! otherwise the refinements will be same...
      copyDomain(original, clone);
      return clone;
    }
    catch (Exception e) {
      Debug.noteln("UID: Cloning of the domain failed");
      Debug.noteException(e, false);
      return null;
    }
  }

  private Domain copyDomain(Domain original, Domain newDom) {
    if (newDom == null) newDom = new Domain();
    try {
      //Make a deep copy! otherwise the refinements will be same...
      newDom.setName(original.getName());

      Annotations as = original.getAnnotations();
      if (as != null) as = new Annotations(as);
      newDom.setAnnotations(as);
      
      newDom.setComments(original.getComments());

      newDom.setRefinements(copyConstructs(original.getRefinements(),
					   new LinkedListOfRefinement()));
      newDom.setObjectClasses(copyConstructs(original.getObjectClasses(),
					   new LinkedListOfObjectClass()));
      return newDom;
    }
    catch (Exception e) {
      Debug.noteException(e);
      return null;
    }
  }

  private static List copyConstructs(List originals, List copies) {
    if (originals != null) {
      for (Iterator i = originals.iterator(); i.hasNext(); ) {
	try {
	  Object ob = i.next();
	  Object newOb;
	  if (ob instanceof Refinement) {
	    newOb = ((Refinement)ob).clone();
	    copies.add(newOb);
	  }
	  else if (ob instanceof ObjectClass) {
	    newOb = ((ObjectClass)ob).clone();
	    copies.add(newOb);
	  }
	}
	catch (CloneNotSupportedException e) {
	  Debug.noteln("UID: Cloning of a construct failed; skipping it.");
	  Debug.noteException(e, false);
	}
      }
    }
    return copies;
  }



  public String toString() { return name; }
  public String print() { 	
    List refinements = getAllRefinements();
    List objectClasses = getAllObjectClasses();
    return "UIDomain[" + name + ", "  + 
      refinements.size() + "        refinements," +
      objectClasses.size() + "        object classes]";
  }
  public String printSets() { 	
    List refinements = getAllRefinements();
    List objectClasses = getAllObjectClasses();
    return "UIDomain[" + name + ", "  +UIUtil.lineSeparator+ 
      " refinements " + refSetEditor.printSet() +UIUtil.lineSeparator+ 
      " object classes " + classSetEditor.printSet()+"]";
  }
  public String printDetailedSets() { 	
    List refinements = getAllRefinements();
    List objectClasses = getAllObjectClasses();
    return "UIDomain["+name+", "  + UIUtil.lineSeparator+
      " refinements " + refSetEditor.printDetails()+UIUtil.lineSeparator+
      " object classes " + classSetEditor.printDetails()+"]";
  }

  public void setLoading(boolean onOff) {
    loading = onOff;
  }
  public boolean isLoading() {
    return loading;
  }
  public void setOverwrite(boolean onOff) {
    refSetEditor.setOverwrite(onOff);
    classSetEditor.setOverwrite(onOff);
  }


  /**
   * Loads a domain from a file into the draft domain. 
   * loads the domain object into this, adds all refinements, and fires change.
   */
  public void insertDomain() {
    setOverwrite(true);
    setLoading(true);
    Cursor cursor = editor.deFrame.getCursor();
    try {
      editor.deFrame.setCursor(new Cursor(Cursor.WAIT_CURSOR));
      //fullSaveToDomain(); //any changes - put them into draft

      Domain newDomain = new Domain();
      File file = DomainParser.loadDomain(editor.deFrame, newDomain);
      //Debug.noteln("UID - loading domain. Name is", newDomain.getName());
      if (file != null) {
	domainFile = file;
	//copy the domain's new own fields
	insertFromDomain(newDomain);  
	//draftDomain.takeFrom(newDomain);
	//copy the new refinements
	List refinements = newDomain.getRefinements();
	for (Iterator i = refinements.iterator(); i.hasNext(); ) {
	  IXObject ixo = (IXObject)i.next();
	  addConstruct(getUIObject(ixo));
	}
	//copy the new object classes
	LinkedList classes = (LinkedList)newDomain.getObjectClasses();
	//Debug.noteln("UID: new classes are", UIUtil.show(classes));
	/* this does not work for the tree!
	List newClasses = new ArrayList();
	for (Iterator j = classes.iterator(); j.hasNext(); ) {
	  IXObject ixoc = (IXObject)j.next();
	  addConstruct(getUIObject(ixoc));
	}
	*/
	classSetEditor.setObjects(classes);
	//Debug.noteln(" now in uid:", UIUtil.show(getAllObjectClasses()));
	//Debug.noteln("UID: this",UIUtil.show(getAllRefinements()));
	//Debug.noteln(" draft",UIUtil.show(draftDomain.getRefinements()));
	//Debug.noteln(" current",UIUtil.show(publicDomain.getRefinements()));
      }
      editor.deFrame.setCursor(cursor);
      setLoading(false);
      setOverwrite(false);
      fireDomainEdited();
    }
    catch (Exception e) {
      Debug.noteException(e);
      editor.deFrame.setCursor(cursor);
      setLoading(false);
      setOverwrite(false);
      fireDomainEdited();
    }
  }

  /**
   * Clears the draft domain (noting all of original domain as removed) 
   * and loads a domain-file into the draft domain.
   */
  public void loadDomain() {
    //clear added, edited, removed; remove and remember all in draft domain
    clearDraft();
    draftDomain = new Domain(); //needed so that the parser knows it is empty
    insertDomain();
    //loadFromDomain(draftDomain);  

    /*
    String name = draftDomain.getName();
    String name = draftDomain.getName();
    Debug.noteln("UID loaded name to tidy", name);
    if ((name != null) && (name.startsWith("+"))) {
      name = name.substring(1);
      draftDomain.setName(name);
      setName(name);
    }
    */			  
  }





  public void addGraph(Named g) {
    graphs.put(g.getName(), g);
    fireGraphAdded(g);
  }

  public void fireGraphAdded(Object g) {
  }

  public LList getAllGraphs() {
    LListCollector list = new LListCollector(graphs.values());
    return list.contents();
  }
  public Named getNamedGraph(String name) {
    return (Named)graphs.get(name);
  }
  public void removeNamedGraph(String name) {
    graphs.remove(name);
  }


  //-------------------------------ObjectClass stuff--------------------



}


// Issues:
// * tidy up load/insert domain stuff 
// * 
// * 
