/****************************************************************************
 * A manager that keeps track of an object's edits for undo/redo purposis
 *
 * @author Jussi Stader
 * @version 4.1
 * Updated: Mon Oct 24 14:42:37 2005
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iview;

import java.lang.Math;
import java.util.*;
import javax.swing.*;       
import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.Component;
import java.awt.event.*;
import ix.*;
import ix.util.Debug;
import ix.util.Util;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.iview.*;
import ix.iview.domain.*;
import ix.iview.util.*;
import ix.iview.event.*;

/****************************************************************************
 * A manager that keeps track of objects' edits for undo/redo purposes.
 *
 * Keeps a current object and its edits.
 *
 * Keeps a list of undos for each object that has been edited.
 *
 * To use an UndoManager, do the following:
 * 
 *****************************************************************************
 */
public class UndoManager 
  implements DataChangeListener, CurrentConstructListener
{

  //---------------------Fields-------------------------------------------
  /** A list of slot / previous value pairs for the current object **/
  protected ArrayList currentUndos = new ArrayList();
  protected EditableObject currentObject;

  protected boolean undoing = false;
  protected int undoIndex = 0; //changes when undo/redo is done 
  /** a map of undos for previously edited objects **/
  protected HashMap undoObjects = new HashMap();
  private HashSet undoListeners = new HashSet();
  private String cannotUndo = 
    "Undo/Redo is only available in form-based views";

  private UndoEditing editor;   // source of editing for data lookup/setting.


  //---------------------Constructors----------------------------------------

  /**
   * Creates an undo manager with the given source of the editing for
   * data lookup and setting.
   */    
  public UndoManager(UndoEditing editor){
    super();
    this.editor = editor;
    Debug.noteln("UndoMan created for", editor);
  }

  //--------------------- Public services

  /**
   * Checks if the manager is currently undoing a change
   */
  public boolean isUndoing() {
    return undoing;
  }

  /**
   * Notes an undoable change.
   */
  public synchronized void noteUndo(EditableObject uio, String field, 
				       Object oldValue, Object newValue) {
    //Debug.noteln("UndoMan: noting undo");
    if (uio == null) return;
    //if (uio.sameValue(field, oldValue, newValue)) return; //--->no change
    ensureConstructSet(uio);
    //remove redo stack
    while (undoIndex > 0) {
      undoIndex = undoIndex-1;
      currentUndos.remove(undoIndex);
    }
    UndoNote note = new UndoNote(uio, field, oldValue, newValue);
    //Debug.noteln("UndoMan: undo note", note.toString());
    currentUndos.add(0, note);
    fireUndoChange();
    //Debug.noteln("UndoMan: undo note", note.toString());
  }

  /**
   * Undoes an undoable change.
   */
  public synchronized boolean undo() {
    //Debug.noteln("UndoMan: undo start");
    if (!canUndo()) {
      //Debug.noteln(" cannot undo");
      fireUndoChange(); //must have made a mistake!
      return false;
    }
    undoing = true;
    if (undoIndex < 0) undoIndex = 0;
    UndoNote note = findUndoNote();
    if (note == null) {
      Debug.noteln("Cannot find anything to undo at step", undoIndex);
      Debug.noteln(" in notes of size", currentUndos.size());
      fireUndoChange();
      return false;
    }
    //note.noteDebug();
    //Debug.noteln("Undoing note:", note);
    editor.undoSetValue((String)note.field, note.oldValue);
    note.undone = true;
    note.redone = false;
    undoIndex = undoIndex+1;
    fireUndoChange();
    undoing = false;
    return true;
  }
  /**
   * As above, but for a given object. The manager will swap current
   * object if the given one does not match the current one.
   */
  public synchronized boolean undo(EditableObject uio) {
    if (uio == null) return false;
    ensureConstructSet(uio);
    return undo();
  }
  /**
   * Redoes a change that has previously been undone.
   */
  public synchronized boolean redo() {
    if (!canRedo()) {
      fireUndoChange(); //must have made a mistake!
      return false;
    }
    undoing = true;
    if (undoIndex >= currentUndos.size()) undoIndex = currentUndos.size() - 1;
    UndoNote note = findRedoNote();
    if (note == null) {
      Debug.noteln("Cannot find anything to redo");
      fireUndoChange();
      return false;
    }
    //Debug.noteln("Redoing note:", note);
    //note.noteDebug();
    editor.undoSetValue((String)note.field, note.newValue);
    note.undone = false;
    note.redone = true;
    undoIndex = undoIndex-1;
    fireUndoChange();
    undoing = false;
    return true;
  }
  /**
   * As above, but for a given object. The manager will swap current
   * object if the given one does not match the current one.
   */
  public synchronized boolean redo(EditableObject uio) {
    if (uio == null) return false;
    ensureConstructSet(uio);
    return redo();
  }


  //look along undos to find first that has not been undone yet (this or next)
  private synchronized UndoNote findUndoNote() {
    //Debug.noteln("UndoMan: Finding undo note at", undoIndex);
    //Debug.noteln(" undos are of size", currentUndos.size());
    try {
      UndoNote note = (UndoNote)currentUndos.get(undoIndex);
      if (note == null) {
	Debug.noteln("Found null undo note at index", undoIndex);
      }
      //Debug.noteln("got note"); note.noteDebug();
      if (!note.undone) {
	//ignore if already same old value! 
	// (needed for edit-pairs, e.g. constraint-condition edits)
	Object currentValue = editor.undoGetValue((String)note.field);
	//Debug.noteln("got current value");
	if (currentObject.sameValue(note.field, note.oldValue, currentValue)) {
	  //nothing to undo in this one - already on old value!
	  Debug.noteln(" skipping step that makes no change");
	  note.redone = false;
	  note.undone = true;
	  undoIndex = undoIndex+1;
	  fireUndoChange();
	  return findUndoNote();
	}
	else {
	  //Debug.noteln("Found undo note!");
	  return note;
	}
      }
      else {
	Debug.noteln(" note already undone; skipping");
	undoIndex = undoIndex+1;
	fireUndoChange();
	return findUndoNote();
      }
    }
    catch (Exception e) {
      Debug.noteln("UndoMan: something wrong finding undo note");
      Debug.noteln("Exception class", e.getClass());
      Debug.noteln("Exception", e);
      Debug.describeException(e);
      e.printStackTrace();    
      return null;
    }
  }
  private UndoNote findRedoNote() {
    //Debug.noteln("UndoMan: Finding redo note at", undoIndex);
    try {
      UndoNote note = (UndoNote)currentUndos.get(undoIndex);
      if (!note.redone) {
	//ignore if already same new value! (pairs, e.g. constr/cond)
	// (needed for edit-pairs, e.g. constraint-condition edits)
	Object currentValue = editor.undoGetValue((String)note.field);
	if (currentObject.sameValue(note.field, note.newValue, currentValue)) {
	  //nothing to redo in this one - already on new value!
	  note.redone = true;
	  note.undone = false;
	  undoIndex = undoIndex-1;
	  fireUndoChange();
	  return findRedoNote();
	}
	else return note;
      }
      else {
	undoIndex = undoIndex-1;
	fireUndoChange();
	return findUndoNote();
      }
    }
    catch (Exception e) {
      Debug.describeException(e);
      return null;
    }
  }

  public void clearUndos() {
    try {
      currentUndos = new ArrayList();
      undoIndex = 0;
      fireUndoChange();
    }
    catch (Exception e) {
      Debug.describeException(e);
    }
  }

  public void ensureConstructSet(EditableObject uio) {
    // given object already set
    if ((currentObject != null) && currentObject.equals(uio)) return;
    // file undos if there is an old object and it is not the given one
    else if (currentObject != null) {
      fileUndos(currentObject);
    }
    retrieveUndos(uio);
  }
  public void setConstruct(EditableObject uio) {
    //file undos if there is an old object and it is not the given one
    if ((currentObject != null) && !currentObject.equals(uio)) {
      fileUndos(currentObject);
    }
    retrieveUndos(uio); //no old one or different
  }

  private void fileUndos(EditableObject uio) {
    if ((currentUndos.size() > 0) && (uio != null)) {
      while (undoIndex > 0) {  //remove redo stack (this will be lost)
	undoIndex = undoIndex-1;
	currentUndos.remove(undoIndex);
      }
      //replace note for this object for later
      undoObjects.remove(uio);
      if (currentUndos.size() > 0) undoObjects.put(uio, currentUndos);
    }
    //clear undos after filing
    clearUndos();
  }
  //retrieve any previous undos for object (or make new list)
  // and start listening to changes
  private void retrieveUndos(EditableObject uio) {
    ArrayList undos = (ArrayList)undoObjects.get(uio); 
    if (undos == null) currentUndos = new ArrayList();
    else currentUndos = undos;
    currentObject = uio;
    if (uio != null) uio.addDataChangeListener(this);
    fireUndoChange();
  }


  public boolean canUndo() {
    //Debug.noteln("index: ", undoIndex);
    //Debug.noteln("size: ", currentUndos.size());
    if ((currentUndos != null) && 
	(currentUndos.size() > 0) && (undoIndex < currentUndos.size())) {
      //Debug.noteln("there are undos");
      //we do have some undos
      if (undoIndex == currentUndos.size() - 1) { 
	//only one undo left. undone it already?
	UndoNote undo = (UndoNote)currentUndos.get(undoIndex);
	return !undo.undone;
      }
      else return true; //more than one left, so there must be scope!
    }
    else return false; //no undos available or none left
  }
  public boolean canRedo() {
    //Debug.noteln("UndoMan: checking can redo for index", undoIndex);
    if ((currentUndos != null) && 
	(currentUndos.size() > 0) && (undoIndex >= 0)) {
      //Debug.noteln("there are redos");
      //we do have some redos
      if (undoIndex == 0) { 
	//only one redo left. redone it already?
	UndoNote undo = (UndoNote)currentUndos.get(undoIndex);
	return (undo.undone);
      }
      else return true; //more than one left, so there must be scope!
    }
    else return false; //no redos available or none left
  }




  //---------------------- Listener things ----------------------

  /** make sure you only call this if data really has changed! **/
  public void dataChanged(EditableObject object, String field, 
			  Object oldValue, Object newValue) {
    //if (undoing) Debug.noteln("UM ignoring own data changes");

    if (undoing) return;  //ignore own data changes
    //Debug.noteln("UndoMan: got UIO change for ", field);

    if ((object != null) &&   //only listen to change in current object
	object.equals(currentObject)) {
      noteUndo(object, field, oldValue, newValue);
      //validate(); //make sure the screen updates; not sure whether this helps
    }
  }

  public void constructChanged(Component source, 
			       UIObject oldUIO, UIObject newUIO) {
    /*
    Debug.noteln("UM: construct changed");
    if (oldUIO != null) Debug.noteln("from ", oldUIO);
    else Debug.noteln("from null");
    if (newUIO != null) Debug.noteln("to ", newUIO);
    else Debug.noteln("to null");
    */

    try {((EditableObject)oldUIO).removeDataChangeListener(this); }
    catch (Exception e) {}
    setConstruct((EditableObject)newUIO);
    //test();
  }

  public void test() {
    Debug.noteln("UM**************", this);
    Debug.noteln("  ***** for", editor.getClass());
    if (currentObject == null) Debug.noteln(" current Object null");
    else {
      Debug.noteln(" current Object:", currentObject);
      Debug.noteln(" current undos:", UIUtil.show(currentUndos));
      UIObject uio = ((ConstructEditing)editor).getUIConstruct();
      if (uio == null) Debug.noteln(" Editor''s object: null");
      else {
	Debug.noteln(" Editor''s object:", uio);
	Debug.noteln("  details:", uio.print());
      }
    }
  }

  //--------------------------- Event things ------------------------------
  protected void fireUndoChange() {
    //Debug.noteln("UndoMan: firing undo change");
    try {
      if (undoListeners.isEmpty()) return;
      //if (mainPanel == null) return;
      else {
	Iterator e = undoListeners.iterator();
	while (e.hasNext()) {
	  UndoChangeListener l = (UndoChangeListener)e.next();
	  l.undoChanged(this, currentObject);
	}
      }    
    }
    catch (Throwable t) {
      //not important enough to fall over this!
      Debug.noteln("UndoMan: something went wrong firing undo change");
      //Debug.describeException(t);      
      Debug.noteException(t);      
    }
  }

  public void addUndoListener(UndoChangeListener listener) {
    undoListeners.add(listener);
  }


  protected class UndoNote {
    EditableObject o;
    String field;
    Object oldValue;
    Object newValue;
    boolean undone = false;
    boolean redone = false;

    protected UndoNote(EditableObject o, String field, 
		       Object oldValue, Object newValue) {
      super();
      //Debug.noteln("ACFP: making new undo note");
      this.o = o;
      this.field = field;
      this.oldValue = oldValue;
      this.newValue = newValue;
      //Debug.noteln("new undo is done");
    }

    public void noteDebug() {
      try{
	Debug.note("Undo note: ");
	Debug.note(o.toString());
	Debug.note(" ");
	Debug.note(field);
	Debug.note(" ");
	if (oldValue == null) Debug.note("null");
	else Debug.note(oldValue.toString());
	Debug.note(" -to- ");
	if (newValue == null) Debug.note("null");
	else Debug.note(newValue.toString());
	Debug.noteln("; undone:", undone);
      }
      catch (Exception e) {
	Debug.describeException(e);
      }
     }

  }

}

/*Issues***************************************************************
 *
/*Todos***************************************************************
 *
 * - 
 *  Wait for 
 *
 *****************************************************************************
 */
