/****************************************************************************
 * A simple text-input editor for entering specifications, e.g. for nodes
 * and issues.
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Thu Jan 25 10:12:50 2007
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import javax.swing.text.JTextComponent;       
import java.util.*;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import ix.*;
import ix.util.*;
import ix.util.Debug;
import ix.util.lisp.LList;
import ix.util.lisp.Lisp;
import ix.icore.domain.*;
import ix.icore.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;

/****************************************************************************
 * A simple text-input editor for entering specifications, e.g. for nodes
 * and issues. It is like a JOptions dialogue, but it has an "Add More" 
 * button.
 *
 * The editor is implemented as a frame using JTextField objects
 * and JButtons.  <p>
 *
 * The editor can be created with no arguments or with a given refinement
 * which will be set as the current refinement for which the specifications
 * are to be defined. Subsequently, the refinement in the editor can be set
 * using the setRefinement method.
 *****************************************************************************
 */
public class SpecEditor extends ConstraintEditor 
{
  
  public String type;
  private AbstractEditorPanel specEd;

  /**
   * Creates an editor for specifying time point relations.
   * The editor has components for selecting two time points of two
   * sub-activities, and two buttons (Ok, Cancel)
   */
  public SpecEditor(Frame owner, String type) {
    this(owner, type, "Please enter the pattern of the "+type+".");
  }
    
  /**
   * Creates an editor for specifying time point relations.
   * The editor has components for selecting two time points of two
   * sub-activities, and two buttons (Ok, Cancel)
   */
  public SpecEditor(Frame owner, String type, String message) {
    super(owner, "Specification Editor", message);
    this.type = type;
    this.setSize(405, 120);
  }
    
  
  /**
   * Makes all items on the relations panel - just a text field.
   */
  protected Component makeRels() {
    JTextField specField = new JTextField();
    IVUtil.enableVars(this, specField);
    specEd = new ThingEditorPanel(null, "", specField);
    return specEd;
  }

  public void setType(String type) {
    this.type = type;
  }
  public void setText(String text) {
    specEd.setData(text);
  }

  protected JComponent getFocusComponent() {
    return specEd.getItemComponent();
  }

  public void start() {
    setVisible(true);
    initFromObject();
  }

  /**
   * Nothing to do to initialise the editor from an object.
   */
  protected void initFromObject() {
    setText("");
    //specEd.getItemComponent().requestFocus();
    JComponent c = specEd.getItemComponent();
    getFocusComponent().requestFocus();
  }
  /**
   * Nothing to do to initialise the editor from an object.
   */
  protected void setFromObject(Object original) {
    LList pattern = Lisp.NIL;
    if (Issue.class.isInstance(original)) 
      pattern = ((Issue)original).getPattern();
    else if (NodeSpec.class.isInstance(original)) 
      pattern = ((NodeSpec)original).getPattern();
    
    setText(UIUtil.listToDisplay(pattern));
    //specEd.getItemComponent().requestFocus();
    getFocusComponent().requestFocus();
  }


  /**
   * Reads the selected relation and let interested parties know.
   * Checks that selections have been made.
   */
  protected Object collectConstraint() {
    String spec = specEd.getData().toString().trim();
    if (spec.equals("")) return null;
    else {
      //Debug.noteln("SE: Got spec:", spec + ".");
      LList pSpec = null;
      try {
	pSpec = Lisp.elementsFromString(spec);
      }
      catch (Exception e) {
	  Debug.displayException(e);
	  return null;
      }
      if (type.equals("issue")) {
	if (isEditing()) {
	  try {
	    Issue issue = (Issue)((Issue)original).clone();
	    issue.setPattern(pSpec);
	    return issue;
	  }
	  catch (CloneNotSupportedException cnse) {
	    Debug.describeException(cnse);
	    return new Issue(pSpec);
	  }
	}
	else return new Issue(pSpec);
      }
      else if (type.equals("sub-activity")) {
	if (isEditing()) {
	  try {
	    NodeSpec nSpec = (NodeSpec)((UINodeSpec)original).clone();
	    nSpec.setPattern(pSpec);
	    return nSpec;
	  }
	  catch (CloneNotSupportedException cnse) {
	    NodeSpec nSpec = (NodeSpec)original;
	    UIObject uio = (UIObject)currentObject;
	    return new UINodeSpec(uio.getUIDomain(), nSpec.getId(), pSpec);
	  }
	}
	else {
	  UIRefinement uio = (UIRefinement)currentObject;
	  Name newId = uio.getNewNodeId();
	  return new UINodeSpec(uio.getUIDomain(), newId, pSpec);

	}
      }
      else {
	Debug.noteln("SpecEditor: cannot do specs of type", type);
	(new Throwable()).printStackTrace();
	return null;
      }
    }
  }
}

