/****************************************************************************
 * An editor for specifying temporal relations between time points in
 * refinements.
 *
 * @author Jussi Stader
 * @version 3.1
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import java.util.*;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import ix.*;
import ix.util.*;
import ix.util.Debug;
import ix.util.lisp.LList;
import ix.util.lisp.Lisp;
import ix.icore.domain.*;
import ix.iview.util.NodeNumberRenderer;
import ix.iview.domain.*;
import ix.iface.ui.*;

/****************************************************************************
 * An editor for specifying temporal relations between time points in
 * refinements.
 *
 * The nodes to be related are the children of a given refinement. <p>
 *
 * The editor is implemented as a frame using JList objects, a JCheckBox
 * and JButtons.  <p>
 *
 * The editor can be created with no arguments or with a given refinement
 * which will be set as the current refinement whose children are to be
 * related. Subsequently, the refinement in the editor can be set using the
 * setRefinement method.
 *****************************************************************************
 */
public class OrderingEditor extends ConstraintEditor
{
  //relation items
  protected JList actListA;
  protected JList timeListA;
  protected JLabel beforeLabel;
  protected JList timeListB;
  protected JList actListB;
  
  /**
   * Creates an editor for specifying time point relations.
   * The editor has components for selecting two time points of two
   * sub-activities, and two buttons (Ok, Cancel)
   */
  public OrderingEditor(Frame owner){
    super(owner, "Activity Relation Editor", 
	  "Please specify orderings (select 2 activities and 2 time points).");
    }
    
  /**
   * Creates an editor for specifying time point relations within the given
   * refinement.
   * As OrderingEditor() but with a given refinement.
   * @param refinement the refinement whose sub-activities are to be related
   */
  public OrderingEditor(Frame owner, UIRefinement refinement){
    super(owner, "Activity Relation Editor", 
	  "Please specify orderings (select 2 activities and 2 time points).",
	  refinement);
  }
  
  
  /**
   * Makes all items on the relations panel.
   * These are two identical activity lists and two identical time point lists
   * and a "Before" label between the two
   */
  protected Component makeRels() {
    Box relsBox = new Box(BoxLayout.X_AXIS);
    //make the first activity list box
    //Box boxAct1 = makeActBox(this.actListA);
    actListA = new JList();
    ListEditorPanel actPanel1 = makeActPanel(this.actListA);

    
    //make sub-panels for the time points
    timeListA = new JList(End.values().toArray());
    ListEditorPanel timePanelA = new ListEditorPanel(this,"Time",timeListA);
    timeListB = new JList(End.values().toArray());
    ListEditorPanel timePanelB = new ListEditorPanel(this,"Time",timeListB);
    this.timeListA.setAutoscrolls(true);
    this.timeListA.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    this.timeListA.setSelectedIndex(1);
    this.timeListB.setAutoscrolls(true);
    this.timeListB.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    this.timeListB.setSelectedIndex(0);
  
    //make the second activity list box
    //Box boxAct2 = makeActBox(this.actListB);
    actListB = new JList();
    ListEditorPanel actPanel2 = makeActPanel(this.actListB);

    //add items to the panel
    relsBox.add(actPanel1);
    relsBox.add(timePanelA);
    relsBox.add(Box.createHorizontalStrut(5));
    relsBox.add(Box.createGlue());
    beforeLabel = new JLabel("Before");
    relsBox.add(beforeLabel);
    relsBox.add(Box.createGlue());
    relsBox.add(Box.createHorizontalStrut(5));
    relsBox.add(actPanel2);
    relsBox.add(timePanelB);
    return relsBox;
  }

  /*
   * Make a sub-panel with the given activity list to be filled later
   * @param list the JList that will hold the activities
   * Uses a NodeNumberRenderer to display the list items
   * @return the new panel
   */
  private ListEditorPanel makeActPanel(JList list){
    ListEditorPanel panelAct = new ListEditorPanel(this,"Activity",list);
    list.setAutoscrolls(true);
    list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    //list.setBackground(Color.white);
    list.setCellRenderer(new NodeNumberRenderer(false));
    return panelAct;
  }


  /**
   * Sets the refinement whose components (children) are to be realted.
   * Fills the activity lists of the panel with the children of the refinement.
   * @param refinement the refinement whose children are to be related 
   *
   * @throws NullPointerException if the refinement has no children.
   */
  protected void initFromObject() throws NullPointerException {
    UIRefinement refinement = (UIRefinement) currentObject;
    if (refinement == null) return;
    List children = refinement.getNodes();
    // Debug.noteln("Children are", children);
    if (children.isEmpty()) {
      //Debug.noteln("empty");
      String message = "the refinement " + refinement.getName() +
	" has no children to relate.";
      throw new NullPointerException(message);    //---->
    }
    else {
      this.makeActList(this.actListA, children.toArray());
      this.makeActList(this.actListB, children.toArray());
    }
  }

  protected void setFromObject(Object original) {
    actListA.clearSelection();
    actListB.clearSelection();
    timeListA.clearSelection();
    timeListB.clearSelection();
    if (original == null) return;

    if (!(original instanceof Ordering)) {
      String message = "Condition/Effect editor can only deal with orderings"
	+ " not objects of class " + original.getClass().getName();
      JOptionPane.showMessageDialog(this,message);
      return;
    }

    Ordering ordering = (Ordering)original;
    NodeEndRef fromE = ordering.getFrom();
    NodeEndRef toE = ordering.getTo();
    //get the NodeSpec from the node end ref
    Name fromNode = fromE.getNode();
    Name toNode = toE.getNode();

    actListA.setSelectedValue(findNodeSpec(fromNode,actListA), true);
    actListB.setSelectedValue(findNodeSpec(toNode,actListB), true);
    timeListA.setSelectedValue(fromE.getEnd(), true);
    timeListB.setSelectedValue(toE.getEnd(), true);
    
  }

  private void makeActList(JList list, Object[] children){
    list.removeAll();
    list.setListData(children);
  }

  /**
   * Reads the selected relation and let interested parties know.
   * Checks that selections have been made.
   */
  protected Object collectConstraint() {
    
    Object nodeA = this.actListA.getSelectedValue();
    Object nodeB = this.actListB.getSelectedValue();
    End timeA = (End) this.timeListA.getSelectedValue();
    End timeB = (End) this.timeListB.getSelectedValue();
    if ((nodeA == null) || (nodeB == null) ||
	(timeA == null) || (timeB == null)){
      Object[] message = {"Please select two activities and two time points.", 
			  "Click Cancel to cancel."};
      JOptionPane.showMessageDialog(this, message);
      return null;
    }
    Name idA = ((NodeSpec)nodeA).getId();
    Name idB = ((NodeSpec)nodeB).getId();
    NodeEndRef fromNode = new NodeEndRef(timeA ,idA);
    NodeEndRef toNode = new NodeEndRef(timeB ,idB);
    if (isEditing()) {
      //***should clone first!
      Ordering ordering = new Ordering(fromNode, toNode);
      return ordering;
    }
    else return new Ordering(fromNode,toNode);
  }


  private NodeSpec findNodeSpec(Name nodeName, JList list) {
    if (nodeName == null) return null;
    ListModel model = list.getModel();
    for (int i=0; i<model.getSize(); i++) {
      Object n = model.getElementAt(i);
      if (n instanceof NodeSpec) {
	NodeSpec ns = (NodeSpec) n;
	if (nodeName.equals(ns.getId())) return ns;
      }
    }
    return null;
  }

}

//This would probably be neater with JComboBox objects but that didn't work.
// see RelationEditor for attempts.

