/****************************************************************************
 * A grammar editor/viewer panel 
 *
 * @author Jussi Stader
 * @version 2.3+
 * Updated: Thu Mar 15 13:59:56 2007
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iview;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
//import java.awt.*;
import java.awt.GridLayout;
import java.awt.Dimension;
import java.awt.event.*;
import javax.swing.table.*;

import ix.*;
import ix.util.Debug;
import ix.util.Util;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.table.*;
import ix.iface.domain.*;
import ix.iview.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;

/****************************************************************************
 * A grammar editor/viewer panel
 *
 * Waiting to be documented and implemented properly. <p>
 * Example code for using the GrammarEditorPanel:
 *<PRE><code>
 *    AConstructFramePanel grammarPanel = new GrammarEditorPanel(ide);
 *</code></PRE>
 *****************************************************************************
 */
public class GrammarEditorPanel extends AConstructFramePanel
implements ActionListener, UIDomainListener {
  private AbstractEditorPanel issuePatternList;
  private AbstractEditorPanel nodePatternList;
  private AbstractEditorPanel constraintPatternList;

  private HashSet issuePatterns = new HashSet();
  private HashSet nodePatterns = new HashSet();
  private List constrPatterns = new ArrayList();

  private MouseListener ml = new MouseAdapter() {};

  private boolean listsAsText = true;//flag for showing lists as list or text

  public GrammarEditorPanel(DomainEditorFrame frame) {
    super(frame);
    //updatePatterns();
  }

  protected JPanel setupMainPanel() {
    //Debug.noteln("GrEP: setting up main panel");
    //scroll panes have ONE child, so use mainPanel for everything
    //mainPanel = new JPanel(new BorderLayout());
    mainPanel = new GrammarFormPanel(this);

    scrollPane.getViewport().add(mainPanel); //mainPanel
    
    listsAsText = parent.listsAsText();
    //JScrollPane jsp = new JScrollPane();
    //jsp.getViewport().add(list);
    //nodePatternList = 
    //  new ListEditorPanel(ml,"Node Patterns",jsp,buttons);
    makeLists();
    //mainPanel.add(patternList, BorderLayout.CENTER);
    return mainPanel;
  }


  public void refresh() {
    updatePatterns();
  }

  public boolean beforeSwitching() {
    return true;
  }
     

  public void setListsAsText(boolean asText) {
    if (listsAsText != asText) {
      listsAsText = asText;
      removeLists();
      makeLists();
      updatePatterns();
    }
  }

  private void makeLists() {
    issuePatternList = makeList(ml, "Issue Patterns");
    nodePatternList = makeList(ml, "Node Patterns");
    constraintPatternList = makeList(ml, "Constraint Patterns");
  }
  private void removeLists() {
    mainPanel.remove(issuePatternList);
    mainPanel.remove(nodePatternList);
    mainPanel.remove(constraintPatternList);
    issuePatternList.removeData();
    nodePatternList.removeData();
    constraintPatternList.removeData();
    issuePatternList = null;
    nodePatternList = null;
    constraintPatternList = null;
  }

  private AbstractEditorPanel makeList(MouseListener ml, String label) {
    //Debug.noteln("GrEP: making list. asText:", new Boolean(listsAsText));
    String[] buttons = {"Add","Delete","Clear"};
    AbstractEditorPanel lep;
    if (label.equals("Constraint Patterns")) {
      //make a table to display generic constraints
      
      AbstractIXTableModel tableModel = 
	new GenericConstraintTableModel(parent.getUIDomain());
      IXTable table = new IXTable(tableModel);
      table.setPreferredScrollableViewportSize(new Dimension(200, 100));
      lep = new ListEditorPanel(this, label, true, table, buttons);
    }
    else if (listsAsText) {
      JTextArea thing = new JTextArea();
      thing.setRows(5);
      lep = new ThingEditorPanel(ml, label, thing, buttons);
    }
    else {
      JList thing = new JList();
      thing.setVisibleRowCount(5);
      thing.setCellRenderer(new IXLabelRenderer(false) { //IXTextRenderer
	public void setInfo(Object object) {
	  String text = "";
	  if (object instanceof Collection)
	    text = UIUtil.listToDisplay((Collection) object);
	  else text = object.toString();
	  setText(text);
	}
      });
      lep = new ListEditorPanel(ml, label, thing, buttons);
    } 
    lep.setEnabled(false);
    mainPanel.add(lep);
    return lep;
  }


  protected UIObject cloneConstruct(UIObject uiConstruct) {
    return null;
  }

 /** 
   * Make a new construct of the panel's type and get ready to edit it.
   */
  public void newConstruct() {
    //this.setConstruct(null);
  }
  public UIObject makeNewConstruct() {
    return null;
  }
  /** 
   * Save a construct of the panel's type into the draft domain.
   */
  public void saveConstruct() {
    UIUtil.notImplemented(this,"Saving grammars");
  }
  /** 
   * Notes the content of the editing panel into the current(new?) UIObject.
   */
  protected UIObject noteConstruct() {
    return null;
  }
  /** 
   * Put a given construct of the panel's type into the panel ready for editing
   */
  public void setConstruct(IXObject construct) {
    if (construct == null) {
      issuePatternList.removeData();
      nodePatternList.removeData();
      constraintPatternList.removeData();
    }
    else UIUtil.notImplemented(this,"Setting grammars");
  }
  /** 
   *  Check a construct of the panel's type for errors and inconsistencies
   */
  public boolean checkConstruct() {
    UIUtil.notImplemented(this,"Checking grammars");
    return true;
  }
  /** 
   * Get a full list of the constructs of the panel's type in the domain.
   */
  public List getAllConstructs() {
    return new LListCollector();
  }
  public Named getNamedConstruct(String name) {
    return null;
  }

  public void constructAdded(UIDomainEvent se) {
    updatePatterns();
  }
  public void constructAdded(DomainEvent se) {
    updatePatterns();
  }
  public void constructEdited(UIDomainEvent se) {
    updatePatterns();
  }
  public void constructRemoved(UIDomainEvent se) {
    updatePatterns();
  }
  public void domainCleared(UIDomainEvent se) {
    //se.getDomain().getGrammar().initGenericConstraints();
    updatePatterns();
  }
  public void domainEdited(UIDomainEvent se) {
  }
  public void domainSet(UIDomainEvent se) {
    //se.getDomain().getGrammar().initGenericConstraints();
    updatePatterns();
  }

  private void updatePatterns() {
    issuePatterns.clear();
    nodePatterns.clear();
    constrPatterns.clear();
    constrPatterns = getConstraintPatterns(null, constrPatterns);
    UIDomain dom = parent.getUIDomain();
    if (dom != null) {
      List refs = dom.getAllRefinements();
      for (Iterator i = refs.iterator(); i.hasNext(); ) 
	addAllPatterns(i.next());
    }
    //Debug.noteln("Constraint patterns:", UIUtil.show(constrPatterns));
    updateListPatterns(issuePatternList, issuePatterns);
    updateListPatterns(nodePatternList, nodePatterns);
    updateListPatterns(constraintPatternList, constrPatterns);
  }

  private void addAllPatterns(Object o) {
    if (o == null) return;

    Iterator i = null;
    List rn = null;
    List ri = null;
    List rc = null;

    Object pattern = null;
    if (Refinement.class.isInstance(o)) {
      Refinement r = (Refinement)o;
      nodePatterns.add(r.getPattern());
      rn = r.getNodes();
      ri = r.getIssues();
      rc = r.getConstraints();
    }
    else if (UIRefinement.class.isInstance(o)) {
      UIRefinement r = (UIRefinement)o;
      nodePatterns.add(r.getPattern());
      rn = r.getNodes();
      ri = r.getIssues();
      rc = r.getConstraints();
    }
    else {
      Debug.noteln("GEP: Cannot get patterns from class ", 
		   o.getClass().getName());
      return;
    }

    //if (rn != null)  
    //   nodePatterns = getNodePatterns(rn.iterator(), nodePatterns);
    nodePatterns = getNodePatterns(rn, nodePatterns);
    if (ri != null)  
      issuePatterns = getIssuePatterns(ri.iterator(), issuePatterns);
    
    constrPatterns = getConstraintPatterns(rc, constrPatterns);
  }


  private void updateListPatterns(AbstractEditorPanel list, 
				  Collection patterns) {
    LinkedList pList = new LinkedList(patterns);
    //Collections.sort(pList, new UIUtil.ToStringComparator());
    list.setData(pList);
  }

  private void updateListPatterns(AbstractEditorPanel list, List patterns) {
    list.setData(patterns);
  }

  public HashSet getNodePatterns(Collection list, HashSet patterns) {
    if (list != null) {
      Iterator i = list.iterator();
      if (i != null)
	while (i.hasNext()) 
	  patterns.add(((NodeSpec)i.next()).getPattern());  
    }
    return patterns;
  }
  public HashSet getIssuePatterns(Iterator i, HashSet patterns) {
    if (i != null) 
      while (i.hasNext()) 
	patterns.add(((TaskItem)i.next()).getPattern());  
    return patterns;
  }
  public List getConstraintPatterns(Collection constraints, List sofar) {
    //ignore sofar - UIGrammar collects them.
    UIGrammar grammar = parent.getUIDomain().getGrammar();
    grammar.addGenericConstraints(constraints);
    return grammar.getConstraintPatterns();
  }


  public boolean frameActionPerformed(ActionEvent event){
    String command = event.getActionCommand();
    if (command == "newConstruct") newConstruct();
    else if (command == "editConstruct") {
      //***dummy
    }
    else if (command == "modify") saveConstruct();
    else if (command == "check") checkConstruct();
    else return false;
    return true;
  }


  /**
   * Wakes up the ActionListener with a user action.
   * This is called when a KeyStroke happens in which the ActionListener
   * registered its interest.
   */
  public void actionPerformed(ActionEvent ae){
    String command = ae.getActionCommand();
    Debug.noteln("Grammar editor: Got action command", command);
  }


  public class GrammarFormPanel extends JPanel implements ConstructEditing {

    private AConstructFramePanel parent;
    private UIObject uiConstruct;

    public GrammarFormPanel(AConstructFramePanel theParent) {
      super(new GridLayout(3, 1));
      parent = theParent;
    }

    //----------------------------Construct Editing things---------------------

    public UIDomain getUIDomain() {
      return parent.getUIDomain();
    }
    public void setConstruct(IXObject construct) {
      setUIConstruct(getUIDomain().getUIObject(construct));
    }
    public IXObject getConstruct() {
      if (uiConstruct == null) return null;
      else return uiConstruct.getBaseObject();
    }
    public void setUIConstruct(UIObject construct) {
      setUIConstructOnly(construct);
      //getModel().setObject(construct);
    }
    public void setUIConstructOnly(UIObject construct) {
      UIObject old = uiConstruct;
      uiConstruct = construct;
      //getModel().setObject(construct);
      parent.fireConstructChanged(old, construct, this);
    }
    public UIObject getUIConstruct() {
      return uiConstruct;
    }

    public void loadFromObject() {
      //displayModelData();
    }
    public void saveToObject(EditableObject eo) {
      //displayModelData();
    }

  }

  protected class GenericConstraintTableModel extends AbstractIXTableModel 
  {

    protected UIDomain uiDomain;

    public GenericConstraintTableModel(UIDomain uiDomain) { 
      super(false); //not markable
      setupColumns();
      setDomain(uiDomain);
    }
    private void setupColumns() {
      Class[] classes = {Symbol.class, Symbol.class, String.class, 
			 Object.class};
      String[] names = {"Type", "Relation", "Parameters", "Example"};
      super.setColumnClasses(classes);
      super.setColumnNames(names);
    }
    public void setDomain(UIDomain domain) {
      uiDomain = domain;
      reloadData();
    }
   
    public void reloadData() {
      UIGrammar grammar = uiDomain.getGrammar();
      List constructs = 
	new LinkedList(grammar.getAllConstraintPatterns(uiDomain));
      /*
	if (sort)
	  constructs = 
	    (List)IVUtil.sortNamedCollection(new ArrayList(constructs));   
	//Debug.noteln(" after sort:", UIUtil.listToDisplay(constructs));
      */
      if (constructs == null) clearData();
      else setData(constructs.toArray());
    }

    public Object getCellValueAt(Object o, int columnIndex) {
      if (o == null) return "";
      try {
	GenericConstraint gc = (GenericConstraint)o;
	if (columnIndex == 0) return gc.getType();
	else if (columnIndex == 1) return gc.getRelation();
	else if (columnIndex == 2) return gc.getForm();
	else if (columnIndex == 3) return gc.getExample();
	else return "";
      }
      catch (Exception e) {
	Debug.noteException(e);
	return "";
      }
    }

  }

}

/*Issues***************************************************************
 *
 * implement this?
 *
 */
/*Todos***************************************************************
 *
 *****************************************************************************
 */
