/****************************************************************************
 * An editor for specifying variable declarations
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Thu Feb  1 15:13:45 2007
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import javax.swing.table.*;       
import javax.swing.event.TableModelEvent;       
import javax.swing.event.ListSelectionEvent;       
import javax.swing.event.ListSelectionListener;       
import java.util.*;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Frame;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.lang.reflect.Array;
import ix.*;
import ix.util.*;
import ix.util.Debug;
import ix.util.lisp.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.iview.util.*;
import ix.iview.table.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.iface.ui.table.*;
import ix.iface.domain.*;
import ix.ip2.Ip2ModelManager;

/****************************************************************************
 * An editor for specifying variable declarations
 *
 * Declarations are attached to self (the refinement). <p>
 *
 * The editor is implemented as a frame using a JTextArea and JRadioButtons.<p>
 *
 * The editor can be created with no arguments or with a given refinement
 * which will be set as the current refinement whose variable declarations
 * are to be specified. Subsequently, the refinement in the editor can be set
 * using the setRefinement method.
 *****************************************************************************
 */
public class DeclarationEditor extends ConstraintEditor
  implements PropertyConstraintsListener, ListSelectionListener, 
	     UIDomainListener, DataChangeListener
{
  //relation items
  

  //protected JLabel varLabel; 
  protected VarTypeTableModel varTableMod; 
  protected VarListEditor varEd; 
  protected JRadioButton any;
  protected JRadioButton given;
  protected JRadioButton none;
  private ButtonGroup group;
  protected JButton otherEdBut;

  private JSplitPane tableSplit;
  private JPanel topPanel;
  private JPanel propTablePan;


  private ListEditorPanel propConstraintEd;
  private PropertyTableModel propTableModel;

  //itemVar - map(prop - set(Constraints))
  //private HashMap propConstraints = new HashMap();

  private HashMap varTypeMap = new HashMap(); //itemVar-Type
  //itemVar-(prop-ConstraintSet)
  private HashMap varConstMap = new HashMap(); 

  //var-constraintSet map for constraints that do not fit type
  private HashMap noObConstMap = new HashMap();

  private HashMap varTypeConstraints = new HashMap(); 


  // Keeps track of variable name changes. newVar-oldVar map ***not used
  private HashMap oldVars = new HashMap();

  private boolean saving = false;

  private ObjectProperty typeProp = new ObjectProperty();
  ix.ip2.ObjectView.ValueParser vp = new ix.ip2.ObjectView.ValueParser();

  /**
   * Creates an editor for specifying variable declarations.
   * The editor has a table for specifying type declarations, and a button
   * group for specifying "declarations" as 
   * - any variables (no constraint),
   * - none (no variables allowed)
   * - as listed
   * On start-up,
   * - original declarations are saved (in case of cancel), 
   *   as are type and property constraints.
   * - currently used variables are put into the table with their type
   *   (if any).
   * On ok-exit, the refinement is updted:
   * - declarations are 
   *   - cleared if "none" button is set
   *   - added if "only given" button is set
   * - all non-empty type constraints for variables in the table are added
   * - all property constraints are added
   */
  public DeclarationEditor(Frame owner){
    super(owner, "Variable Declaration Editor", 
	  "Please specify variable declarations, variable types," 
	  + " and property constraints");
    typeProp.setName(Symbol.intern(IVUtil.TYPE_STRING_a));
    typeProp.setSyntax(ObjectProperty.Syntax.OBJECT);
		       
    }
    
  /**
   * Creates an editor for specifying variable declarations within the given
   * refinement.
   * As DeclarationEditor(owner) but with a given refinement.
   * @param refinement the refinement whose variable declarations are to be
   * specified
   */
  public DeclarationEditor(Frame owner, UIRefinement refinement) {
    this(owner);
    setObject(refinement);
  }
  
  
  /**
   * Makes all items on the relations panel.
   * These are a table (var, type, constraints) and
   *           a button group (none, any, as given)
   */
  protected Component makeRels() {
    //make a panel that can take all the bits
    JPanel framePanel = new JPanel();
    topPanel = new JPanel(new GridBagLayout());
    propTablePan = new JPanel(new BorderLayout());

    tableSplit = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
    tableSplit.setOneTouchExpandable(true);
    tableSplit.setTopComponent(topPanel);
    tableSplit.setBottomComponent(propTablePan);
    tableSplit.setResizeWeight(0.5);
    framePanel.setLayout(new BorderLayout());
    framePanel.add(tableSplit, BorderLayout.CENTER);

    JLabel varLabel = new JLabel("Variable declarations");
    makeDeclarationButtons(); //sets any/none/given and their group


    varTableMod = new VarTypeTableModel();
    IXTable varTable = new IXTable(varTableMod);
    varTable.setPreferredScrollableViewportSize(new Dimension(500, 120));
    TableColumn col = 
      varTable.getColumnModel().getColumn(varTableMod.TYPE_COL);
    col.setCellRenderer(new NamedRenderer(false));
    typeBox = new JComboBox();
    typeBox.addItem(" "); //make an empty one
    DefaultCellEditor typeCellEditor = new DefaultCellEditor(typeBox);
    col.setCellEditor(typeCellEditor);
    col = varTable.getColumnModel().getColumn(varTableMod.CONSTRAINTS_COL);
    varTable.setToolTipText(varTableMod.CONSTRAINTS_COL, 
			    "Has property constraints?");
    col.setResizable(false);
    col.setPreferredWidth(35);
    col.setMaxWidth(35);
    col.setMinWidth(35);
    col = varTable.getColumnModel().getColumn(varTableMod.NOCONST_COL);
    varTable.setToolTipText(varTableMod.NOCONST_COL, 
			    "Has constraints outside type?");
    col.setResizable(false);
    col.setPreferredWidth(35);
    col.setMaxWidth(35);
    col.setMinWidth(35);
    varTable.doLayout();

    varTabPopup = new JObjectPopup();
    propTabPopup = new JObjectPopup();
    popListener = new ActionListener() {
	public void actionPerformed(ActionEvent ae) {
	  try {
	    Component parent = ((Component)ae.getSource()).getParent();
	    String command = ae.getActionCommand();
	    if (parent instanceof JObjectPopup) {
	      Object o = ((JObjectPopup)parent).getObject();
	      if (o == null) {
		String[] message = {"Please select a row to " + command};
		JOptionPane.showMessageDialog(null, message);
	      }
	      else if (o instanceof ItemVar) {
		DeclarationEditor.this.userShowNoDecs((ItemVar)o);
	      }
	      else if (o instanceof Constraint) {
		if (IVUtil.isEmptyConstraint((Constraint)o)) {
		  String[] message = {"Selected row is empty. ",
				      command + " cannot be done."};
		  JOptionPane.showMessageDialog(null, message);
		}
		else DeclarationEditor.this.removeConstraint((Constraint)o);
	      }
	    }
	  }
	  catch (Exception e) {
	    Debug.noteException(e);
	    String[] message = {"Problem during " +ae.getActionCommand()+":",
				Debug.foldException(e)};
	    JOptionPane.showMessageDialog(null, message);
	  }
	}
      };

    JMenuItem mi;
    mi = new JMenuItem("Show constraints outside type");
    mi.addActionListener(popListener);
    varTabPopup.add(mi);
    mi = new JMenuItem("Delete selected constraint");
    mi.addActionListener(popListener);
    propTabPopup.add(mi);

    varEd = new VarListEditor(this,"Variables and Types", varTable);

    propConstraintEd = makePropPanel();

    otherEdBut = new JButton("Use Simple Constraint Editor");
    otherEdBut.addMouseListener(this);

    //add it all to the panel
    GridBagConstraints c;
    c = new GridBagConstraints(0, 1, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,12),0,0); //insets and padding
    topPanel.add(varLabel, c);
    c = new GridBagConstraints(1, 1, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    topPanel.add(any, c);
    c = new GridBagConstraints(2, 1, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    topPanel.add(given, c);
    c = new GridBagConstraints(3, 1, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    topPanel.add(none, c);

    c = new GridBagConstraints(0, 2, //x,y
			       GridBagConstraints.REMAINDER, //width 
			       1, //height
			       1.0,1.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.BOTH,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    topPanel.add(varEd, c);
    /*
    c = new GridBagConstraints(0, 3, //x,y
			       GridBagConstraints.REMAINDER, //width 
			       1, //height
			       1.0,1.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.BOTH,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    */
    propTablePan.add(propConstraintEd, BorderLayout.CENTER);
    propTablePan.add(otherEdBut, BorderLayout.SOUTH);
    return framePanel;
  }


  private ListEditorPanel makePropPanel() {
    propTableModel = new PropertyTableModel(); //not markable
    IXTable table = new IXTable(propTableModel);
    propTableModel.adjustTable(table);
    table.addFocusListener(propTableModel);

    propConstraintEd = new ListEditorPanel(this,"Property Constraints", table);
    //propTableModel.setObjectClass(objectClass);
    propTableModel.addPropertyConstraintsListener(this);
    propConstraintEd.removeListSelectionListener(this); //not interested
    return propConstraintEd;
  }

  //Makes the buttons and their group and sets any/none/given variables
  private void makeDeclarationButtons() {
    any = new JRadioButton("Any");
    none = new JRadioButton("None");
    given = new JRadioButton("As Given Here");
    group = new ButtonGroup();
    group.add(any);
    group.add(none);
    group.add(given);
    any.setActionCommand("Any");
    any.setToolTipText("Allow any variables");
    none.setActionCommand("None");
    none.setToolTipText("Allow no variables");
    given.setActionCommand("As Given Here");
    given.setToolTipText("Allow only given variables");

    any.addMouseListener(this);
    none.addMouseListener(this);
    given.addMouseListener(this);
    any.setSelected(true);
  }

  /**
   * Sets the refinement whose variables are to be edited
   * Fills the current declarations (if any) into the list and selects
   * the appropriate radio button.
   * Perhaps this should really be done in setFromObject(varDecs), but we
   * only ever edit the refinement's variable declarations rather than adding
   * a declaration one at a time, so we do it here.
   */
  protected void initFromObject() throws NullPointerException {
    propConstraintEd.setLabel("Property Constraints");
    propTableModel.clear();
    UIRefinement refinement = (UIRefinement) currentObject;
    if (refinement == null) return;    
    //if (uiDomain != refinement.getUIDomain()) //better!do this when listening
    refinement.addDataChangeListener(this);
    if (refinement.getUIDomain() != null) {
      refinement.getUIDomain().addUIDomainListener(this);
      loadObjectClasses(refinement.getUIDomain());
    }

    Set varDecs = new HashSet(); //set of itemVars (used and declared)
    List decs = refinement.getVariableDeclarations();
    SortedSet vars = refinement.getVariablesUsed();
    //Debug.noteln("DecEd: decs", UIUtil.show(decs));
    //Debug.noteln(" vars", UIUtil.show(vars));
    if (vars != null) varDecs = new HashSet(vars);
    //use all declared and all used variables
    boolean match = true; //false if a mismatch is found in vars vs decs
    if (decs != null) 
      for (Iterator i = decs.iterator(); i.hasNext();) {
	VariableDeclaration dec = (VariableDeclaration)i.next();
	if (match) match = !varDecs.add(dec.getName());
      }

    //this also loads object constraints for uir
    setListData(refinement, varDecs); 

    if (decs == null) {
	//Debug.noteln("DecE: got no declarations");
      any.setSelected(true);
    }
    else if (decs.size() == 0) {
      //Debug.noteln("DecE: no vars allowed");
      none.setSelected(true);
    }
    else {
      //there are some declarations, so should be only given!
      //set to given even if mis-match
      given.setSelected(true);
      //Debug.noteln("DecE: vars allowed:", UIUtil.listToDisplay(varDecs));
    }

    
  }

  /**
   * Sets the data in the variable-type table
   * Handles lists or sets of ItemVar
   */
  public void setListData(UIRefinement refinement, Collection data) {
    //Debug.noteln("DecE: setting list data", UIUtil.listToDisplay(data));
    varEd.removeData();

    //load the object constraints into the editor and save them for Cancel
    loadObjectConstraints(refinement);

    //note the values
    if (data != null) varEd.setData(refinement, data);
  }

  /**
   * We do editing of all variable declarations from initFromObject so this is
   * obsolete.
   */
  protected void setFromObject(Object original) {
  }

  /**----------------Managing notes------------------------------------
   * load: uir into map         * save: map into uir
   * note: into map             * get: from map
   * for 
   *   var type (varTypeMap)
   *   var type constraint (varTypeConstraints)
   *   var constraints (varConstraints) and no-type var constrs (noObConstMap)
   */

  /**
   * Looks up the type constraint of the given var in the given refinement.
   * puts a note into varTypeMap and notes the constraint used.
   */
  protected void loadVarType(UIRefinement uir, ItemVar var) {
    if ((var == null) || (uir == null)) return;
    //Debug.noteln("DecEd: loading var type", var); 
    Constraint c = IVUtil.findClassConstraint(var, uir.getConstraints());
    if (c != null) {
      //Debug.noteln(" type constraint", c); 
      Object type = IVUtil.extractType(c, uir.getUIDomain());
      //Debug.noteln(" type class", type.getClass()); 
      if ((type != null) && (!type.toString().equals("")))
	varTypeMap.put(var, type);
      varTypeConstraints.put(var, c);
    }
  }
  /**
   * Saves the var type (class) in the varTypeMap into the refinement.
   * Make type constraint (if necessary), delete previous in uir
   * (checking for varName change) set constraint in uir. 
   */
  protected void saveVarType(UIRefinement uir, ItemVar var) {
    if (var == null) return;
    //make constraint
    Constraint c = getVarTypeConstraint(var);
    if (c == null) c = makeTypeConstraint(var);
    //delete previous constraint (checking for var name change)
    ItemVar oldVar = getOldVar(var);
    if (oldVar != null) 
      uir.setTypeConstraint(oldVar, null); //remove
    //set constraint
    uir.setTypeConstraint(var, c); //replaces old with same var
  }
  /**
   * Notes the given type for the given variable in the varTypeMap.
   * Goes through all variable constraints to filter out ones that don't 
   * fit the new type. Also updates propTableModel.
   */
  protected void noteVarType(ItemVar var, UIObjectClass type) {
    if (var == null) return;
    if (type != null) type.addDataChangeListener(this);
    varTypeMap.put(var, type);
    reloadVarConstraints(var);
    propTableModel.setConstraints(var, type, getVarConstraints(var));
    propConstraintEd.setLabel("Property Constraints for variable " 
			      + var.toString());
  }
  /**
   * Looks up the var type for the given variable in the varTypeMap
   * @return the type of the given variable or null if none noted
   */
  protected UIObjectClass getVarType(ItemVar var) {
    Object oType = varTypeMap.get(var);
    if (oType instanceof UIObjectClass) return (UIObjectClass)oType;
    else if (oType == null) return null;
    else {
      String typeName = oType.toString();
      UIDomain dom = ((UIRefinement)currentObject).getUIDomain();
      if (dom != null) {
        Object oc = dom.getNamedObjectClass(typeName);
	UIObjectClass type=null;
	if (oc instanceof UIObjectClass)
	  type = (UIObjectClass)oc;
	else Debug.noteln("DeclarationEditor found class", oc);
	if (type != null) {
	  varTypeMap.put(var, type);
	  return type;
	}
      }
      //no UIOC and cannot find one, so give up
      Debug.noteln("DecEd: var type is of class", oType.getClass());
      return null;
    }
  }
  protected void removeVarType(ItemVar var) {
    varTypeMap.remove(var);
  }
  /**
   * Looks up the var type for the given variable in the varTypeMap and 
   * makes a constraint that sets it.
   * @return a constraint that sets the variable type or null if none noted
   */
  protected Constraint getVarTypeConstraint(ItemVar var) {
    return (Constraint)varTypeConstraints.get(var);
  }
  /**
   * Notes the var type for the given variable in the varTypeConstraints map
   */
  protected void noteVarTypeConstraint(ItemVar var, Constraint c) {
    if (var == null) return;
    varTypeConstraints.put(var, c);
  }
  /**
   * Notes the var type for the given variable in the varTypeConstraints map
   */
  protected void noteVarTypeConstraint(ItemVar var, UIObjectClass oClass) {
    if (var == null) return;
    Constraint c = makeTypeConstraint(var);
    varTypeConstraints.put(var, c);
  }


  /**
   * Makes a type constraint for the given variable from the current settings.
   */
  protected Constraint makeTypeConstraint(ItemVar var) {
    if (var == null) return null;
    Object oType = varTypeMap.get(var);
    //Debug.noteln("DecEd: noted type is", oType);
    UIObjectClass oClass = null;
    if (oType instanceof String) {
      //Debug.noteln(" type is a string");
      UIDomain dom = ((UIRefinement)currentObject).getUIDomain();
      if (dom != null) 
        oClass = (UIObjectClass)dom.getNamedObjectClass((String)oType);
      if (oClass == null)
	  JOptionPane.showMessageDialog(null, "Unknown variable type: "
					  + oType);
    }
    else if (oType instanceof UIObjectClass)
      oClass = (UIObjectClass)oType;
    //else Debug.noteln("DecEd: noted type is of class", oType.getClass());
    if (oClass == null) return null;

    LList pat = Lisp.list(Symbol.intern(IVUtil.TYPE_STRING_a), var); 
    Object cType = vp.read(typeProp, oClass.getName(), null);
    Debug.noteln("DecEd: type syntax is", cType.getClass());
    PatternAssignment pa = new PatternAssignment(pat, cType);
    Constraint newC = new Constraint(Refinement.S_WORLD_STATE, 
				     Refinement.S_CONDITION,
				     Lisp.list(pa));
    return newC;
  }

  /**
   * Loads the constraints of the refinement into the varConstMap 
   */
  protected void loadVarConstraints(UIRefinement uir, ItemVar var) {
    UIObjectClass varType = getVarType(var);
    List vcs = uir.getVariableConstraints(var);
    HashMap propConstMap = constraintsToProperty(var, varType, vcs);
    //Debug.noteln(" var constraints", UIUtil.show(propConstMap.getValues())); 
    if ((vcs != null) && !vcs.isEmpty())
      noteVarConstraints(var, propConstMap);
  }
  /**
   * Checks applicability of editor constraints and noObCons for all vars.
   * do this when object system changed
   */
  protected void reloadAllVarConstraints() {
    Collection vars = varConstMap.keySet();
    for (Iterator i = vars.iterator(); i.hasNext(); ) 
      reloadVarConstraints((ItemVar)i.next());
  }
  /**
   * Checks applicability of editor constraints and noObCons 
   */
  protected void reloadVarConstraints(ItemVar var) {
    UIObjectClass varType = getVarType(var);
    Object noNote = noObConstMap.get(var);
    noObConstMap.remove(var);
    Set vcs;
    if (noNote != null) 
      vcs = new HashSet((Collection)noNote);
    else vcs = new HashSet();
    List currentCs = UIUtil.getFlatValues(getVarConstraints(var));
    if ((currentCs != null) && !currentCs.isEmpty()) vcs.addAll(currentCs);
    HashMap propConstMap = constraintsToProperty(var, varType, vcs);
    //Debug.noteln(" var constraints", UIUtil.show(propConstMap.getValues())); 
    if ((vcs != null) && !vcs.isEmpty())
      noteVarConstraints(var, propConstMap);
  }
  /**
   * Saves the constraints in the varConstMap into the refinement
   */
  protected void saveVarConstraints(UIRefinement uir, ItemVar var) {
    //check whether the varname has changed. If so, remove refinement's
    //notes for the old var, if not remove this var's constraints
    //Then add the noted constraints to the refinement.
    ItemVar oldVar = getOldVar(var);
    if (oldVar != null) 
	uir.setVariableConstraints(oldVar, new ArrayList()); //remove old
    List constraints = UIUtil.getFlatValues((HashMap)varConstMap.get(var));
    Set noConstraints = getNoObjectConstraints(var);
    if (constraints == null) constraints = new ArrayList();
    if (noConstraints != null) constraints.addAll(noConstraints);
    uir.setVariableConstraints(var, constraints); //replaces old with same var
  }
  /**
   * Notes the given constraints for the given variable in the varConstMap.
   * Constraints are in a properties HashMap: (property-constraintSet) 
   */
  protected void noteVarConstraints(ItemVar var, HashMap propConsts) {
    //Debug.noteln("DecEd: noting constraints for var:", var);
    //Debug.noteln("  :", propConsts);
    //(new Throwable()).printStackTrace();
    if (var != null) 
      varConstMap.put(var, propConsts);
    //Debug.noteln("  varConstMap now", varConstMap);
  }
  /**
   * Looks up the constraints for the given variable in the varConstMap
   * @return the propertiesMap (property-constraintSet) of the given
   * variable or null if none noted. 
   */
  protected HashMap getVarConstraints(ItemVar var) {
    if (var == null) return null;
    return (HashMap)varConstMap.get(var);
  }
  protected void removeVarConstraints(ItemVar var) {
    varConstMap.remove(var);
  }
  protected void removeVarTypeConstraint(ItemVar var) {
    varTypeConstraints.remove(var);
  }


  /**
   * Remove the constraint; 
   * @return true if removed, false if constraint not found.
   */
  protected boolean removeConstraint(Constraint c) {
    if (c == null) return false;
    return removeConstraint(IVUtil.variableFromConstraint(c), c);
  }
  /**
   * Remove the constraint; 
   * @return true if removed, false if constraint not found.
   */
  protected boolean removeConstraint(ItemVar var, Constraint c) {
    if (var != null) return removeConstraint(var, getVarType(var), c);
    //else //look in all variables, find it. perhaps assume current var?
    Debug.noteln("DecEd: remove constraint - no var in", c);
    return false;    
  }
  /**
   * Remove the constraint; 
   * @return true if removed, false if constraint not found.
   */
  protected boolean removeConstraint(ItemVar var, UIObjectClass type, 
				     Constraint c) {
    ObjectProperty prop = null;
    if (type != null) 
      return removeConstraint(var, type, 
			      IVUtil.propertyFromConstraint(type, c), c);
    /*
    else {
      String sProp = IVUtil.propertyFromConstraint(c);
      prop = propertyFromConstraint(type, c);
    }
    */
    Debug.noteln("DecEd: remove constraint - no type in", c);
    return false;
  }

  /**
   * Remove the constraint; 
   * @return true if removed, false if constraint not found.
   */
  protected boolean removeConstraint(ItemVar var, UIObjectClass type, 
				     ObjectProperty prop, Constraint c) {
    if ((var == null) || (type == null) || (c == null)) return false;
    boolean ok = false;
    if (prop != null) {
      HashMap varConstraints = getVarConstraints(var);
      if (varConstraints != null) {
	Object propCs = varConstraints.get(prop);
	if (propCs != null) {
	  Set propConstraints = (Set)propCs;
	  int size = propConstraints.size();
	  Set newConstraints = IVUtil.removeConstraint(propConstraints, c);
	  if (newConstraints == null) ok = (size == 1);
	  else ok = (size > newConstraints.size());
	  if (ok) //changed, so note just in case new set is not old one
	    varConstraints.put(prop, newConstraints);
	}
      }
    }
    /*
    else {
    }
    */
    else Debug.noteln("DecEd: remove constraint - no prop in", c);

    if (ok) //change has been made, so tell property table
      propTableModel.setConstraints(var, type, getVarConstraints(var));
    return ok;
  }

    /** *********not used yet**********/
  private ItemVar getOldVar(ItemVar var) {
    return (ItemVar)oldVars.get(var);
  }
  private void noteOldVar(ItemVar var, ItemVar oldVar) {
    Object oldOld = oldVars.get(oldVar);
    if (oldOld != null) {//this var has changed previously. Remember original
      oldVars.remove(oldVar);
      oldVars.put(var, oldOld);
    }
    else oldVars.put(var, oldVar);
  }

  /**
   * If variable name has changed, replace note of type and constraints
   * and note that the change was made (to clean up uir on exit)
   * 
   */
  protected void replaceNotes(ItemVar oldVar, ItemVar newVar) {
    //Debug.noteln("DecEd: changing var name notes", oldVar +" to "+ newVar);
    Object oldO = getVarType(oldVar);
    if (oldO != null) {
      removeVarType(oldVar);
      noteVarType(newVar, (UIObjectClass)oldO);
    }
    HashMap oldCs = getVarConstraints(oldVar);
    if (oldCs != null) {
      removeVarConstraints(oldVar);
      HashMap newCs = IVUtil.replaceVarInConstraints(oldVar, newVar, oldCs);
      noteVarConstraints(newVar, oldCs);
      propTableModel.setConstraints(newVar, (UIObjectClass)oldO, oldCs);
    }
    oldO = varTypeConstraints.get(oldVar);
    if (oldO != null) {
      varTypeConstraints.remove(oldVar);
      varTypeConstraints.put(newVar, oldO);
    }
  }
	  


  /**
   * Note the constraints for the given variable according to the property
   * that is being constrained.
   * Can handle all null parameters.
   * @return a map of property - {constraints} entries
   */
  private HashMap constraintsToProperty(ItemVar var, UIObjectClass oClass,
					Collection constrs) {
    if (var == null) return null;
    //Debug.noteln("DecEd: mapping constraints to property for var", var);
    HashMap varPropConstraints = new HashMap();

    if (constrs == null) return varPropConstraints;

    for (Iterator i = constrs.iterator(); i.hasNext(); ) {
      try { 
	Constraint c = (Constraint)i.next();
	PatternAssignment ass = c.getPatternAssignment();
	if (ass != null) {
	  ObjectProperty prop = 
	      IVUtil.propertyFromPattern(ass.getPattern(), oClass);
	  //Debug.noteln(" DecEd: got property", prop);
	  if ((prop != null) 
	      && IVUtil.isSubjectInPattern(var, ass.getPattern())
	      && IVUtil.isSimplePropertyAssignment(ass, prop, var))
	    varPropConstraints = 
	      propTableModel.noteProperty(varPropConstraints, oClass, prop, c);
	  else if (!IVUtil.isClassConstraint(var, c)) {
	    Debug.noteln("DecEd: not a property constraint:",ass.getPattern());
	    noteNoObjectConstraint(var, c);
	  }
	}
      } catch (Exception e) {} //only interested in constraints
    }
    return varPropConstraints;
  }

  /**
   * Note those variable constraints that do not fit the object system
   * or are of the wrong kind
   */
  private void noteNoObjectConstraint(ItemVar var, Constraint constr) {
    if ((var == null) || (constr == null)) return;
    Object note = noObConstMap.get(var);
    Set noSet = new HashSet();
    if ((note != null) && (note instanceof Set))
      noSet = (Set)note;
    noSet.add(constr);
    noObConstMap.put(var, noSet);
  }
    
  protected Set getNoObjectConstraints(ItemVar var) {
    return (Set)noObConstMap.get(var);
  }

  /**
   * Reads the given declarations and type and property constraints and updates
   * the current refinement.
   * Declarations any: set to null; none: set to empty; as given: see constrs
   * constraints: skip if "none"; go through table: any non-empty types, 
   *   add constraint; if declarations to be given, make and collect.
   */
  protected Object collectConstraint() {
    //Debug.noteln("DecEd: collecting constraint");
    UIRefinement uir = (UIRefinement)currentObject;

    saving = true;

    updateRefinementConstraints(uir);

    //get the selection.
    ButtonModel bm = group.getSelection();
    //Debug.noteln("DecE: selection is", bm.getActionCommand());
    if ((bm.getActionCommand() == null) ||
	bm.getActionCommand().equals("Any")) {
      setAllConstraints((UIRefinement)currentObject);
      uir.setVariableDeclarations(null);
      saving = false;
      return "any";
    }
    else if (bm.getActionCommand().equals("None")) {
      List decs = new ArrayList();
      uir.setVariableDeclarations(decs);
      saving = false;
      return decs;
    }
    else if (bm.getActionCommand().equals("Given") ||
	     bm.getActionCommand().equals("As Given Here")) {
      List varDecs = new ArrayList(setAllConstraints(uir));
      //Debug.noteln("DecEd: var decs are", UIUtil.show(varDecs));
      uir.setVariableDeclarations(varDecs);
      saving = false;
      return varDecs;
    }
    uir.setVariableDeclarations(null);

    saving = false;
    return "any"; //***should not get here!
  }

  private void updateRefinementConstraints(UIRefinement uir) {
    //Debug.noteln("DecEd: updating refinement constraints");
    if (uir == null) return;

    //Debug.noteln(" setting constraints");
    Set keys = varConstMap.keySet();  
    if (keys != null) {
      for (Iterator i = keys.iterator(); i.hasNext(); ) {
	ItemVar var = (ItemVar)i.next();
	saveVarConstraints(uir, var);
      }
    }
  }


  /**
   * Gets variable type and property constraints from the given refinement, 
   * loads them into the editor and keeps a copy in case of cancel.
   */
  protected void loadObjectConstraints(UIRefinement refinement) {
    varConstMap.clear();
    varTypeMap.clear();
    varTypeConstraints.clear();

    //get all variables used
    Set vars = refinement.getVariablesUsed();
    if (vars == null) return; // no vars, so done.
    
    //for each variable, 
    for (Iterator i = vars.iterator(); i.hasNext(); ) {
      ItemVar var = (ItemVar)i.next();
      //find out the variable type and constraints and note them all
      loadVarType(refinement, var);
      loadVarConstraints(refinement, var);
      //get the table to note it
    }
  }


  /**
   * Goes through the variable type constraints in the var-type table and
   * sets them all in the refinement. Also collects a set of declarations
   * and returns them.
   */
  protected Set setAllConstraints(UIRefinement refinement) {
    Set declarations = new HashSet();
    List constraints = new ArrayList();
    Collection tableVars = Arrays.asList((Object[])varEd.getData());
    if (tableVars != null) {
      for (Iterator i = tableVars.iterator(); i.hasNext(); ) {
	Object next = i.next();
	VariableDeclaration varDec = null;
	if (next instanceof ItemVar) 
	  varDec = new VariableDeclaration((ItemVar)next);
	else if (next instanceof VariableDeclaration)
	  varDec = (VariableDeclaration)next;
	else {
	  try {
	    varDec = 
	      new VariableDeclaration((ItemVar)Symbol.intern(next.toString()));
	  }
	  catch (Exception e) {}
	}
	if (varDec != null) {
	  declarations.add(varDec);
	  constraints.add(makeTypeConstraint(varDec.getName()));
	}
      }
    }
    refinement.setTypeConstraints(constraints);
    return declarations;
  }


  private JComboBox typeBox;
  private UIDomain uiDomain;
  private void loadObjectClasses(UIDomain domain) {
    if (domain == null) return;
    uiDomain = domain;
    List classes = domain.getObjectClassTree();
    typeBox.removeAllItems();
    if (classes != null) {
      for (Iterator i = classes.iterator(); i.hasNext(); ) {
	typeBox.addItem(i.next());
      }
      typeBox.addItem(""); //make an empty one
    }
    if (propTableModel != null) propTableModel.updatePropertyBox();
  }


  /**
   * This is not necessary - declarations are handled explicitly on exit
   * Note any variable declarations set in this editor.
   */
  protected void handleVariables() {
  }

  protected Object noteNewValue(Object constraint) {
    //Debug.noteln("DecEd: constraint is ", constraint);
    try {
      //lookup what was noted (varDecs have been cleared!)
      UIRefinement uiRef = (UIRefinement)currentObject;
      List decs = uiRef.getVariableDeclarations();
      //Debug.noteln("DecEd: noteValue returning", decs);
      return decs;
    }
    catch (Exception e) {
      Debug.noteException(e);
      return constraint;
    }
  }

  public void userShowNoDecs(ItemVar var) {
    if (var == null) return;
    Object nos = noObConstMap.get(var);    
    if ((nos == null) || ((Collection)nos).isEmpty())
      JOptionPane.showMessageDialog(this, "All of the constraints for "
				    +var+ " fit its class");
    else {
      Collection noDecs = (Collection)nos;
      String[] msg = 
	(String[])Array.newInstance(String.class, noDecs.size()+1);
      String[] consts = UIUtil.listToTextArray(noDecs);
      msg[0] = "The constraints for "+var+" that do not fit its class are:";
      System.arraycopy(consts, 0, msg, 1, noDecs.size());
      JOptionPane.showMessageDialog(this, msg);
    }
  }

  protected void finishEdits() {
    try {
      JTable table = (JTable)propConstraintEd.getItemComponent();
      CellEditor ed;
      if (table.isEditing()) {
        //Debug.noteln("DecEd: finishing edits of prop table");
	ed = table.getCellEditor();
	ed.stopCellEditing();
      }
      table = (JTable)varEd.getItemComponent();
      if (table.isEditing()) {
	//Debug.noteln("DecEd: finishing edits of var table");
	ed = table.getCellEditor();
	ed.stopCellEditing();
      }
    }
    catch (Exception e) { Debug.noteException(e);}
      
  }
  

  public void start(String title, Object uir, Point location) {
    setTitle(title);
    super.start(uir, location);
  }

  public void start() {
    //moreBut.setVisible(false);
    //moreBut.setEnabled(false);
    moreBut.setText("Note Edits");
    super.start();
  }

  public void closeEditor() {
    if (currentObject != null) 
      ((EditableObject)currentObject).removeDataChangeListener(this);
    if (conditionEditor != null) conditionEditor.setVisible(false);
    super.closeEditor();
  }

  public ConditionEffectEditor conditionEditor;
  public void ensureConditionEditor(){
    if (conditionEditor == null){
      conditionEditor = new ConditionEffectEditor((Frame)getOwner());
      conditionEditor.setVisible(false);
    } 
  }

  //---------------------------Listener things-------------------------------

  /**
   * mouse clicked
   * - radio button "none": warn about clearing declarations
   * - button "Ok": finish editing, collect constraint, and exit
   * - button "Other Editor": use simple cond/eff editor
   * - button "Note Edits": put constraints into refinement
   */
  public void mouseClicked(MouseEvent e){
    //Debug.noteln("DecEd: source of click is", e.getSource());
    if (e.getSource().equals(this.none)) {
      //warn the user then clear the declarations and the constraints
      Set decs = ((UIRefinement)currentObject).getVariablesUsed();
      if ((decs != null) && (decs.size() != 0)) {
	//warn the user
	String[] message = {"This will clear all declarations and constraints",
			    "in this editor.", 
			    "Are you sure you want to do this?"};
	int ok = JOptionPane.showConfirmDialog(this, message, "Warning", 
					       JOptionPane.OK_CANCEL_OPTION);
	if (ok == JOptionPane.OK_OPTION)
	  varEd.removeData();
	else this.any.setSelected(true);
      }
    }
    else if (e.getSource().equals(super.okBut)) {
      //make sure last edits are kept
      finishEdits();
      //collect the rest of the data and update UIR
      propTableModel.noteOldConstraints();
      processConstraint();
      closeEditor(); 
      return;
    }
    else if (e.getSource().equals(otherEdBut)) {
      //Debug.noteln("Adding free-form condition or effect for refinement",
      //		   currentObject);
      ensureConditionEditor();
      conditionEditor.start(currentObject);

    }
    else if (e.getSource().equals(super.moreBut)) {
      //make sure last edits are kept, collect prop constraints, updateUIR
      finishEdits();
      propTableModel.noteOldConstraints();
      processConstraint(); 
      //stay up!
    }
    else super.mouseClicked(e);
  }
  public void mousePressed(MouseEvent me){
    //Debug.noteln("DecEd: got mouse pressed", me);
    if (SwingUtilities.isRightMouseButton(me)) {
      Component table = me.getComponent();
      if (table instanceof IXTable) {
        TableModel tabMod = ((IXTable)table).getModel();
	if (tabMod instanceof VarTypeTableModel)
	  showTablePopup(varTabPopup, me);
	else if (tabMod instanceof PropertyTableModel)
	  showTablePopup(propTabPopup, me);
      }      
    }
  }
  private ActionListener popListener;
  private JObjectPopup varTabPopup;
  private JObjectPopup propTabPopup;
  private void showTablePopup(JObjectPopup pop, MouseEvent me) {
    Component c = me.getComponent();
    if (c instanceof IXTable) {
      IXTable table = (IXTable)c;
      Object o = table.getSelectedObject();
      pop.setObject(o);
      pop.show(me.getComponent(), me.getX(), me.getY());
    }
    else 
      Debug.noteln("DecEd: WARNING show popup from odd component!", c);
  }

  public void valueChanged(ListSelectionEvent e) {
    if (e.getValueIsAdjusting()) return;
    //Debug.noteln("DecEd: got valueChanged event", e);
    //Debug.noteln(" source is:", e.getSource());
    //Debug.noteln("  varConstMap now", varConstMap);
    finishEdits();
    ListSelectionModel mod = 
      ((IXTable)varEd.getItemComponent()).getSelectionModel();
    //Debug.noteln(" var selection model is:", mod);
    if (e.getSource() == mod) { //value change in varTypeTable
      ItemVar currentVar = (ItemVar)varEd.getSelectedObject();
      //propTableModel.clear();
      if (currentVar == null) {
	propConstraintEd.setLabel("Property Constraints");
	propTableModel.clear();
	return;
      }
      //Debug.noteln("DecEd: selected item var", currentVar);
      UIObjectClass type = getVarType(currentVar);
      
      //hack! constrs is either hashMap or collection
      //Debug.noteln("  varConstMap now", varConstMap);
      Object constrs = getVarConstraints(currentVar);
      //Debug.noteln("Setting constraints in propTM", constrs);
      HashMap constraints;
      if (constrs instanceof HashMap) constraints = (HashMap)constrs;
      else constraints = 
	     constraintsToProperty(currentVar, type, (Collection)constrs);
      noteVarConstraints(currentVar, constraints);
      propConstraintEd.setLabel("Property Constraints for variable " 
				+ currentVar.toString());
      propTableModel.setConstraints(currentVar, type, constraints);
    }
  }


  public void dataChanged(EditableObject object, String field, 
			  Object oldValue, Object newValue) {
    if (!isVisible() || (object == null)) return;
    //super.dataChanged(object, field, oldValue, newValue);
    //Debug.noteln("DecEd:EVENT got data change", object);
    if (object instanceof UIObjectClass) {//classes changed
      updatePropertiesList(((UIRefinement)currentObject).getUIDomain());
      reloadAllVarConstraints();
      updateTables();
    }
    else if (object.equals(currentObject))
      updateData();
  }

  public void constructAdded(UIDomainEvent se) {
    if (isVisible() && (se.getObject() instanceof UIObjectClass))
      updateClassesList(se);
  }
  public void constructAdded(DomainEvent se) {
    //wait until UIDomain notices
  }
  public void constructEdited(UIDomainEvent se) {
    if (!isVisible()) return;
    //Debug.noteln("DecEd:EVENT construct edited", se.getObject());
    if (se.getObject() instanceof UIObjectClass) {//classes changed
      updatePropertiesList(se);
      reloadAllVarConstraints();
      updateTables();
    }
    else if ((se.getObject() instanceof UIRefinement) //current refinement
	     && (currentObject != null)
	     && currentObject.equals(se.getObject()))
      updateData();
   }
  public void constructRemoved(UIDomainEvent se) {
    if (!isVisible()) return;
    else if (se.getObject() instanceof UIObjectClass) {
      updateClassesList(se);
      String[] message = {"Object class removed: " + se.getObject(),
			  "Please ensure no variable declarations use it",
			  "(No automatic recovery yet)"};
      JOptionPane.showMessageDialog(this, message);
    }
    else if ((se.getObject() instanceof UIRefinement) //current refinement
	     && (currentObject != null)
	     && currentObject.equals(se.getObject())) {
      String[] message = {"Current refinement removed: " + se.getObject(),
			  "Closing editor."};
      JOptionPane.showMessageDialog(this, message);
      closeEditor(); 
    }
  }
  public void domainCleared(UIDomainEvent se) {
    if (!isVisible()) return;
    else {
      updateClassesList(se);
      closeEditor();
    }
  }
  public void domainEdited(UIDomainEvent se) {
    if (!isVisible()) return;
    //Debug.noteln("DecEd:EVENT got domain edited", se.getObject());
    updateClassesList(se);
    updateData();
  }
  public void domainSet(UIDomainEvent se) {
    if (!isVisible()) return;
    else {
      updateClassesList(se);
      updateData();
    }
  }
  

  private void updateTables() {
    finishEdits();
    //varTableMod.updateConstCols();
  }

  private void updateClassesList(UIDomainEvent se) {
    loadObjectClasses(se.getDomain());
    updatePropertiesList(se);
  }
  private void updatePropertiesList(UIDomainEvent se) {
    updatePropertiesList(se.getDomain());
  }
  private void updatePropertiesList(UIDomain domain) {
    propTableModel.updatePropertyBox();
    loadObjectClasses(domain);
  }

  /** 
   * Called when changes (may) have been made to the refinements in the
   * UIDomain. Reload the object data into the tables and the declarations.
   * NOTE: make sure to keep edits too!!!
   */
  private void updateData() {
    if (saving || !isVisible()) return; //closed or I'm updating! ignore  
    //Debug.noteln("DecEd: updating object data");
    /*
    String[] message = {"The domain changed, reloading information.",
			"Note: you will lose changes you made in this editor"};
    JOptionPane.showMessageDialog(this, message);
    */
    initFromObject();
  }

  public void variableConstraintsChanged(ItemVar variable, 
					 HashMap propConsts) {
    //Debug.noteln("DecEd: varConstraints changed", variable);
    noteVarConstraints(variable, propConsts);
  }
  public void variableNameChanged(ItemVar oldVar, ItemVar newVar) {
    Debug.noteln("DecEd: **********Cannot deal with var name change*********");
  }
  //should do some checking and filtering?
  public void variableTypeChanged(ItemVar var, UIObjectClass oldType, 
				  UIObjectClass newType) {
    noteVarType(var, newType);
  }

  //---------------------------Classes---------------------------------


  /**
   * A list editor panel that will find type constraints for the data variables.
   *******************************************/
  public class VarListEditor extends ListEditorPanel {
    public VarListEditor(MouseListener ml, String label, JComponent c) {
      super(ml, label, c);
    }
    public VarListEditor() {super();}

    public void setData(UIRefinement uir, Collection itemVars) {
      IXTable table = (IXTable)getItemComponent();
      VarTypeTableModel model = (VarTypeTableModel)table.getModel();
      model.setData(uir, itemVars);      
    }

  }

  /**
   * A table model for ItemVar-Type pairs
   *******************************************/
  public class VarTypeTableModel extends AbstractIXTableModel {
    String[] columns = {"Variable", "Type (Object Class)", "Cs", "OCs"};
    Class[] classes = {String.class, ObjectClass.class, 
		       Boolean.class, Boolean.class};
    private int VAR_COL = 0;
    private int TYPE_COL = 1;
    private int CONSTRAINTS_COL = 2;
    private int NOCONST_COL = 3;
    UIRefinement refinement;


    VarTypeTableModel() {
      super(false);
      setColumnNames(columns);
      setColumnClasses(classes);
    }


    public void clear() {
      rows.clear();
      clearMarks();
    }

    public void setData(UIRefinement uir, Collection itemVars) {
      rows.clear();
      clearMarks();
      refinement = uir; //keep for adding data
      if (itemVars != null) 
	for (Iterator i = itemVars.iterator(); i.hasNext(); ) {
	  Object next = i.next();
	  if (next instanceof ItemVar) {
	      //noteVar((ItemVar)next, uir); 
	    rows.add(next);
	  }
	}
      //then add an empty row for new variables
      rows.add(null);

      fireTableDataChanged();
    }

    public void addData(ItemVar itemVar) {
      if (itemVar == null) return;
      //noteVar(itemVar, refinement);

      int i = getObjectRow(null);
      //Debug.noteln("DecEd: null row is", i);
      rows.remove(null);
      rows.add(itemVar);
      rows.add(null);
      fireTableDataChanged();
    }

    public Object getCellValueAt(Object baseObject, int columnIndex) {
      ItemVar iVar = (ItemVar)baseObject;
      if (columnIndex == VAR_COL) {
	if (iVar == null) return "";
	else return iVar.toString();
      }
      else if (columnIndex == TYPE_COL) { 
	return DeclarationEditor.this.varTypeMap.get(iVar);
      }
      else if (columnIndex == CONSTRAINTS_COL) { 
	Object cs = DeclarationEditor.this.getVarConstraints(iVar);
	//true for non-empty collections and maps
	return Boolean.valueOf((cs != null) 
			       && (((cs instanceof Collection) 
				    && !((Collection)cs).isEmpty()) ||
				   ((cs instanceof Map) 
				    && !((Map)cs).isEmpty())));
      }
      else if (columnIndex == NOCONST_COL) { 
	Object cs = DeclarationEditor.this.getNoObjectConstraints(iVar);
	//Debug.noteln("DecEd: nodecs for "+iVar+" are", cs);
	//true for non-empty sets
	return Boolean.valueOf((cs != null) 
			       && (cs instanceof Collection) 
			       && !((Collection)cs).isEmpty());
      }
      else return null;
    }

    public boolean isCellEditable(int row, int col) {
      if ((col == NOCONST_COL) || (col == CONSTRAINTS_COL)) return false;
      else return true;
    }

    public void setValueAt(Object val, int row, int col) {
      //Debug.noteln("DecE: settingValue " + val + " of class",val.getClass());
      Object rowOb = getRowObject(row);
      if (col == VAR_COL) {
	//var name edited; check new name is ok, if no - change or cancel
	//new name ok: replace map entry, replace in data but keep selected
	Symbol newVar = Symbol.intern(val.toString());
	if (!(newVar instanceof ItemVar)) {
	  String[] message = 
	    {val.toString() + " is not a valid variable name",
	     "Please specify a name like this: ?person",
	     "(Nothing done)"};
	  JOptionPane.showMessageDialog(null, message);
	  return;
	}
	else if (rowOb == null) { //new name, so add the data
	  addData((ItemVar)newVar);
	}
	else { //old name, so replace 
	  //replaceData(row, (ItemVar)rowOb, (ItemVar)newVar);
	  replaceData((ItemVar)rowOb, (ItemVar)newVar);
	  replaceNotes((ItemVar)rowOb, (ItemVar)newVar);
	  //fireVarNameChanged((ItemVar)rowOb, (ItemVar)newVar);
	}
      }
      else if (col == TYPE_COL) {
	//type changed. Check if var name set. If not, complain and stop.
	if ((rowOb == null) || rowOb.toString().equals("")) {
	  JOptionPane.showMessageDialog(null, 
					"Please set the variable name first");
	}
	//check new value is proper type. If so, replace map entry (null ok)
	else if ((val == null) || (val instanceof UIObjectClass)) {
	  DeclarationEditor.this.noteVarType((ItemVar)rowOb, (UIObjectClass)val);
	}
	else Debug.noteln("DecEd: new value of class", val.getClass());
	//else the new type may be a string which is not allowed!
      }
    }

    protected void updateConstCols() {
      TableModelEvent tme = 
	new TableModelEvent(this, 1, getRowCount(), CONSTRAINTS_COL);
      fireTableChanged(tme);
      tme = new TableModelEvent(this, 1, getRowCount(), NOCONST_COL);
      fireTableChanged(tme);
    }
  }

}


/** Issues
 * 
 * the type in the constraint may not be known (what if not!)
 **/
/** Help
 *
 *
 **/
/** Changes
 **/
