/****************************************************************************
 * An abstract class with useful things for building constraint editors.
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Mon Oct  9 09:44:06 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import java.util.*;
//import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Frame;
import java.awt.Component;
import ix.*;
import ix.util.Debug;
//import ix.util.lisp.LList;
//import ix.util.lisp.Lisp;
import ix.icore.domain.*;
import ix.icore.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iface.ui.*;

/****************************************************************************
 * An abstract class with useful things for building constraint editors.
 *
 * The editor is implemented as a frame using Boxes and JButtons.  <p>
 *
 * The editor can be created with no arguments or with a given object
 * which will be set as the current object to which the constraint applies.
 * Subsequently, the object in the editor can be set using the setObject
 * method. <p>
 *
 * Concrete editors that extend this class should implement the following 
 * methods:
 * <ul> makeRels() - makes all components needed to specify the constraint.
 * <li> initFromObject() - initialises components with information from the
 *      current object, e.g. lists of children are filled with those of the
 *      current refinement.
 * <li> setFromObject(Object original) - set the constraints using the given
 *      object (used when editing constraints).
 * <li> collectConstraint() - collects user input from the components and 
 *      creates (and returns) a Constraint object from that information.
 *      Also deals with edit vs add behaviour.
 * <li> If there is a component that should have the keyboard focus when the 
 *      editor comes up, also define protected JComponent getFocusComponent()
 * </ul>
 * The rest can be handled by the generic methods defined in this class.
 * For an example of an implementing class see TPRelationEditor.
 *
 * Implements the VarSpecifier interface for UIRefinements. Use
 * IVUtil.enableVars(VarSpecifier editor, JTextComponent textBit);
 * to activate variable input help. Collects variable declarations and
 * updates the construct when the constraint is processed (in case of cancel)
 *
 *****************************************************************************
 */
public abstract class ConstraintEditor extends JConstraintEditor
  implements VarSpecifier
{
  //Variable declarations go in when editing is ok-ed only
  protected List varDecs = new ArrayList();

  public ConstraintEditor(Frame owner, String title, String message, 
			  Object object) {
    super(owner, title, message, object);
  }
  public ConstraintEditor(Frame owner, String title, String message) {
    super(owner, title, message);
  }
  public ConstraintEditor(Frame owner, String message) {
    super(owner, message);
  }

  public void start() {
    if (varDecs == null) varDecs = new ArrayList();
    else varDecs.clear();
    super.start();
  }


  public List getVarsToOffer() {
    try {
      UIRefinement uiRef = (UIRefinement)currentObject;
      return uiRef.getVariableDeclarations();      
    }
    catch (ClassCastException cce) {
      return null;
    }
  }

  public String addVariableDeclaration(String name) {
    VariableDeclaration dec = IVUtil.makeVariableDeclaration(name);
    if (dec != null) {
      varDecs.add(dec);
      return dec.getName().toString();
    }
    return "";
  }


  /**
   * Reads the selected relation and let interested parties know.
   * Checks that selections have been made.
   * If selections have not been made, a message will have been shown and 
   * listeners are not notified.
   * If an original constraint has been edited (rather than new constraints
   * added), the listeners are notified with a null constraint object (they
   * already hold the original that has now been updated).
   */
  protected boolean processConstraint() {
    //Debug.noteln("CE: processing constraint");
    Object constraint = collectConstraint();
    if (constraint == null) return false; // constraint not specified
    //Debug.noteln("CE: made constraint", constraint);
    handleVariables();
    //Debug.noteln(" original is", original);
    Object newValue = noteNewValue(constraint);
    if (this.constraintListeners == null) return true;
    for (int i=0; i<this.constraintListeners.size(); i++) {
      JConstraintListener l = 
	(JConstraintListener) this.constraintListeners.get(i);
      l.gotConstraint(this, this.currentObject, newValue);
    }
    return true;
  }

  /**
   * Pass on declared variables to the object specification.
   * Many constraint editors can add new variables which should be
   * added to the object's variable declarations. 
   * Note: The variable declaration editor is the only one that should
   * overwrite variables instead (so it must implement its own handling
   * of declared variables)
   */
  protected void handleVariables() {
    if ((varDecs != null) && (varDecs.size() > 0)) {
      //Debug.noteln("CE: got varDecs", varDecs);     
      try {
	UIRefinement uiRef = (UIRefinement)currentObject;
	List vars = new ArrayList();
	List currentDecs = uiRef.getVariableDeclarations();
	if (currentDecs != null) vars.addAll(currentDecs);
	if (vars == null) vars = new ArrayList();	
	for (Iterator i = varDecs.iterator(); i.hasNext(); ) {
	  VariableDeclaration dec = (VariableDeclaration)i.next();
	  vars.add(dec);
	}
	uiRef.setVariableDeclarations(vars);
	varDecs.clear();
      }
      catch (Exception e) {
	Debug.noteException(e);
      }
    }
  }

  private String field = "";

  protected String getField(Object constraint) { return field;}

  protected List getPreviousList(Object constraint) {
    if (constraint instanceof Ordering) {
      field = "orderings";
      return new ArrayList(((UIRefinement)currentObject).getOrderings()); 
    }
    if (constraint instanceof Constraint) {
      field = "constraints";
      return new ArrayList(((UIRefinement)currentObject).getConstraints());
    }
    if (constraint instanceof UISpec) {
      field = "issues";
      return new ArrayList(((UIRefinement)currentObject).getIssues()); 
    }
    if (constraint instanceof Issue) {
      field = "issues";
      return new ArrayList(((UIRefinement)currentObject).getIssues());
    }
    if (constraint instanceof NodeSpec) {
      field = "nodes";
      return new ArrayList(((UIRefinement)currentObject).getNodes()); 
    }
    field = "";
    return null;
  }


  protected Component makeBits() {return makeRels();}

  /**
   * Makes all items on the panel used to specify parts of the constraint.
   */
  abstract protected Component makeRels();
  /**
   * Fills the controls with relevant object-sensitive information.
   * To be implemented by the sub-class. E.g. fill ordering editor with the
   * sub-activities of the refinement.
   */
  abstract protected void initFromObject();
  /**
   * Sets the editor up to edit (rather than add) an original constraint
   * object.
   */
  abstract protected void setFromObject(Object original);
  /**
   * Collects input from the panel components and builds the right kind of
   * constraint.
   */
  abstract protected Object collectConstraint();
  //also define protected JComponent getFocusComponent()

}

/** Changes
 * 
 * 16/02/04: Fixed bug on variable declarations.
 * 25/10/04: updated variable handling problem to split DeclarationEditor 
 *           behaviour from this one
 *
 */
