/****************************************************************************
 * An abstract editor/viewer panel for domain structures
 *
 * @author Jussi Stader
 * @version 4.1
 * Updated: Thu Mar 15 14:02:45 2007
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iview;

import java.lang.Math;
import java.util.*;
import javax.swing.*;       
//import javax.swing.text.*;       
//import java.awt.*;
import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.Component;
import java.awt.event.*;
import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.iview.*;
import ix.iview.domain.*;
import ix.iview.util.*;
import ix.iview.event.*;

/****************************************************************************
 * An abstract editor/viewer panel for domain constructs.
 *
 * This kind of panel is intended to sit within a frame (domain
 * editor); it will usually contain one or more ConstuctEditing
 * panels. E.g. the ActionEditorPanel has alternative ConstructEditing
 * panels for simple/comprehensive/graphical views. <p>
 *
 * This abstract class manages the generic window building (scoll,
 * connection to frame, etc) and undo/redo, options for modified
 * objects, notification of ConstructListeners, open/close/switching
 * behaviour, and variable declarations
 *
 * Abstract methods to define are: <p>
 *
 * protected JPanel setupMainPanel() to set up the main editing
 * panel. This is called during creation/initialisation. Often, this
 * is a simple wrapper panel that contains a ConstructEditing panel
 * and not much else. <p>
 *
 * public void refresh() to refresh the panel, making sure
 * updates are shown etc.  <p>
 *
 * public void newConstruct() to clear the panel and set up a new
 * construct of the kind to be edited in this panel <p>
 *
 * public UIObject makeNewConstruct() to create a construct of the
 * kind to be edited in this panel. Returns the new construct. Note
 * that this method should not change the panel display. <p>
 *
 * public boolean frameActionPerformed(ActionEvent event) for handling
 * user requests that are generated in the frame (e.g. in the menu)
 * rather than in the panel <p>
 *
 * public void actionPerformed(ActionEvent ae) Wakes up the
 * ActionListener with a user action. This is called when a KeyStroke
 * or other panel event (e.g. button click) happens in which the
 * ActionListener registered its interest. <p>
 *
 * protected UIObject noteConstruct() transfer information from the
 * panel into the object that is being edited. <p>
 *
 * Optional methods to overwrite are:
 *
 * public void viewChanges() to show the user changes between the
 * edited object and the version in the draft domain <p>
 *
 * public void viewChangesFromOriginal() to show the user changes between the
 * edited object and the version in the published domain <p> <p>
 *
 * public Set search(String pattern) to search the domain constructs<p>
 *
 *
 * To add a new ConstructFramePanel to an editor frame, subclass this
 * panel, implement the abstract methods, put a ConstructEditing panel
 * inside (use setupMainPanel to do this), and implement its
 * methods. Tie the methods to the commands using frameActionPerformed
 * and set up your own commands using buttons etc. and the
 * actionPerformed method.
 *
 * 
 *****************************************************************************
 */
public abstract class AConstructFramePanel extends JPanel 
  implements ConstructFraming, VarSpecifier, ActionListener, UndoEditing,
	     DataChangeListener
	     //FormDataListener
{

  //---------------------Fields-------------------------------------------
  /** The panels parent - a DomainEditor */
  public DomainEditorFrame parent;
  /** The scrollPane in which all the panel's contents are placed */
  protected JScrollPane scrollPane = new JScrollPane();
  /** The panel's main panel in which editing occurs */
  protected JPanel mainPanel; /******should this be ConstructEditing or sim?*/
  /**
   ****seems like this should not be here! define lower down? used
   ****lots in ActionEditorPanel and GlobalFramePanel, not here or in
   ****DomainEditor 
  */
  protected IFormPanel formPanel;

  private String alwaysSave = "always";
  private boolean doSwitch = true; //passes cancel from modal dialogue
  protected boolean switching = false;
  private JModifyDialog modifyDialog;
  private UndoManager undoManager;

  protected boolean hasOverview = false;
  public static boolean DEFAULT_SORT = true;


  //---------------------Constructors----------------------------------------

  /**
   * Creates an editor panel. Don't use this
   */    
  public AConstructFramePanel(){
    super();   
  }
  /**
   * Creates an editor panel with the given parent.
   * Creates all panel components, the mouse listener, and the relation editor.
   *
   * @param parent the DomainEditorFrame that the panel belons to. Used 
   *   mainly to keep toggle buttons and menus in synch
   */    
  public AConstructFramePanel(DomainEditorFrame theParent){
    super();
    parent = theParent;
    setupEditor();
    if (mainPanel instanceof UndoEditing) {
      //Debug.noteln("ACFP: Setting up undo for ", this);
      undoManager = new UndoManager(this);
      undoManager.addUndoListener((UndoChangeListener)parent);
      this.addCurrentConstructListener(undoManager);
      undoManager.setConstruct(getUIConstruct());
    }
  }
  
  //-----------------------UndoEditing things-----------------------
  // reading/updating the model will have the desired effect

  /** Sets the given field to the given value in the editor. */
  public void undoSetValue(String field, Object value) {
    if (mainPanel instanceof UndoEditing) 
      ((UndoEditing)mainPanel).undoSetValue(field, value);
    else {
      Debug.noteln("AFP: got undo set for field "+field+" val", value);
      Debug.noteln("Panel "+mainPanel.getName()+" cannot undoEdit");
    }
  }

  /** Gets the given field to the given value in the editor. */
  public Object undoGetValue(String field) {
    if (mainPanel instanceof UndoEditing) 
      return ((UndoEditing)mainPanel).undoGetValue(field);
    else {
      Debug.noteln("AFP: got undo get for field", field);
      return null;
    }
  }


  //---------------------Window building-------------------------------------
  /**
   * Sets the panel's layout and dimensions and the main panel's layout.
   */
  protected void setupEditor() {
    setupOuterPanel();
    mainPanel = setupMainPanel();
    scrollPane.getViewport().add(mainPanel); //mainPanel
  }

  protected void setupOuterPanel() {
    this.setLayout(new BorderLayout());
    //scrollPane.setBounds(0,0,405,505);
    this.add(scrollPane,BorderLayout.CENTER);
  }
  /**
   * Sets up the main panel in which construct editing will
   * happen. Often, this is a simple wrapper panel that contains a
   * ConstructEditing panel and not much else.
   */
  abstract protected JPanel setupMainPanel();

  public ConstructEditing getEditingPanel() {
    if (mainPanel instanceof ConstructEditing) 
      return (ConstructEditing)mainPanel;
    else if (this instanceof ConstructEditing) 
      return (ConstructEditing)this;
    else if (formPanel instanceof ConstructEditing) 
      return (ConstructEditing)formPanel;
    else return null;
  }


  public void registerListKeys(JComponent comp, String commandEnd) {
    String name = "add " + commandEnd;
    Action act = new AbstractAction(name) {
      public void actionPerformed(ActionEvent e) {
	Debug.noteln("ACFP: Got key action");
	AConstructFramePanel.this.actionPerformed(e);
      }
    };
    InputMap iMap = comp.getInputMap(); //default: WHEN_FOCUSED
    ActionMap aMap = comp.getActionMap();
    aMap.put(name, act);
    iMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER,0), name);
    iMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_INSERT,0), name);

    name = "delete " + commandEnd;
    act = new AbstractAction(name) {
      public void actionPerformed(ActionEvent e) {
	AConstructFramePanel.this.actionPerformed(e);
      }
    };
    aMap.put(name, act);
    iMap.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE,0), name);
  }

  public ConstructEditing getMainPanel() {
    return (ConstructEditing)mainPanel;
  }

  //---------------------Public services--------------------------------------

  public void editConstruct(Object construct) {
    if (construct instanceof UIObject) 
      setUIConstruct((UIObject)construct);
    else if (construct instanceof IXObject) 
      setConstruct((IXObject)construct);
    else if (construct instanceof NodeSpec) {
      UIRefinement uir = new UIRefinement(getUIDomain());
      uir.setPattern(((NodeSpec)construct).getPattern());
      if (uir.getName() == null) uir.setName("undefined");
      setUIConstruct(uir);
    }
  }

  /**
   * User request to delete the current action.
   */
  public void deleteConstruct() {
    getUIDomain().removeConstruct(getUIConstruct());
    newConstruct();
  }

  /**
   * User request to copy the current action.
   */
  public void copyConstruct() {
    UIObject uiConstruct = getUIConstruct();
    if (!uiConstruct.isEmpty() && beforeSwitching()) {
      UIObject copy = cloneConstruct(uiConstruct);
      if (copy == null) {
	UIUtil.notImplemented(this,"Copying this kind of construct");
      }
      else {
	String answer = 
	  JOptionPane.showInputDialog(null,"Please enter the new name",
				      uiConstruct.getName());
	copy.setName(answer);
	//Debug.noteln("ACFP: copy is", copy.print());
	copy.saveToDomain();
	getUIDomain().addConstruct(copy);
	setUIConstructPanel(copy); 
	afterSwitching();
      }
    }
  }

  /** overwrite this to not call before/after switching */
  public void setUIConstructPanel(UIObject construct) {
    setUIConstruct(construct);
  }


  public Set search(String pattern) {
    List constructs = getAllConstructs();
    return IVUtil.findNameMatches(pattern, constructs);
  }

  /** 
   * User request to view the changes from the original
   * object. Collects the changes and shows them to the user.
   */
  public void viewChangesFromOriginal() {
    UIObject uiConstruct = getUIConstruct();
    if (uiConstruct == null) 
      JOptionPane.showMessageDialog(this,"No construct to check");
    else {
      List changes = uiConstruct.collectChangesFromOriginal();
      if ((changes == null) || (changes.size() == 0))
	JOptionPane.showMessageDialog(this,"No changes found");
      else {
	TextAreaFrame taf = new 
	  TextAreaFrame("Changes for " + uiConstruct.getName());
	for (Iterator i=changes.iterator(); i.hasNext();)
	  taf.appendLine((String)i.next());
	taf.setVisible(true);
      }
    }
  }
  /**//***into abstract
   * User request to view the changes from the draft version of the object.
   * object. Collects the changes and shows them to the user.
   */
  public void viewChanges() {
    UIObject uiConstruct = getUIConstruct();
    if (uiConstruct == null) 
      JOptionPane.showMessageDialog(this,"No construct to check");
    else {
      List changes = uiConstruct.collectChanges();
      if ((changes == null) || (changes.size() == 0))
	JOptionPane.showMessageDialog(this,"No changes found");
      else {
	TextAreaFrame taf = new 
	  TextAreaFrame("Changes for " + uiConstruct.getName());
	for (Iterator i=changes.iterator(); i.hasNext();)
	  taf.appendLine((String)i.next());
	taf.setVisible(true);
      }
    }
  }
  /**
   * User request to save a construct into the draft domain.
   */
  public void saveConstruct(){
    UIObject uiConstruct = noteConstruct();
    setUIConstructPanel(uiConstruct);
    //uiConstruct.saveToDomain();
  }
  /**
   * User request to revert an action to when it was last saved in 
   * the draft domain.
   */
  public void revertConstruct(){
    UIObject uiConstruct = getUIConstruct();
    uiConstruct.loadFromDomain();
    //currentPanel.setUIConstruct(uiConstruct);
    setUIConstructPanel(uiConstruct);
  }
  /**
   * User request to revert an action to when it was last saved in 
   * the draft domain.
   */
  public void revertOConstruct(){
    UIObject uiConstruct = getUIConstruct();
    uiConstruct.loadFromOriginal();
    //currentPanel.setUIConstruct(uiConstruct);
    setUIConstructPanel(uiConstruct);
    //uiConstruct.saveToDomain();
  }

  /**
   * Sets the given construct to be viewed/edited. 
   * Makes a UI object unless there is already one, displays its details in the
   * relevant fields of the editor, and remembers the UI object as current.
   * Checks whether this object has previously been edited and recovers undo
   * information if available.
   *
   * @param object the object to be viewed/edited
   */    
  public void setConstruct(IXObject object) {
    //Debug.noteln("ACFP: setting construct in", this);
    UIObject uio = getUIConstruct(); //get current one
    ((ConstructEditing)mainPanel).setConstruct(object);
    UIObject newUIO = getUIConstruct();
    //if ((uio != null) && !uio.equals(newUIO)) //fired in ConstructEditing
    //  fireConstructChanged(uio, newUIO);
  }

  /**
   * Sets the given UI construct to be viewed/edited. 
   * Displays the ojbect's details in the relevant fields of the editor, 
   * and remembers the object as current.
   * Manages the undo lists.
   *
   * @param object the UIObject to be viewed/edited
   */    
  public void setUIConstruct(UIObject object) {
    //Debug.noteln("ACFP: Setting UI construct in", this);
    ConstructEditing editPanel = (ConstructEditing)mainPanel;
    //set the object in the actual panel
    UIObject current = editPanel.getUIConstruct();
    if ((current != null) && current.equals(object)) {
      editPanel.loadFromObject(); //in case object changed ***think undo!
      if (undoManager != null) undoManager.fireUndoChange(); //just in case
    }
    else {
      editPanel.setUIConstruct(object);
    }
    //if ((current == null) || !current.equals(object)) //something has changed
    //fireConstructChanged(current, object);//fired in ConstructEditing
    //else if (undoManager != null) undoManager.fireUndoChange();//just in case
   }

  /**
   * Sets the given UI construct to be viewed/edited. 
   * Displays the ojbect's details in the relevant fields of the editor, 
   * and remembers the object as current.
   * Manages the undo lists.
   *
   * @param object the UIObject to be viewed/edited
   */    
  public void OLDsetUIConstruct(UIObject object) {
    //Debug.noteln("ACFP: Setting UI construct in", this);
    ConstructEditing editPanel = (ConstructEditing)mainPanel;
    //set the object in the actual panel
    UIObject current = editPanel.getUIConstruct();
    if ((current != null) && current.equals(object)) {
      editPanel.loadFromObject(); //in case object changed ***think undo!
      if (undoManager != null) undoManager.fireUndoChange(); //just in case
    }
    else {
      editPanel.setUIConstruct(object);
    }
    //if ((current == null) || !current.equals(object)) //something has changed
    //fireConstructChanged(current, object);//fired in ConstructEditing
    //else if (undoManager != null) undoManager.fireUndoChange();//just in case
   }

  /**
   * Looks up the current construct in the main panel.
   */
  public UIObject getUIConstruct() {
      return ((ConstructEditing)mainPanel).getUIConstruct();
    }

  public Domain getDomain() {
    return getUIDomain().getDomain();
  }
  public UIDomain getUIDomain() {
    return parent.getUIDomain();
  }
  public void setUIDomain(UIDomain uid) {
    if (uid.equals(getUIDomain())) return;
    else parent.setUIDomain(uid);
  }

  public void clear() {
    newConstruct();
  }


  public boolean hasOverview() { return hasOverview;}
  public void updateOverview() {}
  public void setSort(boolean sortIt){}
  public boolean getSort() {return DEFAULT_SORT;}

  public boolean canUndo() {
    if (undoManager == null) return false;
    return undoManager.canUndo();
  }
  public boolean canRedo() {
    if (undoManager == null) return false;
    return undoManager.canRedo();
  }

  public boolean undo() {
    if (undoManager == null) return false;
    else return undoManager.undo();
  }
  public boolean redo() {
    if (undoManager == null) return false;
    else return undoManager.redo();
  }


  public void populatePopup(JObjectPopup popup, Object construct) {}


  //--------------------------- Private useful things 

  public void addUndoListener(UndoChangeListener listener) {
    if (undoManager != null) undoManager.addUndoListener(listener);
  }

  //--------------------------- Event things ------------------------------

  //do constructChange listeners here because they don't care which of
  //the editing panels is in charge

  HashSet currentConstructListeners = new HashSet();
  public void addCurrentConstructListener(CurrentConstructListener ccl) {
    currentConstructListeners.add(ccl);
  }
  public void removeCurrentConstructListener(CurrentConstructListener ccl) {
    currentConstructListeners.add(ccl);
  }
  public void fireConstructChanged(UIObject oldUIO, UIObject newUIO, 
				   Component source) {
    //Debug.noteln("ACFP: Object may have changed in", this.getClass());
    if ((this instanceof DataChangeListener) && (newUIO != null))
      newUIO.addDataChangeListener((DataChangeListener)this); 
    //exit if no change
    if (oldUIO == null) { if (newUIO == null) return; }
    else { if (oldUIO == newUIO) return; }  //only ignore if same OBJECT
    //Debug.noteln("ACFP: Object really may have changed");
    for (Iterator i = currentConstructListeners.iterator(); i.hasNext(); ) {
      //Debug.noteln("ACFP: firing to construct change listeners");
      CurrentConstructListener ccl = (CurrentConstructListener)i.next();
      ccl.constructChanged(source, oldUIO, newUIO);
    }
  }

  //--------------------------- Switching things ------------------------------

  public boolean beforeClosing() {
    //fileUndos(((ConstructEditing)mainPanel).getUIConstruct());
    return beforeSwitching();
  }
  public boolean beforeOpening() {
    if (ConstructEditing.class.isInstance(mainPanel))
      setUIConstruct(getUIConstruct());
    afterSwitching();
    return true;
  }

  /**
   * Ensures that displays are updated for new view/construct.
   * Currently just makes sure that undos are shown properly (switching
   * between graphical and form-based).
   */
  public void afterSwitching() {
    if (undoManager != null) undoManager.fireUndoChange();
  }

  /**
   * Notes the current panel's contents in the current construct before 
   * switching. 
   * @return true if the switch should still take place, false to cancel the
   * switch.
   */
  public boolean beforeSwitching() {
     //if (switching) return true;

    if (!isVisible()) return true;

    switching = true;
    //Debug.noteln("ACFP: Preparing for switch of", getName());
    if (alwaysSave.equals("never")) {
      //Debug.noteln("AbCFP: never modifying");
      switching = false;
      return true;
    }
    
    UIObject uio = getUIConstruct();
    if (uio == null) uio = makeNewConstruct();
    ((ConstructEditing)mainPanel).saveToObject(uio);
    ((ConstructEditing)mainPanel).setUIConstruct(uio);

    //if there is no object or it has not changed since the last save,
    // just switch
    //Debug.noteln("ACFP: checking uio for change - uio:", uio.getName());
    if (!uio.hasChanged()) {
      switching = false;
      return true;
    }
    //Debug.noteln(getName() + " uio has changed.");

    if (alwaysSave.equals("always")) {
      //Debug.noteln("AbCFP: always modifying");
      noteDraftConstruct(uio);
      switching = false;
      return true;
    }
    else {
      doSwitch = true;
      askModifyOptions(uio);
      //askModifyDialog(uio);
      switching = false;
      return doSwitch;    
    }
  }

  protected void askModifyOptions(UIObject uio) {
    Debug.noteln(getName() + " panel: Asking modify options for",uio);
    String[] message = 
      {"You have made changes in the panel.",
       "Do you want to modify the draft?"};
    Object[] options = {"Modify Draft", "Discard Changes" , "Always Modify", 
			"Always Discard"};
    Object o = JOptionPane.showInputDialog(parent, message, "Confirm", 
					   //JOptionPane.DEFAULT_OPTION, 
					   JOptionPane.QUESTION_MESSAGE,
					   null, options, options[0]);
    if (o == null) { //cancel
      //Debug.noteln("ACFP: cancel switch");
      doSwitch = false; 
    }
    else if (o.equals("Modify Draft")) { //save
      noteDraftConstruct(uio);
      doSwitch = true;
    }
    else if (o.equals("Discard Changes")) { //discard
      //reset the construct to its last saved state, then switch
      uio.loadFromDomain();
      //Debug.noteln("ACFP: discarded change - name is now", uio.getName());
      ((ConstructEditing)mainPanel).setUIConstructOnly(uio);
      ((ConstructEditing)mainPanel).loadFromObject();
      doSwitch = true;
    }
    else if (o.equals("Always Modify")) { //always save
      alwaysSave = "always";
      noteDraftConstruct(uio);
      doSwitch = true;
    }
    else if (o.equals("Always Discard")) { //always discard
      alwaysSave = "never";
      //reset the construct to its last saved state, then switch
      uio.loadFromDomain();
      ((ConstructEditing)mainPanel).setUIConstructOnly(uio);
      ((ConstructEditing)mainPanel).loadFromObject();
      doSwitch = true;
    }
  }

  protected void askModifyDialog(UIObject uio) {
    //Debug.noteln("AbCFP: asking modify dialogue");
    if (modifyDialog == null) 
      modifyDialog = new JModifyDialog(parent, this, parent.modifyIcon);
    modifyDialog.ask(uio);
  }


  /**
   * Note the UIObject in the (right) draft domain
   */
  protected void noteDraftConstruct(UIObject uio) {
    uio.noteChange();
    //make sure the construct is noted into the right domain
    if (!getUIDomain().equals(uio.getUIDomain())) 
      uio.moveToUIDomain(getUIDomain());
    getUIDomain().updateConstruct(uio);
    ((ConstructEditing)mainPanel).setUIConstructOnly(uio);
    //Debug.noteln("ACFP: Noted construct");
    //Debug.noteln(" ",uio.print());
  }



  public void dataChanged(EditableObject o, String field, 
			  Object oVal, Object nVal) {
    if (o == getUIConstruct()) 
      if (field.equals("name")) 
	if (getUIDomain() != null) {
	  getUIConstruct().noteChange();
	  getUIDomain().updateConstruct(getUIConstruct());
	}
  }

  //---------------------Inner Classes----------------------------------------
  protected class FieldValue {
    protected String field;
    protected Object value; 
    protected FieldValue(String f, Object v) {
      field = f;
      value = v;
    } 
  }


  protected class JModifyDialog extends JDialog {
    UIObject uio;
    AConstructFramePanel cPanel;
    String topString = "Modify draft?";
    String[] message = {"You have made changes in the editor.",
			"Do you want to:-",
       "- Modify: put the changes into the draft, then continue?",
       "- Discard: continue without noting changes?",
       "- Cancel: Do nothing?"};
    Object[] options = {"Modify", "Discard" , "Cancel"};
    
    JLabel topLabel; 

    JToggleButton askToggle = new JCheckBox("Always ask", true);
    JToggleButton modToggle = new JCheckBox("Always modify", false);
    JToggleButton discardToggle = new JCheckBox("Always discard", false);
    ButtonGroup bg = new ButtonGroup();

    MouseListener ml;
    JPanel butPanel;
    JPanel topPanel;


    protected JModifyDialog(JFrame frame, AConstructFramePanel panel, Icon i) {
      this(frame, panel);
      topLabel.setHorizontalTextPosition(SwingConstants.LEFT);
      topLabel.setIcon(i);
    }
    protected JModifyDialog(JFrame frame, AConstructFramePanel panel) {
      super(frame, true);

      setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
      setLocationRelativeTo(panel);

      this.cPanel = panel;


      ml = new MouseAdapter() {
	public void mouseClicked(MouseEvent event) {
	  Component component = event.getComponent();
	  if (!AbstractButton.class.isInstance(component)) return;
	  
	  AbstractButton button = (AbstractButton) component;
	  String command = button.getActionCommand();
	  if (command.equals("Modify")) modify();
	  else if (command.equals("Discard")) discard();
	  else if (command.equals("Cancel")) cancel();
	}
      };


      JPanel togPanel = new JPanel();
      JPanel mPanel = new JPanel();
      topPanel = new JPanel();
      butPanel = new IXButPanel(ml, BoxLayout.X_AXIS, options);
      bg.add(askToggle);
      bg.add(modToggle);
      bg.add(discardToggle);
      askToggle.setActionCommand("");
      modToggle.setActionCommand("always");
      discardToggle.setActionCommand("never");

      topLabel = new JLabel(topString);

      new BoxLayout(togPanel, BoxLayout.X_AXIS);
      GridLayout gl = new GridLayout(message.length, 1);
      mPanel.setLayout(gl);
      for (int i = 0; i < message.length; i++) {
	mPanel.add(new JLabel(message[i]));
      }
      togPanel.add(askToggle);
      togPanel.add(modToggle);
      togPanel.add(discardToggle);
      

      BorderLayout topBL = new BorderLayout();
      topBL.setVgap(topBL.getVgap() + 10);
      topPanel.setLayout(topBL);
      topPanel.add(mPanel, BorderLayout.CENTER);
      topPanel.add(togPanel, BorderLayout.SOUTH);


      BorderLayout contBL = new BorderLayout();
      contBL.setVgap(contBL.getVgap() + 10);
      getContentPane().setLayout(new BorderLayout());

      getContentPane().add(topLabel, BorderLayout.NORTH);
      getContentPane().add(topPanel, BorderLayout.CENTER);
      getContentPane().add(butPanel, BorderLayout.SOUTH);

      pack();
      //Debug.noteln("Dimension is", getSize());
      setSize(getWidth(), getHeight() + 1);
      //Debug.noteln("Dimension is", getSize());
      validate(); 
    }
    
    public void ask(UIObject uio) {
      //(new Throwable()).printStackTrace();
      //Debug.noteln("AbCFP-JMD: asking");
      this.uio = uio;
      topLabel.setText(topString + " (" + uio.getName() + ")"); 
      this.setVisible(true);
    }

    private void modify() {
      setAlways();
      if (uio == null) return;
      else noteDraftConstruct(uio);
      cPanel.doSwitch = true;
      dispose();
    }
    private void discard() {
      setAlways();
      if (uio == null) return;
      else {
	uio.loadFromDomain();
	//Debug.noteln("ACFP: discarded change - name is now", uio.getName());
	((ConstructEditing)cPanel.mainPanel).setUIConstructOnly(uio);
	((ConstructEditing)cPanel.mainPanel).loadFromObject();
      }
      cPanel.doSwitch = true;
      dispose();
    }
    private void cancel() {
      cPanel.doSwitch = false;
      dispose();
    }
    private void setAlways() {
      ButtonModel bm = bg.getSelection();
      String command = bm.getActionCommand();
      //Debug.noteln("AbCFP: Got always", command);
      if (command == null) return;
      else cPanel.alwaysSave = command;
    }

  }


  /**
   * Offers the declared variables (if any) for completion, splices in 
   * selection (if any)
   */
  public List getVarsToOffer() {
    try {
      UIRefinement uiRef = (UIRefinement)getUIConstruct();
      return uiRef.getVariableDeclarations();
    }
    catch (Exception e) {
      Debug.describeException(e);
      return null;
    }
  }

  public String addVariableDeclaration(String name) {
    VariableDeclaration dec = IVUtil.makeVariableDeclaration(name);
    if (dec != null) {
      try {
	UIRefinement uiRef = (UIRefinement)getUIConstruct();
	List vars = uiRef.getVariableDeclarations();
	if (vars == null) vars = new ArrayList();
	vars.add(dec);
	uiRef.setVariableDeclarations(vars);
	return dec.getName().toString();
      }
      catch (Exception e) {
	Debug.describeException(e);
	return "";
      }
    }
    return "";
  }


  /**
   * Makes a new construct of the panel's type and get ready to edit
   * it by seting it in the panel. 
   * Overwrite if this is not the thing to do. 
   * Note: this used to be an abstract method.
   */
  public void newConstruct(){
    setUIConstruct(makeNewConstruct()); 
  }

  //---------------------Abstract services------------------------------------


  abstract public void refresh();

  abstract public UIObject makeNewConstruct();
  //return null if this is not possible (used to copyConstruct)
  abstract protected UIObject cloneConstruct(UIObject original);

  abstract public boolean frameActionPerformed(ActionEvent event);
  /**
   * Wakes up the ActionListener with a user action.
   * This is called when a KeyStroke or other panel event (button
   * click) happens in which the ActionListener registered its
   * interest.
   */
  abstract public void actionPerformed(ActionEvent ae);
  /**
   * Note the contents of the panel into the current object.
   */
  abstract protected UIObject noteConstruct();

}

/*Issues***************************************************************
 *
/*Todos***************************************************************
 *
 * - 
 *  Wait for 
 *
 *****************************************************************************
 */
