/****************************************************************************
 * A table viewer for an agenda (set of agenda items).
 * 
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Mon Sep 25 09:40:58 2006
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.ip2;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import javax.swing.table.*;
import javax.swing.text.JTextComponent;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

import java.net.URL;

import ix.ip2.event.*;

import ix.icore.*;
import ix.icore.domain.Refinement;

import ix.iface.util.*;
import ix.iface.ui.util.*;
import ix.iface.ui.*;
import ix.iface.ui.table.*;

import ix.iview.util.*;
import ix.iview.table.*;

import ix.util.*;
import ix.util.lisp.*;

/**
 * A table viewer for an agenda (set of agenda items).
 * The items are managed by an AgendaManager (usually an Agenda).  
 */
abstract public class AgendaTableViewer extends IXTreeTable
    implements AgendaViewer, AgendaListener, MouseListener
{

  protected Ip2 ip2;

  AgendaManager agendaManager;
  ItemEditor itemEditor;

  AgendaItemTableModel model;

  private static final ImageIcon editableIcon = 
     Util.resourceImageIcon("ix-symbol-editable.gif");

  final PriorityPopupMenu priorityPopup = new PriorityPopupMenu();

  private DefaultCellEditor mouseableEditor;
  /** a non-editable text field for mousable string cells **/
  private JTextField noEditTF  = new JTextField(); 

  /** 
   * Says which action to use when an item has more than one that
   * have the same description.  The two keys are the item and
   * the description.
   */
  private TwoKeyHashMap actionShadowingTable = new TwoKeyHashMap();


  /**
   * Constructs a viewer for the indicated agent.
   */
  public AgendaTableViewer(Ip2 ip2) {
    super();
    this.ip2 = ip2;
    setupAgendaTable();      
    //addMouseListener(new AgendaTableMouser());
    model.setParentsBold(true);
  }

  private void setupAgendaTable() {
    model = new AgendaItemTableModel(this);
    setModel(model);
    setAgendaRenderersNEditors();
    setPreferredScrollableViewportSize(new Dimension(500, 90));
    setRowSelectionAllowed(false);
    setColumnSelectionAllowed(false);
    //Debug.noteln("AgendaTV: done setting up table", this);
  }
      

  private void setAgendaRenderersNEditors() {
    PriorityRenderer pr = new PriorityRenderer(false);
    //pr.enableLink(new HTMLFrame(""));
    pr.setToolTipText("Click for choices");
    setDefaultRenderer(Priority.class, pr);
    setDefaultRenderer(EditablePriority.class, pr);
    setDefaultRenderer(DefaultColourField.class, new NDRenderer(false));
    
    int dc = ((AgendaItemTableModel)getModel()).DESCRIPTION_COL;
    TableColumn descriptCol = getColumnModel().getColumn(dc);

    mouseableEditor = new DefaultCellEditor(noEditTF);

    descriptCol.setCellEditor(mouseableEditor);
    setDefaultEditor(String.class, mouseableEditor);
    noEditTF.setEditable(false);
    noEditTF.addMouseListener(this);
    //mouseableEditor.getComponent().addMouseListener(this); //same as above
    //noEditTF.setBackground(Color.lightGray);
    noEditTF.setBackground(Color.white);
  }


  public void setItemEditor(ItemEditor itemEditor) {
    this.itemEditor = itemEditor;
  }
  public void ensureItemEditor() {
    if (itemEditor == null) itemEditor = makeItemEditor();
    itemEditor.setVisible(true);
  }

  abstract protected ItemEditor makeItemEditor();

  /**
   * Sets the viewer back to something approximating its initial state.
   */
  public synchronized void reset() {
    clearTable();
    invalidate();
  }

  /**
   * Clears what's displayed without resetting anything else.
   * NOTE: the model clears its rows and nodes and notifies listeners.
   */
  public synchronized void clearTable() {
    model.clearData();
  }

  /**
   * Clears the table then puts everything back, taking account
   * of any changes in open/closed status.
   */
  //***cannot restore after a clear like that!
  /*
  public synchronized void redisplay() {
    clearTable();
    restoreTable();
    invalidate();
    SwingUtilities.getRoot(this).validate();
  }
  */

  /**
   * Undoes a <code>clearTable()</code>, taking into acount any
   * changes in open/closed status.
   */
  /*
  public void restoreTable() {
    Debug.noteln("Restoring item viewing table***");
  }
  */

  public HandlerAction findHandlerAction(AgendaItem item,
					 String shortDescription) {
    HandlerAction shadowingAction
      = (HandlerAction)actionShadowingTable.get(item, shortDescription);
    return (shadowingAction != null)
      ? shadowingAction
      : item.findAction(shortDescription);
  }

  protected void setShadowingAction(AgendaItem item,
				    String shortDescription,
				    HandlerAction action) {
    actionShadowingTable.put(item, shortDescription, action);
  }


  public void actionSelected(AgendaItem item, String actionName) {
    HandlerAction action = findHandlerAction(item, actionName);
    Debug.expect(action != null, "Can't find action", actionName);
    HandlerAction savedAction = model.getHandlerAction(item);
    if (action.equals(savedAction) || 
	(((savedAction == null) || 
	  HandlerAction.NoAction.class.isInstance(savedAction))
	 && HandlerAction.NoAction.class.isInstance(action)) ||
	((savedAction != null) &&
	 actionName.equals(savedAction.getActionDescription())))
      return;

    // The user can mess around with the actions while the status is BLANK
    // but can't get anything to happen unless the status is POSSIBLE.
    // /\/: This now depends on the item and action classes.
    if (item.actionCanBeTakenNow(action)) {
      // Test was: (item.status == Status.POSSIBLE)
      handleItem(item, action);
    }
    // Put selection back to what it was before
    else {
      Debug.noteln("Ignoring action selection");
      JOptionPane
	.showMessageDialog(this, 
			   "The selected action cannot be taken now",
			   "Action selection",
			   JOptionPane.INFORMATION_MESSAGE);
      model.setHandlerAction(item, savedAction);
    }
  }

  public void handleItem(AgendaItem item, HandlerAction action) {
    HandlerAction savedAction = model.getHandlerAction(item);
    model.setHandlerAction(item, action);
    try { agendaManager.handleItem(item, action); }
    catch (Throwable t) {
      Debug.noteln("Exception while handling item", item);
      Debug.noteException(t);
      JOptionPane
	.showMessageDialog(this, new Object[] 
			   {"Problem while handling item", 
			    Debug.foldException(t)},
			   "Problem while handling item",
			   JOptionPane.ERROR_MESSAGE);
      model.setHandlerAction(item, savedAction);
    }
  }


  public void newHandlerAction(AgendaItem item, HandlerAction act) {
    //Ignore the new action if the item staus indicates that it cant be used
    Status status = item.getStatus();
    if (status == Status.COMPLETE 
	|| status == Status.EXECUTING
	|| status == Status.IMPOSSIBLE) 
      return;
    
    // Make sure the user can see which item we're talking about.
    int savedSelected = getSelectedRow();
    
    int row = model.getObjectRow(item);
    try { setRowSelectionInterval(row, row);}
    catch (Exception e) {}
    
    // do we already know about an action with the same description?
    String target = act.getActionDescription();
    String exists = null;
    java.util.List handlerActions = item.getActions();
    for (Iterator i = handlerActions.iterator(); i.hasNext(); ) {
      HandlerAction ha = (HandlerAction)i.next();
      if (!ha.equals(act) && (ha.getActionDescription().equals(target))) {
	exists = ha.getActionDescription();
	break;
      }
    }
    
    if (exists != null) {
      //There is already an action with the same description.
      //See if the user wants the new action or the old one. We assume that
      // if we do nothing we will continue to find the same action as before.
      if (shouldReplaceAction(item, exists)) {
	// The user wants the new action, so shadow the old.
	setShadowingAction(item, act.getActionDescription(), act);
	Debug.expect(findHandlerAction(item, exists) == act);
      }
      else {
	// Check that lookup doesn't get the new action.
	Debug.expect(findHandlerAction(item, exists) != act);
      }
    }
    else {
      //We have a new action that does not match an existing description,
      // so add it to the list of possible ones
      //Do not mention it if it is a "no action" or a "not appliccable"
      //***This should really not mention things that are just created, only 
      //***ones that are added when things are on the agenda already
      String descript = act.getActionDescription();
      Debug.noteln("Item", Util.quote(item.getShortDescription()));
      Debug.noteln(" has new action", descript);
      String noapp = 
	(new HandlerAction.NotApplicable()).getActionDescription();
      // /\/: NoAction no longer has a 0-arg constructor.  [JD 5 Nov 04]
//    String noact = (new HandlerAction.NoAction()).getActionDescription();
      String done = (new HandlerAction.Manual()).getActionDescription();
      /*
      if (!descript.equals(noapp) &&
	  !descript.equals(noact) &&
	  !descript.equals(done) &&
	  !descript.equals("Expand as below"))
	JOptionPane.showMessageDialog(this, new Object[] {
	  "New action " + Util.quote(act.getActionDescription()),
	    "For item " + Util.quote(item.getShortDescription())
	    },
	  "New action",
	  JOptionPane.INFORMATION_MESSAGE);
      */
      //handlerActions.add(act);
    }
    
    // Restore selected item to before highlighting this one
    if (savedSelected >= 0)
      setRowSelectionInterval(savedSelected, savedSelected);
    else clearSelection();
    
    //no need to fire change - the possible actions are not shown in table
  }

  boolean shouldReplaceAction(AgendaItem item, String actDescription) {
    Object[] answers = {"Replace existing action","Ignore new action"};
    Object selected = JOptionPane.showInputDialog(this,	new Object[] {
      "Item " + Util.quote(item.getShortDescription()),
	"Already has an action " + 
	Util.quote(actDescription)},
      "New Action",			// title
      JOptionPane.INFORMATION_MESSAGE,
      null,				// icon
      answers,
      answers[0]);
    if (selected == null) return false;
    else return selected.equals("Replace existing action");
  }



  /*
   * Instructions from the frame menu bar
   */
  
  public void getNewItemFromUser() {
    ensureItemEditor();
    itemEditor.showNewItem();
  }




  //------------------- AgendaViewer implementation -------------------------

  public AgendaItem makeItem(LList pattern) {
    if (agendaManager == null) {
      Debug.noteln("Error: ***************** AgendaManager not set!");
      //(new Throwable()).printStackTrace();
      return null;
    }
    else return agendaManager.makeItem(pattern);
  }

  public void addItem(AgendaItem i) {
    agendaManager.addItem(i);
  }

  public void saveExpansion(Refinement data) {
    ip2.frame.getDomainEditor().saveExpansion(data);
  }

  public void expandItem(AgendaItem i, Refinement instructions) {
    agendaManager.expandItem(i, instructions);
  }


  /**
   * Useful methods for adding test items.  They assume that the
   * text can be parsed w/o error.  Note that the ItemEditor does
   * not call these methods -- they are called only "internally",
   * specifically for test items.
   */
  public void addItem(String text) {
    addItem(agendaManager.makeItem(text));
  }

  public void addItem(Priority priority, String text) {
	// Assumes that the text can be parsed w/o error.
	// Note that the ItemEditor does not call this method --
	// it's called only "internally", for test items.
	AgendaItem i = agendaManager.makeItem(text);
	i.setPriority(priority);
	addItem(i);
  }


  //------------------- AgendaListener implementation ---------------------


  public void setAgendaManager(AgendaManager agendaManager) {
    this.agendaManager = agendaManager;
  }

  public void itemAdded(AgendaEvent event, AgendaItem i) {
    itemAdded(i);
  }
  /**
   * Called when an itemhas been added, adds the item to the table.
   * An AgendaViewer thing - also used by AgendaListener method.
   */
  public synchronized void itemAdded(AgendaItem i) {
    //Debug.noteln("****AgendaViewer adding item", i);
    model.itemAdded(i);
  }

  public void itemRemoved(AgendaEvent event, AgendaItem i) {
    // /\/: What if the item has children?
    //Debug.noteln("Viewer removing item", i);
    model.itemRemoved(i);
  }

  public void itemHandled(AgendaEvent ae, AgendaItem item, HandlerAction act) {
    // Called when an item has been handled in some way other than
    // the user selecting an action from the menu - in particular when
    // the user has specified a manual expansion.  The JComboBox's
    // listener will be told when we set the selected item and may
    // need to know this is not the usual case.  At present, we let
    // it know by disabling the combobox.  /\/
    model.itemHandled(ae, item, act);
  }

  public void newBindings(AgendaEvent event, Map bindings) {
    //This works but is a bit too sweeping - perhaps better to go through
    // the rows and only change ones that contain variables in question
    //Debug.noteln("Viewer binding variables");
    model.fireTableChanged();
  }


  //------------------- Actions -----------------------------------------------

  public void editingCanceled(ChangeEvent e) {
    //Debug.noteln("ATV: editing canceled");
    super.editingCanceled(e);
    model.fireTableChanged();
  }
  public void editingStopped(ChangeEvent e) {
    //Debug.noteln("ATV: editing stopped");
    super.editingStopped(e);
    model.fireTableChanged();
  }




  /** Hack to make things start properly */
    /*
  public boolean editCellAt(int row, int column, EventObject e) {
      Debug.noteln("AgTV: edit cell at");
    //if not already editing this one, start up as normal
    if (!isEditing() || 
	!((row == getEditingRow()) && (column == getEditingColumn()))) {
      boolean edit = super.editCellAt(row, column, e);
      return edit;
    }
    return true;
  }
    */

  //------ TableModelListener adaptation----------

  public boolean isDummyEditing(int row, int column) {
    //returns true if a field in the row is being dummy-edited (to make it
    // selectable)
    if (isEditing() && getCellEditor().equals(mouseableEditor) 
	&& (row == getEditingRow()) && (column == getEditingColumn())) {
      return true;
    }
    else return false;
  }

  //--------- Mouse listener things ----------------------------------
  private void listenToNoEditTF() {
    MouseListener[] mla = noEditTF.getMouseListeners();
    java.util.List mll = Arrays.asList(mla);
    if ((mll == null) || (!mll.contains(this)))
      noEditTF.addMouseListener(this);
  }

  /**
   * If the left button was clicked on the tree-column, expand/collapse node
   * as per model.
   * If the left button was clicked in a non-editable column, start dummy
   * editing to make the text mouseable.
   */
  public void mouseClicked(MouseEvent me) {
    //if (me.getClickCount() == 1) tryURL();
    //Debug.noteln("AgTV: mouse clicked");
    super.mouseClicked(me);
  }

  
  /**
   * If the right button was pressed in any column bring up the item
   * popup menu.
   * If the left button was pressed on priority, show priority popup menu.
   * If the left button was pressed on Action, show action popup menu.
   */
  public void mousePressed(MouseEvent me) {
    //these do not need the row to be selected - work on where pressed
    int row = rowAtPoint(me.getPoint());
    int column = columnAtPoint(me.getPoint());
    //if (isDummyEditing(row, column)) return; //leave it alone!
    AgendaItem item = (AgendaItem)model.getRowObject(row);
    //Debug.noteln("AgendaTableViewer has mouse event", me.paramString());
    //Debug.noteln(" row " + row + " column ", column);
    if (item == null) {
      Debug.noteln("Pressed on empty item");
      return;
    }
    else if (SwingUtilities.isRightMouseButton(me)) {
      //Debug.noteln("ATV: right pressed in component", me.getComponent());
      ItemPopupMenu popup;
      if (this.equals(me.getComponent())) {
	popup = new ItemPopupMenu(item);
	popup.noticeItemState();
	popup.show(me.getComponent(), me.getX(), me.getY());
      }
      else {
	//Debug.noteln("AgTV: right press in component", me.getComponent());
	//this happens when the event is generated by the noEditTextField;
	if (isEditing()) {
	  row = getEditingRow();
	  column = getEditingColumn();
	  item = (AgendaItem)model.getRowObject(row);
	  popup = new ItemPopupMenu(item);
	  URL url = makeURLFromSelection();
	  if (url != null) {
	    popup.noteURL(url);
	  }
	  popup.noticeItemState();
	  popup.show(me.getComponent(), me.getX(), me.getY());
	}
      }
    }
    else if (SwingUtilities.isLeftMouseButton(me)) {
      if ((item != null) && item.isNew()) item.setIsNew(false); 
      //don't listen to left mouse in text fields except to un-New item
      if (!this.equals(me.getComponent())) return; 
      // Map from table column to model column in case a column
      // has been moved or removed.  [JD 17 Dec 03]
      TableColumn col = getColumnModel().getColumn(column);
      int modCol = col.getModelIndex();
      if (modCol == model.PRIORITY_COL) {
	//Debug.noteln("ATV: priority popup");
        if (model.takesPriority(row)) {
	  priorityPopup.show(me, row, column);
	  priorityPopup.
	      setSelected(Strings.capitalize(item.getPriority().toString()));
	}
      }
      else if (modCol == model.ACTION_COL) {
        //Debug.noteln("ATV: handler action popup");
        if (model.takesAction(row)) {
	  ActionPopupMenu apm = new ActionPopupMenu(this, row, column);
	  apm.show(me, row, column);
	}
      }
      else if ((modCol == model.DESCRIPTION_COL)
	       || (modCol == model.COMMENTS_COL)) {
	if (!isDummyEditing(row, column)) {
	  if (isEditing()) removeEditor();
	  listenToNoEditTF(); //noEditTF.addMouseListener(this);
	  editCellAt(row, column);
	}
      }
    }
  }
  public void mouseEntered(MouseEvent me){}
  public void mouseExited(MouseEvent me){}
  public void mouseReleased(MouseEvent me){}

  private URL makeURLFromSelection() {
    if (!isEditing()) return null;
    int row = getEditingRow();
    int col = getEditingColumn();
    Component c = getEditorComponent();
    String text = "";
    if (c instanceof JTextComponent) 
      text = ((JTextComponent)c).getSelectedText();
    else {
      Object val = model.getValueAt(row, col);
      if (val != null) text = val.toString();
    }
    try {
      URL url = UIUtil.resourceURL(text);
      if (url == null) url = new URL(text);
      return url;
    }
    catch (Exception e) {
      //Debug.noteln("Cannot do a URL for text", text);
      //Debug.noteException(e);
    }
    return null;
  }

  
    //-------------------inner classes---------------


  private class PriorityPopupMenu extends ix.iface.ui.table.TablePopupMenu 
    implements ActionListener 
  {
    public PriorityPopupMenu() {
      super(AgendaTableViewer.this);
      // The values are lowest-first, so we reverse them.
      java.util.List values = new ArrayList(Priority.values());
      Collections.reverse(values);
      for (Iterator i = values.iterator(); i.hasNext();) {
	Priority p = (Priority)i.next();
	JMenuItem item = 
	  add(new DefaultColourField(Strings.capitalize(p.toString()),
				     p.getColor()));
	item.addActionListener(new CatchingActionListener(this));
      }
      pack();
    }

    public void actionPerformed(ActionEvent e) {
      String command = e.getActionCommand();
      Debug.noteln("Priority popup command", command);
      if (command == null) return;
      else model.setPriorityValue(command, row, column);
    }
  }

  private class ActionPopupMenu extends ix.iface.ui.table.TablePopupMenu 
    implements ActionListener 
  {
    AgendaItem agendaItem;
    String currentAction = "No Action";

    public ActionPopupMenu(JTable table, int row, int column) {
      super(table);
      //setBorderPainted(true); //seems to make no difference
      AgendaItemTableModel mainModel = (AgendaItemTableModel)table.getModel();
      agendaItem = (AgendaItem)mainModel.getRowObject(row);
      AgendaItemTableModel.TreeAgendaItem tai = 
	(AgendaItemTableModel.TreeAgendaItem)mainModel.getTreeNode(agendaItem);
    
      if (tai.handlerAction != null) 
	currentAction = tai.handlerAction.getActionDescription();

      ArrayList items = new ArrayList(); //the items that really go in the list
      //make up a list of things that should really go into the editor
      for (Iterator i = agendaItem.getActions().iterator(); i.hasNext(); ) {
	//get the next action in the list
	HandlerAction act = (HandlerAction)i.next();
	//make sure this is the one we are using (user works with descriptions)
	String description = act.getActionDescription();
	act = ((AgendaTableViewer)table).findHandlerAction(agendaItem, 
							   description);
	//do not add it again if it is already there
	if (!items.contains(act)) items.add(act);
      }

      for (Iterator i = items.iterator(); i.hasNext(); ) {
	//get the next action in the list
	HandlerAction act = (HandlerAction)i.next();
	//Set the colour to the same as the item status or grey if not ready
	Color color;
	if (act.isReady()) color = agendaItem.getStatus().getColor();
	else color = Color.gray;
	JMenuItem jmi = 
	  add(new DefaultColourField(act.getActionDescription(), color));
	if (currentAction.equals(act.getActionDescription()))
	  setSelected(jmi);
	jmi.addActionListener(new CatchingActionListener(this));
      }
      pack();
    }

    public void actionPerformed(ActionEvent e) {
      String command = e.getActionCommand();
      actionSelected(agendaItem, command);
    }
  }


  private class ItemPopupMenu extends AbstractAgendaItemPopupMenu
          implements ActionListener 
  {
    private URL url = null;

    ItemPopupMenu(AgendaItem item) {
      super(AgendaTableViewer.this.ip2, item);
    }

    void showDetails() {
      ensureItemEditor();
      //Debug.noteln("showDetails item is", item);
      itemEditor.showItem(item);
    }

    void fold() {
      IXTreeTableNode node = (IXTreeTableNode)model.getTreeNode(item);
      int row = model.getObjectRow(item);
      model.unexpandNode(node, row);
    }
    void unfold() {
      IXTreeTableNode node = (IXTreeTableNode)model.getTreeNode(item);
      int row = model.getObjectRow(item);
      model.expandNode(node, row);
    }

    boolean isOpen() {
      IXTreeTableNode node = (IXTreeTableNode)model.getTreeNode(item);
      return node.expanded;
    }

    public void noteURL(URL url) {
      this.url = url;
      if (url != null) {
	makeMenuItem("Show URL");
	add(makeMenuItem("Show URL"));
      }
    }
    private boolean showURL() {
      //Debug.noteln("showing url");
      if (url == null) return false;
      try {
	HTMLFrame hf = new HTMLFrame(url);
	hf.setVisible(true);
	return true;
      }
      catch (Exception e) {
	//Debug.noteln("Cannot do a URL for text", text);
	//Debug.noteException(e);
	return false;
      }
    }
    public void actionPerformed(ActionEvent e) {
      String command = e.getActionCommand();
      if (command.equals("Show URL")) {
	Debug.noteln("Item popup command", command);
	if (showURL()) return;
	else 
	  JOptionPane.showMessageDialog(null, 
					"Cannot show URL " + url.toString());
      }
      else super.actionPerformed(e);
    }
  }
}

// Issues:
// * What should be done when removing an item that has children?
//   Don't allow it?  Remove descendents as well?
