/****************************************************************************
 * A TreeTableModel that can display Issue objects.
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Thu Mar 16 11:21:25 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.ip2;

import ix.util.*;
import ix.util.lisp.*;
import ix.icore.*;
import ix.iface.ui.table.*;
import ix.iface.ui.*;
import ix.ip2.event.*;
import ix.iview.util.*;


import java.awt.Font;
import java.awt.Color;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.tree.*;
import java.util.*;


/****************************************************************************
 * A TreeTableModel that can display Issue objects.
 *****************************************************************************
 */
public class AgendaItemTableModel extends AbstractTreeTableModel 
//implements AgendaListener// , AgendaItemListener
//AgendaListener done by Viewer, AgendaItemListener done automatically via this
{

  protected AgendaTableViewer table;
  //The columns start with 2 because the TreeTable adds the mark
  //column and the fold/unfold icons in column 0 and 1 - most methods in
  //this class ignore 0 and 1.
  private int offset = 2;
  public int DESCRIPTION_COL = 0 + offset;
  public int COMMENTS_COL = 1 + offset;
  public int PRIORITY_COL = 2 + offset;
  public int ACTION_COL = 3 + offset;

  private Color newColour = new Color(0xffcccc);  //Proirity.HIGH.getColor()


  //A set of things that have explicitly been added.
  //Needed because we mustn't listen to children that have not explicitly
  // been added.
  private HashSet items = new HashSet();
  //Things that are added as completed... must not be disabled - only diable
  // if status *changes* to something disabling
  private HashSet disabled = new HashSet();

  public AgendaItemTableModel(AgendaTableViewer table) { 
    super(false); //make it markable or not
    setColumnVars();
    this.table = table;
    Class[] classes = {FourField.class, String.class, Priority.class, 
		       FourField.class};
    String[] names = {"Description", "Annotations", "Priority", "Action"};
    setColumnClasses(classes);
    setColumnNames(names);
  }

  public void setColumnVars() {
    if (rowsMarkable) offset = 2;
    else offset = 1;
    DESCRIPTION_COL = 0 + offset;
    COMMENTS_COL = 1 + offset;
    PRIORITY_COL = 2 + offset;
    ACTION_COL = 3 + offset;
  }



  /*
  public Class getColumnClass(int columnIndex) {
    //if (columnIndex == 0) return ImageIcon.class;
    if (columnIndex == DESCRIPTION_COL) return FourField.class;
    else if (columnIndex == COMMENTS_COL) return String.class;
    else if (columnIndex == PRIORITY_COL) return Priority.class;
    else if (columnIndex == ACTION_COL) return FourField.class;
    else return super.getColumnClass(columnIndex);
  }  
  */

  /**
   * Only adds the node if the parents are empty (deal with sub-nodes in tree).
   * If there is a parent, make sure it is expanded so that new nodes are
   * shown.
   * Also add an itemListener to the item because now it is officially added to
   * the agenda.
   */
  public synchronized void itemAdded(AgendaItem item) {
    //Debug.noteln("***AITM item added", item);
    if (item == null) return;

    //add to table if it is a root otherwise expand parent
    AgendaItem parent = item.getParent();
    IXTreeTableNode parentTai = null;
    if (parent == null) {
      //Debug.noteln("AITM: adding new toplevel item");
      addData(item);
    }
    else {
      parentTai = (IXTreeTableNode)getTreeNode(parent);
      //Debug.noteln("AgITM: looking for node", parentTai);
      int row = rows.indexOf(parentTai);
      if (row == -1) row = getObjectRow(parent);
      //Debug.noteln("AgITM: got object row", row);
      expandNode(parentTai, row);
    }
    //officially add the item (start listening to it!)
    TreeAgendaItem tai = (TreeAgendaItem)getTreeNode(item);
    //Debug.noteln("AITM: Adding litener to item", item.getShortDescription());
    //Debug.noteln("***   TAI: ", tai.toDebug());      
    //tai.loadHandlerActions();
    item.addItemListener(tai);
    //remember the item was added
    items.add(item);
    fireNodeStructureChanged(parentTai);
  }

  public synchronized void itemRemoved(AgendaItem i) {
    // /\/: What if the item has children?
    Debug.noteln("Agenda Model removing item", i);
    TreeAgendaItem tai = (TreeAgendaItem)nodeMap.get(i);
    if (tai == null) Debug.noteln("Removing item: no such item", i);
    else {
      super.removeData(i);
      nodeMap.remove(i);
      items.remove(i);
      disabled.remove(i);
    }
    
  }

  public void clearData() {
    items.clear();
    super.clearData();
  }


  //-------------------------- editing cells stuff

  /**
   * Gets the name string of the object (row)
   * @return the string to use as a name for the object.
   */
  public String getNameString(TreeNode item) {
    return ((TreeAgendaItem)item).node.getShortDescription();
  }


  protected Object getTreeValueAt(IXTreeTableNode eNode) {
    AgendaItem item = ((TreeAgendaItem)eNode).node;
    //Debug.noteln("AgendaITM: doing tree value", eNode);
    DefaultColourField ff = (DefaultColourField)super.getTreeValueAt(eNode);
    if (item.isNew()) ff.setColour(newColour);
    return ff;
  }

  /**
   * Gets the cell value for normal cells (only one - the pattern).
   * The name cell (description) is done in the method getTreeValueAt.
   * @return the value as an object, null for illegal columns.
   */
  public Object getCellValueAt(Object o, int columnIndex) {
    //Debug.noteln("AITM***********: doing value", o);
    //Debug.noteln(" in column", columnIndex);
    AgendaItem item = ((TreeAgendaItem)o).node;
    //column index will be lower than expected because the TreeTable
    //takes off the marking column and the icon for
    //expand/collapse. The difference is <offset>
    if (columnIndex == COMMENTS_COL-offset) { //comments
	//Debug.noteln(" comments");
      //If there are reports, take the first like of the last one
      // otherwise take the first line of the comments
      List reports = item.getReports();
      String text;
      if ((reports == null) || reports.isEmpty())
	text = item.getComments();
      else {
	Report lastRep = (Report)reports.get(reports.size() - 1);
	text = lastRep.getText();
      }
      String value = Strings.firstLine(text).trim();
      return value;
    }
    else if (columnIndex == PRIORITY_COL-offset) { //priority
	//Debug.noteln(" priority");
      TreeAgendaItem tai = (TreeAgendaItem)o;
      boolean editable = tai.takesPriority();      
      return new EditablePriority(item.getPriority(), editable);
    }
    else if (columnIndex == ACTION_COL-offset) { //action
	//Debug.noteln(" action");
      TreeAgendaItem tai = (TreeAgendaItem)o;
      HandlerAction act = tai.handlerAction;
      Color sColour = item.getStatus().getColor();
      String action;
      if (act == null) {
	List acts = item.getActions();
	if ((acts != null) && !acts.isEmpty())
	  act = (HandlerAction)acts.get(0);
      }
      if (act == null) action = "No Action";
      else action = act.getActionDescription();
      boolean editable = tai.takesAction();
      //Debug.noteln("AITM: node is", item.getShortDescription());
      //Debug.noteln(" tai is", tai.toDebug());
      //Debug.noteln(" status is", item.getStatus());
      //Debug.noteln(" editable is", new Boolean(editable));
      return new EditableColourField(action, sColour, editable);
    }
    else return null;
  }

  public boolean isCellEditable(int row, int col) {
    //Debug.noteln("AITM: checking editable for col", col);
    if ((col == DESCRIPTION_COL) || (col == COMMENTS_COL)) return true;
    else return super.isCellEditable(row, col);
  }

  public boolean takesPriority(int row) {
    TreeAgendaItem tai = (TreeAgendaItem)getRowNode(row);
    return (tai.takesPriority());
  }
  public boolean takesAction(int row) {
    TreeAgendaItem tai = (TreeAgendaItem)getRowNode(row);
    return (tai.takesAction());
  }

  public void setValueAt(Object value, int row, int col) {
    if (value == null) {
      fireTableCellUpdated(row, col);
      return;
    }
    //Debug.noteln("AITM: got new value ", value.toString());
    AgendaItem item = (AgendaItem)getRowObject(row);
    //Debug.noteln(" for item ", item.toString());
    if (col == PRIORITY_COL) { 
      setPriorityValue(item, value, row, col);
    }
    else if (col == ACTION_COL) { 
      String actionName = value.toString();
      //Debug.noteln("Selected action", actionName);
      actionSelected(item, actionName);
    }
    else super.setValueAt(value, row, col);
    fireTableCellUpdated(row, col);
  }

  public void setPriorityValue(AgendaItem item, Object val, int row, int col) {
    try {
      Priority p = Priority.valueOf(Strings.uncapitalize(val.toString()));
      if (!p.equals(item.getPriority())) {
	item.setPriority(p);
	fireTableCellUpdated(row, col);
      }
    }
    catch (IllegalArgumentException iae) {
      Debug.noteException(iae);
    }
  }
  public void setPriorityValue(Object value, int row, int col) {
    setPriorityValue((AgendaItem)getRowObject(row), value, row, col);
  }

  /**
   * only call this when the action has been selected in the table - not if
   * it was set elsewhere (see itemHandled method)
   */
  private void actionSelected(AgendaItem item, String actionName) {
    table.actionSelected(item, actionName);
  }

  public void itemHandled(AgendaEvent ae, AgendaItem item, HandlerAction act) {
    // Called when an item has been handled in some way other than
    // the user selecting an action from the menu - in particular when
    // the user has specified a manual expansion.  The JComboBox's
    // listener will be told when we set the selected item and may
    // need to know this is not the usual case.  At present, we let
    // it know by disabling the combobox.  /\/
    Debug.noteln("AITM: Handled item", item.getShortDescription());
    TreeAgendaItem tai = (TreeAgendaItem)getTreeNode(item);
    // /\/: We don't want to make action-selection any more disabled
    // than it would be if the user selected the action.  Also,
    // some actions want to leave the item in a state that allows
    // further actions to be selected.  [JD, 5 Nov 04]
//  disabled.add(item);
//  tai.disableActions();
    tai.setHandlerAction(act);
  }



  //--------------------------


  public void setHandlerAction(AgendaItem item, HandlerAction action) {
    TreeAgendaItem tai = (TreeAgendaItem)getTreeNode(item);
    tai.setHandlerAction(action);
  }
  public HandlerAction getHandlerAction(AgendaItem item) {
    TreeAgendaItem tai = (TreeAgendaItem)getTreeNode(item);
    return tai.handlerAction;
  }

  /*** This should not be necessary and is dangerous, but it stops
   * the column widths from getting changed when rows are deleted.
   */
  public void fireTableStructureChanged() {}

  public void fireTableChanged() {
    fireTableDataChanged();
  }
  public void fireItemChanged(TreeAgendaItem treeItem) {
    if (treeItem == null) return;
    int row = getObjectRow(treeItem.node);
    int col = table.getEditingColumn();
    if (table.isDummyEditing(row, col)) {
      fireTableRowsUpdated(row, row);
      table.editCellAt(row, col);
    }
    else fireTableRowsUpdated(row, row);
  }
  public void fireActionSelected(AgendaItem item, HandlerAction act) {
    if (item == null) return;
    int row = getObjectRow(item);
    TableModelEvent tme = new TableModelEvent(this, row, row, ACTION_COL);
    fireTableChanged(tme);
  }
  public void fireItemDeleted(AgendaItem item) {
    int row = getObjectRow(item);
    fireTableRowsDeleted(row, row);
  }



  //--------------------TreeNode vs AgendaItem things--------------------------

  public IXTreeTableNode makeIXTreeTableNode(Object original) {
    if (original == null) {
      Debug.noteln("AITM: Cannot make a table node for null objects");
      return null;
    }
    else {
      //Debug.noteln("AgItTM: original of class", original.getClass());
      return new TreeAgendaItem(this, (AgendaItem)original);
    }
  }


  //-------------------- AgendaItem class

  protected class TreeAgendaItem extends IXTreeTableNode 
    implements AgendaItemListener
  {
    public AgendaItem node; //the agenda item that this node wraps
    //the action currently associated with the item - default: no action
    public HandlerAction handlerAction; //new HandlerAction.NoAction();
    public List handlerActions;
    private boolean takesAction = true;
    private boolean takesPriority = true;

    public TreeAgendaItem(AgendaItemTableModel model, AgendaItem agendaItem) {
      super(model, agendaItem);
      node = agendaItem;
      //loadHandlerActions();
      //Debug.noteln("AITM: making TAI for node", node.getShortDescription());
      //Debug.noteln("             TAI: ", this.toDebug());      
      if (agendaItem == null) {
	Debug.noteln("Making TreeAgendaItem with null AgendaItem");
	(new Throwable()).printStackTrace();
      }
      else noticeStatus();
      //if it had been added explicitly, add its listener
      if (items.contains(agendaItem))
	agendaItem.addItemListener(this);
    }

    //enables/disables the action choice
    public boolean takesAction() { return takesAction; }
    public void disableActions() { 
      //Debug.noteln("AITM: disabling act - node", node.getShortDescription());
      //Debug.noteln(" tai is", this.toDebug());
      takesAction = false; 
    }
    public void enableActions() { 
      //Debug.noteln("AITM: enabling act - node", node.getShortDescription());
      takesAction = true; 
    }

    //enables/disables the priority choice
    public boolean takesPriority() { return takesPriority; }
    public void disablePriority() { takesPriority = false; }
    public void enablePriority() { takesPriority = true; }

    protected void loadHandlerActions() {
      handlerActions = new ArrayList();
      //make up a list of things that should really go into the editor
      for (Iterator i = node.getActions().iterator(); i.hasNext(); ) {
	//get the next action in the list
	HandlerAction act = (HandlerAction)i.next();
	//make sure this is the one we are using (user works with descriptions)
	String description = act.getActionDescription();
	act = table.findHandlerAction(node, description);
	//do not add it again if it is already there
	if (!handlerActions.contains(act)) handlerActions.add(act);
      }
    }
    protected void noticeStatus() {
      Status status = node.getStatus();
      //Debug.noteln("AITM: node is", node.getShortDescription());
      //Debug.noteln(" status is", status.toString());
      if ((status == Status.COMPLETE
	   || status == Status.EXECUTING
	   || status == Status.IMPOSSIBLE)
	  && disabled.contains(node)) {
	//Debug.noteln(" Disabling actions and priority");
	disableActions();
	disablePriority();      
      }
    }

    public void setHandlerAction(HandlerAction action) {
      handlerAction = action;
      fireItemChanged(this);
    }


    //---------------TreeNode implementation

    public Enumeration children() { //one level deep only
      final List nodes = node.getChildren();
      return new Enumeration() {
	int i = nodes.size()-1;
	public boolean hasMoreElements() { return i >= 0; }
	public Object nextElement() { return nodes.get(i--); }
      };
    }
    public boolean getAllowsChildren() { 
      return true; 
    }
    public TreeNode getChildAt(int index) { 
      Object item = node.getChildren().get(index);
      return model.getTreeNode(item);
      //return new TreeAgendaItem();
    }
    public int getChildCount() { 
      if (node.getChildren() == null) return 0;
      else return node.getChildren().size(); 
    }
    public int getIndex(TreeNode treeNode) {
      List nodes = node.getChildren();
      return nodes.lastIndexOf(((IXTreeTableNode)treeNode).node);
    }
    public TreeNode getParent() { 
      return model.getTreeNode(node.getParent()); 
    }
    public boolean isLeaf() { 
      if (node == null) {
	Debug.noteln("AITM: null node - should not happen!");
	return true;
      }
      else if (node.getChildren() == null) {
	  //Debug.noteln("AITM: null children");
	return true;
      }
      else {
	return (node.getChildren().size() == 0);
      }
    }


    //---------AgendaItemListener changes

    public void statusChanged(AgendaItemEvent e) {
      //Debug.noteln("AITM: got status change", node.getShortDescription());
      disabled.add(node);
      noticeStatus();
      fireItemChanged(this);
    }
    public void priorityChanged(AgendaItemEvent e) { 
      //Debug.noteln("AITM: got priority change");
      fireItemChanged(this);
    }
    public void newHandlerAction(AgendaItemEvent ae, HandlerAction act) {
      //Debug.noteln("AITM: got new handler action");
      table.newHandlerAction(node, act);
    }
    public void handlerActionsChanged(AgendaItemEvent e) {
      //loadHandlerActions();
      fireItemChanged(this);
    }

	
    public void newReport(AgendaItemEvent e, Report report) {
      //Debug.noteln("AITM: got new report change");
      fireItemChanged(this);
    }
    public void agendaItemEdited(AgendaItemEvent e) {
      //Debug.noteln("AITM: got item change");
      fireItemChanged(this);
    }

  }

}

  
