/****************************************************************************
 * An abstract class for table models.
 *
 * @author Jussi Stader
 * @version 1.0
 * Updated: Mon Jun 26 12:12:08 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui.table;

//import ix.iface.ui.*;
import ix.iface.ui.util.UIUtil;
import ix.util.Debug;
//import ix.util.*;

import java.util.*;
import java.lang.reflect.*;
import javax.swing.table.*;

/****************************************************************************
 * An abstract class for table models.
 *
 * Keeps track of the base objects that are shown in the table.
 * Implements most functions required by JTable and IXTable objects to work.
 * 
 * Implement the following functions (see documentation below):
 * <ul>
 * <li> protected Object getCellValueAt(Object baseObject, int columnIndex)
 * </ul>
 *****************************************************************************
 */
public abstract class AbstractIXTableModel extends AbstractTableModel {
  protected ArrayList columnFields = new ArrayList();
  protected ArrayList columnNames = new ArrayList();
  protected ArrayList columnClasses = new ArrayList();
  protected Class objectClass;
  protected boolean rowsMarkable = false;
  /** contains data objects. cannot do rows because it gets confused
   * with expand/collapse in tree tables! */
  protected HashSet markedRows;


  /** a list of objects that appear in the table */
  protected List rows = new LinkedList();

  
  /**
   * Makes an empty model.
   */
  public AbstractIXTableModel(boolean markable) {
    super();
    rowsMarkable = markable;
    if (markable) markedRows = new HashSet();
  }

  /**
   * Makes a model that contains the given nodes.
   */
  public AbstractIXTableModel(boolean markable, Object[] theRows) {
    super();
    rowsMarkable = markable;
    if (markable) markedRows = new HashSet();
    setData(theRows);
  }

  /**
   * Makes a model as above but with the given strings as column names.
   */
  public AbstractIXTableModel(boolean markable, Object[] theRows, 
			      String[] columns) {
    this(markable, theRows);
    setColumnNames(columns);
  }

  /**
   * Makes a model that contains the given nodes and derives which fields
   * to use. 
   */
  public AbstractIXTableModel(boolean markable, Class theClass, 
			      Object[] theRows) {
    this(markable, theRows);
    objectClass = theClass;
    deriveColumnFields();
  }

  /**
   * Sets the table's rows to the given data objects and notifies listeners.
   * Clears markings on previous rows.
   */
  public void setData(Object[] data){
    rows.clear();
    clearMarks();
    if (data != null) 
	for (int i = 0; i < data.length; i++) {
	    rows.add(data[i]);
	}
    fireTableDataChanged();
  } 

  public List getData() {
    return rows;
  }

  /*
  private Object findData(Object data) {

  }
  */
          
  /**
   * Adds the given data object to the table's rows if it is not already there
   * and notifies listeners.
   */
  public void addData(Object data) {
    if (!rows.contains(data)) {
      rows.add(data);
      fireTableDataChanged();
    }
  }
  /**
   * Adds the given data object to the table's rows at the given
   * position if it is not already there and notifies listeners. 
   */
  public void addData(int row, Object data) {
    if (!rows.contains(data)) {
      rows.add(row, data);
      fireTableDataChanged();
    }
  }
  /**
   * Removes the given data object from the table's rows and notifies
   * listeners if it worked.
   */
  public void removeData(Object data) {
    int row = getObjectRow(data);
    unmarkRow(row);
    if (rows.remove(data)) fireTableDataChanged();
  }

  /**
   * Replaces the given old data object with the new given one and notifies
   * listeners if it worked.
   */
  public void replaceData(Object oldData, Object newData) {
    int row = getObjectRow(oldData);
    if ((0 <= row) && (row < getRowCount())) {
	boolean marked = isMarked(row); //remember marking
	unmarkRow(row);
	rows.remove(oldData);
	if (rows.contains(newData))
	    row = getObjectRow(newData);
	else rows.add(row, newData);
	if (marked) markRow(row);
	fireTableDataChanged();
    }
  }
  /**
   * Removes all data objects from the table's rows and notifies listeners.
   */
  public void clearData() {
    rows.clear();
    clearMarks();
    fireTableDataChanged();
  }

  /** 
   * Gets the object that is in the given row.
   * @return the object or null if there is no such row.
   */
  public Object getRowObject(int rowIndex) {
    if (rowIndex < 0) return null;
    else return rows.get(rowIndex);    
  }

  /** 
   * Gets the row index of the given object
   * @return the object's index or -1 if there is no such object.
   */
  public int getObjectRow(Object object) {
    if ((object == null) || (rows == null)) return -1;
    return rows.indexOf(object);    
  }


  /**
   * Gets the value for a cell in the table defined by the given indices.
   * Looks up the object and calls the abstract object-index method.
   */ 
  public Object getValueAt(int rowIndex, int columnIndex) { 
    Object o = rows.get(rowIndex); 
    if (rowsMarkable) {
      if (columnIndex == 0) return new Boolean(isMarked(rowIndex));
      else return getCellValueAt(o, columnIndex-1);
    }
    return getCellValueAt(o, columnIndex);
  }

  /** JTable required TableModel function. */
  public int getRowCount() {
    return rows.size();
  }
  /** JTable required TableModel function. */
  public int getColumnCount() {
    String[] names = getColumnNames();
    return names.length;
  }
  /** JTable required TableModel function. */
  public String getColumnName(int index) {
    String[] names = getColumnNames();
    return names[index];
  }
  /** JTable required TableModel function (default is Object). */
  public Class getColumnClass(int index) {
    Class colClass = null;
    try {
      colClass = (Class)columnClasses.get(index);
    }
    catch (IndexOutOfBoundsException e) {
      colClass = super.getColumnClass(index);
    }
    return colClass;
  }

  /**
   * Gets the object's value for the given column index.
   * @return an object that can be rendered by the table.
   */
  public abstract Object getCellValueAt(Object baseObject, int columnIndex);

  public boolean isCellEditable(int row, int col) {
    if (rowsMarkable && (col == 0)) return true;
    else return super.isCellEditable(row, col);
  }

  public void setValueAt(Object value, int row, int col) {
    if (rowsMarkable && (col == 0)) {
      if (((Boolean)value).booleanValue()) markRow(row);
      else unmarkRow(row);
    }

  }
  /**
   * Sets the column titles to be used in the table
   */
  public void setColumnNames(String[] fieldNames) {
    columnNames = new ArrayList(Arrays.asList(fieldNames));
    if (rowsMarkable) columnNames.add(0,"");
  }
  /**
   * Sets the column classes of the table
   */
  public void setColumnClasses(Class[] fieldClasses) {
    columnClasses = new ArrayList(Arrays.asList(fieldClasses));
    if (rowsMarkable) columnClasses.add(0, Boolean.class);
  }
  /**
   * Gets the strings to use as column names.
   */
  protected String[] getColumnNames() {
    String[] s = {};
    return (String[])columnNames.toArray(s);
  }
  /**
   * Sets the fields to be used as columns in the table
   */
  public void setColumnFields(Class objectClass, String[] fieldNames) {
    for (int i=0; i<fieldNames.length; i++) {
      String fieldName = fieldNames[i];
      try {
	Field field = objectClass.getField(fieldName);
	String title = UIUtil.toCapitalised(fieldName);
	Class fieldClass = field.getType();
	columnFields.add(fieldName);
	columnNames.add(title);
	columnClasses.add(fieldClass);
      }
      catch (Exception e) {
	Debug.noteln("AIXTM: Problem setting column field " + fieldName);
	Debug.noteException(e);
      }
    }
    if (rowsMarkable) {
      columnFields.add(0,"rowMarked");
      columnClasses.add(0, Boolean.class);
      columnNames.add(0,"");      
    }
  }


  //--------------------------marking rows-----------------------------------

  public boolean isMarkable() {
    return rowsMarkable;
  }

  public void markRow(int row) {
    if (markedRows == null) return;
    //markedRows.add(new Integer(row));
    Object ro = getRowObject(row);
    markedRows.add(ro);
    fireTableMarkAdded(ro);
  }
  public void unmarkRow(int row) {
    if (markedRows == null) return;
    Object ro = getRowObject(row);
    markedRows.remove(ro);
    fireTableMarkRemoved(ro);
  }
  public void clearMarks() {
    if (markedRows == null) return;
    markedRows.clear();
    fireTableMarkCleared();
  }
  public void markAll() {
    if (markedRows == null) return;
    for (int i = 0; i < getRowCount(); i++) markedRows.add(getRowObject(i));
    fireTableMarkedAll();
  }

  public boolean isMarked(int row) {
    if (markedRows == null) return false;
    return markedRows.contains(getRowObject(row));
  }
  public boolean isMarkedEmpty() {
    if (markedRows == null) return true;
    return markedRows.isEmpty();
  }
  public Collection getMarkedRows() {
    return null;
  }
  public Collection getMarkedObjects() {
    return markedRows;
  }


  //-------------------------mark listeners

  private HashSet markListeners = new HashSet();
  public void addMarkListener(TableMarkListener tml) {
    markListeners.add(tml);
  }
  public void removeMarkListener(TableMarkListener tml) {
    while (markListeners.contains(tml)) {
      markListeners.remove(tml);
    }
  }
  public void fireTableMarkAdded(Object rowObject) {
    for (Iterator i = markListeners.iterator(); i.hasNext(); )  {
      TableMarkListener tml = (TableMarkListener)i.next();
      tml.tableMarkAdded(rowObject);
    }
  }
  public void fireTableMarkRemoved(Object rowObject) {
    for (Iterator i = markListeners.iterator(); i.hasNext(); )  {
      TableMarkListener tml = (TableMarkListener)i.next();
      tml.tableMarkRemoved(rowObject);
    }
  }
  public void fireTableMarkCleared() {
    for (Iterator i = markListeners.iterator(); i.hasNext(); )  {
      TableMarkListener tml = (TableMarkListener)i.next();
      tml.tableMarkCleared();
    }
  }
  public void fireTableMarkedAll() {
    for (Iterator i = markListeners.iterator(); i.hasNext(); )  {
      TableMarkListener tml = (TableMarkListener)i.next();
      tml.tableMarkedAll();
    }
  }

  public interface TableMarkListener {
    public void tableMarkAdded(Object rowObject);
    public void tableMarkRemoved(Object rowObject);
    public void tableMarkedAll();
    public void tableMarkCleared();
  }



 


  /*-------------------------------------------------------------------------
   *
   *        Useful methods for generic tables
   *
   *-------------------------------------------------------------------------
   */

  /**
   * Collects usable fields from the objectClass 
   */
  protected void deriveColumnFields() {
    if (objectClass == null) return;
    //class fields
    ArrayList fields = new ArrayList(Arrays.asList(objectClass.getFields()));
    Iterator e = fields.iterator();
    while (e.hasNext()) {
      Field field = (Field)e.next();
      if (useField(field)) {
	columnFields.add(field.getName());
	columnNames.add(UIUtil.toCapitalised(field.getName()));
      }
    }
    if (rowsMarkable) {
      columnFields.add(0,"rowMarked");
      columnClasses.add(0, Boolean.class);
      columnNames.add(0,"");      
    }
  }

  /**
   * Checks whether the given field can be used.
   */
  private static boolean useField(Field field) {
    int mod = field.getModifiers();
    //do not display protected, private, static, or name fields
    return (!Modifier.isStatic(mod) && 
	    !Modifier.isPrivate(mod) &&
	    !Modifier.isProtected(mod));
  }


  /**
   * Sets the fields to be used as table columns
   */
  protected void defaultSetColumnFields(String[] fieldNames) {
    if (objectClass == null) return;
    columnFields.clear();
    if ((fieldNames == null) || (fieldNames.length == 0)) return;
    else {
      for (int i=0; i<fieldNames.length; i++) {
	String name = fieldNames[i];
	try {
	  objectClass.getField(name);
	  columnFields.add(name);
	  columnNames.add(UIUtil.toCapitalised(name));
	}
	catch (Exception e) {
	  Debug.noteException(e);
	  try {
	    String lName = UIUtil.toUncapitalised(name);
	    objectClass.getField(lName);
	    columnFields.add(lName);
	    columnNames.add(name);
	  }
	  catch (Exception ee) {
	    Debug.noteException(ee);
	  }
	}
      }
    }
    if (rowsMarkable) {
      columnFields.add(0,"rowMarked");
      columnClasses.add(0, Boolean.class);
      columnNames.add(0,"");      
    }
  }

  /**
   * Looks up the value in the columnField of the given index.
   * @return the value object or null if something goes wrong.
   */
  protected Object defaultGetCellValueAt(Object o, int columnIndex) {
    if (objectClass == null) return null;
    if ((columnFields == null) || columnFields.isEmpty()) return null;
    try {
      String fieldName = (String)columnFields.get(columnIndex);
      Field field = objectClass.getField(fieldName);
      return field.get(o);
    }
    catch (Exception e) {
      Debug.note("Exception during defaultGetCellValueAt(" + o.toString() + 
		 ", " + columnIndex + ")");
      Debug.noteException(e);
      return null;
    }
  }



}
