/****************************************************************************
 * A simple panel with a label, a collection component, and optional buttons.
 *
 * @author Jussi Stader
 * @version 2.3+
 * Updated: Thu Dec  7 13:05:41 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import java.util.*;
import java.awt.event.*;
import java.lang.reflect.Array;

import ix.util.Debug;
import ix.iface.ui.table.*;
import ix.iface.ui.tree.*;
    
/**
 * A simple panel with a label (title), a component, and optional buttons.
 * The given label will be at the top, the given component at the left and
 * buttons with given labels on the right.
 * Adds the given listener to the buttons and the list.
 */
public class ListEditorPanel extends AbstractEditorPanel 
    implements ActionListener, ListSelectionListener
{
    private HashSet listeners = new HashSet();

  public ListEditorPanel() { super(false);}
  public ListEditorPanel(boolean isBordered) { super(false, isBordered);}
  public ListEditorPanel(EventListener ml, String label, 
			 JComponent component) { 
    this(ml, label, false, component);
  }
  public ListEditorPanel(EventListener ml, String label, String[] buttons) { 
    this(ml, label, false, buttons);
  }
  public ListEditorPanel(EventListener ml, String label, 
			 JComponent component, String[] buttons) { 
    this(ml, label, false, component, buttons);
  }


  public ListEditorPanel(EventListener ml, String label, boolean minimise,
			 JComponent component) { 
    super(ml, label, minimise, component);
    listen(ml);
  }
  public ListEditorPanel(EventListener ml, String label, boolean minimise,
			 String[] buttons) { 
    super(ml, label, minimise, buttons);
    listen(ml);
  }
  public ListEditorPanel(EventListener ml, String label, boolean minimise,
			 JComponent component, String[] buttons) { 
    super(ml, label, minimise, component, buttons);
    listen(ml);
  }

  public boolean isListItem() {return true;}

  /**
   * Gets the current data of a collection component.
   * @return an object that is an array of objects taken from the component.
   */
  public Object getData() {
    Class iClass = itemComponent.getClass();
    if (itemComponent instanceof JList) return getListData();
    else if (itemComponent instanceof JComboBox) return getComboData();
    else if (itemComponent instanceof JTable) return getTableData();      
    else if (itemComponent instanceof JTree) return getTreeData();      
    else {
      Debug.noteln("Cannot get data from object of type",
		   itemComponent.getClass());
      return null;
    }
  }

  private void listen(EventListener ml) {
    if (ml instanceof ListSelectionListener) {
      addListSelectionListener((ListSelectionListener)ml);
    }
    if (itemComponent instanceof JComboBox)
      ((JComboBox)itemComponent).addActionListener(this);
    else if (itemComponent instanceof JList)
      ((JList)itemComponent).addListSelectionListener(this);
    else if (itemComponent instanceof JTable)
      ((JTable)itemComponent).getSelectionModel().
	addListSelectionListener(this);
  }
  /**
   * Gets the current data of a JList component.
   * @return an array of objects that are the current data model of the list
   */
  private Object[] getListData() {
    ListModel  model = ((JList)this.itemComponent).getModel();
    return getModelData(model);
  }
  /**
   * Gets the current data of a JComboBox component.
   * @return an array of objects that are the current data model of the list
   */
  private Object[] getComboData() {
    ListModel  model = ((JComboBox)this.itemComponent).getModel();
    return getModelData(model);
  }
  /**
   * Gets the current data of a JTable component.
   * @return an array of objects that are the current data model of the table
   */
  private Object[] getTableData() {
    TableModel  model = ((JTable)this.itemComponent).getModel();
    Object[] data = {};
    if (model instanceof AbstractIXTableModel) {
      List tabData = ((AbstractIXTableModel)model).getData();
      if (tabData != null) return tabData.toArray();
      else return data;
    }

    Debug.noteln("LEP: Getting data from a table is not yet implemented");
    return data;
  }
  /**
   * Gets the current data of a JTree component.
   * @return an array of objects that are the current data model of the tree
   */
  private Object[] getTreeData() {
    TreeModel  model = ((JTree)this.itemComponent).getModel();
    Object[] data = {};
    if (model instanceof EditableTreeModel) {
      List objects = ((EditableTreeModel)model).findUserObjects();
      if (objects != null) data = objects.toArray();
    }
    else {
      Debug.noteln("LEP: Getting data from a tree is not yet implemented");
    }
    return data;
  }


  private Object[] getModelData(ListModel model) {
    int s = model.getSize();
    Object[] data = new Object[s]; 
    for (int i=0; i<s; i++) data[i] = model.getElementAt(i);
    return data;
  }

  public void setListData(Object[] data) {
    boolean enabled = itemComponent.isEnabled();
    itemComponent.setEnabled(false);
    if (JList.class.isInstance(itemComponent)) {
      ((JList)itemComponent).removeAll();      
      ((JList)itemComponent).setListData(data);
    }
    else if (JComboBox.class.isInstance(itemComponent)) {
      JComboBox box = (JComboBox)this.itemComponent;
      box.removeAllItems();
      for (int i=0; i<data.length; i++) box.addItem(data[i]);
    }
    else if (JTable.class.isInstance(itemComponent)) {
      TableModel model = ((JTable)itemComponent).getModel();
      if (AbstractIXTableModel.class.isInstance(model)) {
	((AbstractIXTableModel)model).setData(data);
	((AbstractIXTableModel)model).fireTableDataChanged();
      }
    }
    else if (itemComponent instanceof JTree) {
      TreeModel model = ((JTree)itemComponent).getModel();
      if (model instanceof EditableTreeModel) {
	try {	
	  EditableTreeModel newModel = 
	    (EditableTreeModel)model.getClass().newInstance();
	  IXTrees.setupTreeModel(newModel, Arrays.asList(data));
	  ((JTree)itemComponent).setModel(newModel);
	}
	catch (Exception e) {
	  Debug.noteException(e);
	}
      }
    }
    itemComponent.setEnabled(enabled);    
    dataSet(data); 
  }

  /**
   * Sets the data of a collection component.
   * @param data an array of objects that is to be used as the data 
   */
  public void setData(Object data) {
    //Debug.noteln("LEP: setting data to",data);
    //removeData();
    if (data == null) removeData();
    else if (data instanceof Collection)
      setListData(((Collection)data).toArray());
    else {
	//Debug.noteln(" data of class", data.getClass());
	try {
	    setListData((Object[])data);
	}
	catch (Exception e) {
	    Debug.noteException(e);
	}
    }
    //dataSet(data); //done in setListData
  }

  public void removeData() {
    String[] data = {};
    setListData(data);
  }

    public void setSelectedIndex(int index) {
      try {
	if (itemComponent instanceof JList) 
	    ((JList)itemComponent).setSelectedIndex(index);
	else if (itemComponent instanceof JComboBox) 
	    ((JComboBox)itemComponent).setSelectedIndex(index);
	else if (itemComponent instanceof JTable) 
	    ((JTable)itemComponent).setRowSelectionInterval(index, index);
	else if (itemComponent instanceof JTree)
	    ((JTree)itemComponent).setSelectionRow(index);
      }
      catch (IllegalArgumentException iae) {
	  //this happens at times if the editor instance sets this. No matter.
	  //Debug.noteln("LEdP: got illegal argument on setSelectedIndex");
      }
    }
    public int getSelectedIndex() {
	if (itemComponent instanceof JList) 
	    return ((JList)itemComponent).getSelectedIndex();
	else if (itemComponent instanceof JComboBox) 
	    return ((JComboBox)itemComponent).getSelectedIndex();
	else if (itemComponent instanceof JTable) 
	    return ((JTable)itemComponent).getSelectedRow();
	else if (itemComponent instanceof JTree) {
	    int[] is = ((JTree)itemComponent).getSelectionRows();
	    if (is.length == 0) return -1;
	    else return is[0];
	}
	return -1;
    }

    public void setSelectedObject(Object object) {
	if (itemComponent instanceof JList) 
	    ((JList)itemComponent).setSelectedValue(object, true);
	else if (itemComponent instanceof JComboBox) 
	    ((JComboBox)itemComponent).setSelectedItem(object);
	else if (itemComponent instanceof IXTable) 
	    ((IXTable)itemComponent).setSelectedObject(object);
	else if (itemComponent instanceof JTable) {
	    TableModel model = ((JTable)itemComponent).getModel();
	    if (model instanceof AbstractIXTableModel) {
		int index = ((AbstractIXTableModel)model).getObjectRow(object);
		((JTable)itemComponent).setRowSelectionInterval(index, index);
	    }
	    else 
		Debug.noteln("WARNING: cannot set table object selection for",
			     model.getClass());
	}
	else if (itemComponent instanceof JTree) 
	    Debug.noteln("WARNING: cannot set object selection for trees");
    }

    public Object getSelectedObject() {
	if (itemComponent instanceof JList) 
	    return ((JList)itemComponent).getSelectedValue();
	else if (itemComponent instanceof JComboBox) 
	    return ((JComboBox)itemComponent).getSelectedItem();
	else if (itemComponent instanceof IXTable) 
	   return ((IXTable)itemComponent).getSelectedObject();
	else if (itemComponent instanceof JTable) {
	   int index = ((JTable)itemComponent).getSelectedRow();
	   if (index < 0) return null; //no selection
	   else {
	       TableModel model = ((JTable)itemComponent).getModel();
	       if (model instanceof AbstractIXTableModel) 
		   return ((AbstractIXTableModel)model).getRowObject(index);
	       else return ((JTable)itemComponent).getValueAt(index, 0);
	   }
	}
	else if (itemComponent instanceof JTree) {
	    TreePath path = ((JTree)itemComponent).getSelectionPath();
	    if (path == null) return null;
	    else {
		TreeNode node = (TreeNode)path.getLastPathComponent();
		if (node == null) return null;
		else if (node instanceof DefaultMutableTreeNode)
		    return ((DefaultMutableTreeNode)node).getUserObject();
		else if (node instanceof IXTreeNode)
		    return ((IXTreeNode)node).getUserObject();
		else return node;
	    }
	}
	else {
	    Debug.noteln("WARNING: cannot get selection from a", 
			 itemComponent.getClass());
	    return null;
	}
    }

  public boolean setRenderer(ListCellRenderer r) {
    //Debug.noteln("ListEditorPanel: Set renderer in ",
    //       itemComponent.getClass());
    if (itemComponent instanceof JList) {
      ((JList)itemComponent).setCellRenderer(r);
      return true;
    }
    else if (itemComponent instanceof JComboBox) {
      ((JComboBox)itemComponent).setRenderer(r);
      return true;
    }
    /*
    else if (itemComponent instanceof JTable) {
      ((JTable)itemComponent).setCellRenderer(r);
      return true;
    }
    */
    else return false;
  }

    public void addListSelectionListener(ListSelectionListener lsl) {
	if (lsl != null) listeners.add(lsl);
    }
    public void removeListSelectionListener(ListSelectionListener lsl) {
	if (lsl != null) listeners.remove(lsl);
    }

    //JList, JTable, JTree selection, 
    public void valueChanged(ListSelectionEvent lse) {
      //Debug.noteln("LEdP: calue changed", lse);
      if ((listeners != null) && !listeners.isEmpty()) 
	for (Iterator i = listeners.iterator(); i.hasNext(); ) {
	  ListSelectionListener l = (ListSelectionListener)i.next();
	  l.valueChanged(lse);
	}
    }
    //JComboBox selection
    public void actionPerformed(ActionEvent ae) {
      //Debug.noteln("LEdP: got action", ae);
      if ((ae.getSource() != null) 
	  && (ae.getSource().equals(itemComponent))
	  && (itemComponent instanceof JComboBox)) {
	//(new Throwable()).printStackTrace();
	int index = getSelectedIndex();
	valueChanged(new ListSelectionEvent(ae.getSource(), index, index, 
					    false));
      }
    }


}
