/****************************************************************************
 * A Preferences dialog that is to be adapted for the application.
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Thu Aug 17 12:05:28 2006
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.iface.ui.util.*;

import java.io.*;
import java.util.*;
import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Component;
import java.awt.event.*;
import javax.swing.*;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

/**
 * An abstarct dialog for editing preferences. Provides lots of useful things.
 *
 * Remember to mark properties that cannot be edited in the property
 * manager, otherwise such preperties will be emptied.
 */
public abstract class JPreferencesDialog extends JDialog 
  implements MouseListener, JPropertiesEditor
{
  public JFrame frame;
  public JPanel prefsPanel;
  public HashSet booleanKeys = new HashSet();
  public HashMap keyComponents = new HashMap();
  protected JPropertiesManager manager;


  public JPreferencesDialog(JFrame parentFrame, 
			    JPropertiesManager propertiesManager) {
    super(parentFrame);
    this.frame = parentFrame;
    this.manager = propertiesManager;

    this.setTitle(this.frame.getTitle() + " - Preferences");
    setModal(false);
    getContentPane().setLayout(new BorderLayout());
    setSize(248,94);
    setVisible(false);
    
    prefsPanel = new JPanel();
    String[] buttons = {"Close", "Apply", "Save", "Default", "Reload"};
    IXButPanel butPanel = new IXButPanel(this, BoxLayout.X_AXIS, buttons);
    getContentPane().add(prefsPanel);
    getContentPane().add(butPanel, BorderLayout.SOUTH);

    makeComponents();
    
    pack();
    setSize(getWidth(), getHeight() + 1);
    validate();

  }

  public void setVisible(boolean v) {
    if (v) setComponentsFromProperties(manager.getProperties());
    super.setVisible(v);
  }


  public void getPropertiesFromComponents(boolean fire) {
    getPropertiesFromComponents(manager.getProperties(), fire);
  }


  public void setComponentsFromProperties(Properties props) {
    Enumeration keys = props.propertyNames();
    while (keys.hasMoreElements() ) {
      String key = (String)keys.nextElement();
      String value = props.getProperty(key);
      List components = (List)keyComponents.get(key);
      if ((components != null) && (components.size() != 0) 
	  && (manager.isBooleanProperty(key, value))) {
	Component c = (Component)components.iterator().next();
	setBooleanComponent(key, c, value);
      }
      else setComponentFromProperty(key, components, value);
    }
  }
  /** fires change only when flag set (only for apply, not save etc) */
  public void getPropertiesFromComponents(Properties props, boolean fire) {
    Enumeration keys = props.propertyNames();
    while (keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      if (manager.isEditable(key)) {
	String oldValue = props.getProperty(key);
	List components = (List)keyComponents.get(key);
	if ((components != null) && (components.size() != 0)) {
	  //Debug.noteln("JPD: components for "+key+":", components.size());
	  String value;
	  if (manager.isBooleanProperty(key, oldValue)) {
	    Component c = (Component)components.iterator().next();
	    value = getPropertyFromBooleanComponent(key, c);
	  }
	  else value = getPropertyFromComponent(key, components);
	  if (!oldValue.equals(value)) {
	    props.setProperty(key, value);
	    if (fire) firePropertyChanged(key, oldValue, value);
	  }
	}
	else {
	  Debug.noteln("JPD: no properties to get for", key);
	}
      }
    }
  }


  //------------------------- User buttons ---------------------------------

  /**
   * Applies the preferences to the application.
   */
  private void applyPreferences() {
    getPropertiesFromComponents(manager.getProperties(), true);
  }
  private void savePreferences() {
    getPropertiesFromComponents(false);
    manager.saveProperties();
  }
  void closePreferences(){
    try {this.setVisible(false);//  Hide the JPreferencesDialog
    } catch (Exception e) {}				
  }

  //------------------------- Useful things ---------------------------------

  public void setEnabledComponents(String key, boolean able) {
    setEnabledComponents(key, (List)keyComponents.get(key), able);
  }
  protected void setEnabledComponents(String key, List components, 
				      boolean able) {
    if (components == null) {
      Debug.noteln("no components for", key);
      return;
    }
    for (Iterator i = components.iterator(); i.hasNext(); ) {
      try {	
	setEnabledComponent((Component)i.next(), able);
      }
      catch (Exception e) { Debug.noteException(e); }
    }
  }
  /** I-DE overwrites this to focus on toggle-buttons only */
  protected void setEnabledComponent(Component c, boolean able) {
    c.setEnabled(able);
  }

  //------------------------- Making Components -----------------------------



  protected Component makeBooleanComponent(String key, String value) {
    boolean b = new Boolean(value).booleanValue();
    return new JCheckBox(key, b);
  }

  protected void setBooleanComponent(String key, Component c, String value) {
    boolean b = new Boolean(value).booleanValue();
    JCheckBox box = (JCheckBox)c;
    box.setSelected(b);
  }
  protected String getPropertyFromBooleanComponent(String key, Component c) {
    JCheckBox box = (JCheckBox)c;
    if (box != null) return new Boolean(box.isSelected()).toString();
    else return "";
  }
  
  public Component makeInputComponent(Container box, String key, String label,
				      String value, String ttt) {
    return makeInputComponent(box, key, label, value, ttt, JTextField.LEFT);
  }
  public Component makeInputComponent(Container box, String key, String label,
				      String value, String ttt, int align) {
    JTextField item = new JTextField(value);
    item.setActionCommand(key); 
    item.setToolTipText(ttt);
    item.setHorizontalAlignment(align);
    IXEditorPanel bit = new ThingEditorPanel(this, label, item);
    box.add((Component)bit);
    return (Component)item;
  }

  protected void makeRadioComponent(String key, String label, List options, 
				    String ttEnd) {
    String current = manager.getProperty(key);
    Box column = new Box(BoxLayout.Y_AXIS);
    List components = new ArrayList();
    column.add(new JLabel(label));
    Box box = new Box(BoxLayout.Y_AXIS);
    ButtonGroup bg = new ButtonGroup();
    for (Iterator i = options.iterator(); i.hasNext();) {
      String value = i.next().toString();
      JToggleButton item = new JRadioButton(value);
      item.setActionCommand(value); 
      item.setToolTipText(value + " " + ttEnd);
      bg.add(item);
      item.setSelected(value.equals(current));
      box.add(item);
      components.add(item);
    }
    column.add(box);
    prefsPanel.add(Box.createHorizontalGlue());
    prefsPanel.add(column);
    keyComponents.put(key, components);
  }


  protected void makeCheckBoxComponent(String key, String label, List options, 
				       String ttEnd) {
    String current = manager.getProperty(key);
    Box column = new Box(BoxLayout.Y_AXIS);
    List components = new ArrayList();
    column.add(new JLabel(label));
    Box box = new Box(BoxLayout.Y_AXIS);
    for (Iterator i = options.iterator(); i.hasNext();) {
      String value = i.next().toString();
      JToggleButton item = new JCheckBox(value);
      item.setActionCommand(value); 
      item.setToolTipText(value + " " + ttEnd);
      box.add(item);
      components.add(item);
    }
    column.add(box);
    prefsPanel.add(Box.createHorizontalGlue());
    prefsPanel.add(column);
    keyComponents.put(key, components);
    setComponentFromProperty(key, components, current);
 }


  //------------------------- Abstract things -------------------------------
  
  public abstract void makeComponents();
    /** Sets the given dialog components for the given key with the value.
     *  @return true if there are components to be set, false otherwise. */
  public abstract boolean
      setComponentFromProperty(String key, List components, String val);
  public abstract String getPropertyFromComponent(String key, List components);
  /** for buttons etc within the panel */
  public abstract void localCommandClicked(String command);


  public void mouseClicked(MouseEvent event) {
      Component component = event.getComponent();
      if (!AbstractButton.class.isInstance(component)) return;
      
      AbstractButton button = (AbstractButton) component;
      String command = button.getActionCommand();
      if (command.equals("Close")) closePreferences();
      else if (command.equals("Default")) {
	Properties props = new Properties();
	manager.setDefaultProperties(props);
	setComponentsFromProperties(props);
      }
      else if (command.equals("Apply")) applyPreferences();
      else if (command.equals("Save")) savePreferences();
      else if (command.equals("Reload")) {
	Properties props = new Properties();
	manager.loadProperties(props);
	setComponentsFromProperties(props);
      }
      else if (command.equals("Cancel")) {
	setComponentsFromProperties(manager.getProperties());
	closePreferences();
      }
      else localCommandClicked(command);
  }
  public void mouseEntered(MouseEvent e) {}
  public void mouseExited(MouseEvent e) {}
  public void mousePressed(MouseEvent e) {}
  public void mouseReleased(MouseEvent e) {}

  //--------------------------Notifying listeners-----------------------------

  private HashSet listeners = new HashSet();

  public void addPropertyChangeListener(PropertyChangeListener listener) {
    listeners.add(listener);
  }
  public void setPropertyChangeListeners(HashSet listeners) {
    this.listeners = listeners;
  }
  public void removePropertyChangeListener(PropertyChangeListener listener) {
    listeners.remove(listener);
  }
  public void firePropertyChanged(String key, String oldVal, String newVal) {
    PropertyChangeEvent e = new PropertyChangeEvent(this, key, oldVal, newVal);
    for (Iterator i = listeners.iterator(); i.hasNext();) {
      PropertyChangeListener listener = (PropertyChangeListener)i.next();
      listener.propertyChange(e);
    }
  }
 
  
}


/** Changes
 *
 * 16/02/04: documentation
 *
 */
