/****************************************************************************
 * A frame with a menu bar, a tool bar, about and exit dialogues, a set of
 * panels.
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Tue May  2 13:41:09 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.*;
import ix.util.Debug;
import ix.util.lisp.Symbol;
import ix.iface.ui.util.*;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import java.util.*;
//import java.net.*;

/**
 * A frame with a set of panels.
 *
 * Can make a default Windows and Styles menu (add to View menu?)
 * Provides useful functions for filling a windows menu, adding panels.
 * Keeps track of all known panels and styles and the current panel and style
 * 
 * provides default functions for setPanelStyle, setCurrentPanel
 */
public class JFrameMTP extends JFrameMT implements ActionListener, 
  ChangeListener, ItemListener, FrameSymbols
{
  /** set when the panel is created so that change events can be ignored */
  public boolean isAdjusting = true;
  private ArrayList panelStyles;
  private String panelStyle;

  // main panel for application stuff
  private ArrayList panels = new ArrayList();
  protected JTabbedPane tabbedPane;
  protected CardPanel cardsPanel;
  private JPanel currentPanel;

  // menu bar-------------
  public JMenu windowsMenu;
  public JMenu styleMenu;

  public JFrameMTP(){
    super();
    isAdjusting = true;
    tabbedPane = new JTabbedPane();
    tabbedPane.addChangeListener(this);
    cardsPanel = new CardPanel(this);

    String[] s = {"single", "tabbed", "card"};
    panelStyles =  new ArrayList(Arrays.asList(s));
    panelStyle = "single";
    makeMenuToolBars();
    isAdjusting = false;
    fillMainPanel();    
  }
  /**
   *
   */
  public JFrameMTP(String style){
    this();
    setPanelStyle(style);
  }


  public String getPanelStyle() {
    return panelStyle;
  }
  public boolean setPanelStyle(String style) {
    //Debug.noteln("Setting style to", style);
    if (panelStyle.equals(style)) {
      setStyleInMenu(style); //make sure menus are in synch
      return true;
    }
    if (panelStyles.contains(style)) {
      Dimension size = getSize();
      
      panelStyle = style;
      changePanelStyle();
      //Debug.noteln(" Done. Changing menu if necessary");
      // ensure menus are set ok
      setStyleInMenu(style);

      setSize(size);
      validate();
      return true;
    }
    else {
      String message = "Style " + style + " is not recognised." + "\nl" +
	"Please use one of " + panelStyles.toString() + ".";
      Debug.warn(message);
      return false;
    }
  }

  protected void setStyleInMenu(String style) {
    JMenuItem menuThing;
    if (styleMenu == null) return;
    Component[] menuThings = styleMenu.getMenuComponents();
    String menuStyle = "style " + style;
    //Debug.noteln("JFMTP: looking for ", menuStyle);
    //Debug.noteln(" in", menuThings);
    for (int i=0; i<menuThings.length; i++) {
      menuThing = (JMenuItem)menuThings[i];
      if (((Object)menuThing.getActionCommand()).equals(menuStyle))
	menuThing.setSelected(true);
    }
  }

  public ArrayList getPanelStyles() {
    return panelStyles;
  }
  public void setPanelStyles(ArrayList styles) {
    //Debug.noteln("Setting styles to", styles);
    panelStyles = styles;
  }

  public ArrayList getPanels() {
    return panels;
  }

  public JPanel getACurrentPanel() {
    //Debug.noteln("Getting current panel",currentPanel);
    if (currentPanel == null) {
      if (panels.size() > 0) setCurrentPanel((JPanel)panels.get(0));
    }
    return currentPanel;
  }
  public JPanel getCurrentPanel() {
    return currentPanel;
  }
  public void setCurrentPanel(JPanel panel) {
    if (this.updateCurrentPanel(panel))
      fillMainPanel();
    else Debug.noteln("JFMTP: not updated current panel");
  }

  /** 
   * Carry out updates when the panel has been changed in the window via
   * tab or card selection.
   * The panel is already displayed in the window, update currentPanel and
   * menus.
   * @return true if there were changes to be made, false if not
   */
  protected boolean updateCurrentPanel(JPanel panel) {
    if (panel == null) return false;
    String panelName = panel.getName();
    //Debug.note("JFP: Setting current panel");
    //if (currentPanel != null) Debug.note(" from " + currentPanel.getName());
    //Debug.noteln(" to", panelName);
    //do nothing but make sure the panel is changed (important for cancel)
    if (panel.equals(currentPanel)) return true;

    //do someting
    if (currentPanelChanging(currentPanel, panel)) {
      //Debug.noteln("JFP: done currentPanelChanging");
      currentPanel = panel;
      // ensure current panel selections in menus etc.
      //Debug.noteln("JFP: updating menu selection");
      this.setWindowsMenuSelection(panelName);
      //Debug.noteln("JFP: updatinged menu selection");
      this.adaptMenusToPanel(panel);
      //Debug.noteln("JFP: adapted menus");
      return true;
    }
    else {
      Debug.noteln("JFMTP: CANCEL current panel changing");
      Debug.noteln(" Current panel now is", currentPanel.getName());
      setCurrentPanel(currentPanel);
      return true;  //make sure old panel comes back
    }
  }
  protected boolean updateCurrentPanel(String panelName) {
    JPanel panel = getNamedPanel(panelName);
    return updateCurrentPanel(panel);
  }

  /**
   * Changing current panel from old to new.
   * @return true if the panels should really change, false if the old panel
   * should remain.
   */
  protected boolean currentPanelChanging(JPanel oldPanel, JPanel newPanel) {
    return true;
  }
  protected void adaptMenusToPanel(JPanel panel) {
  }

  protected JPanel getNamedPanel(String panelName) {
    JPanel panel;
    for (int i=0; i<panels.size(); i++) {
      panel = (JPanel)panels.get(i);
      if (panel.getName() == panelName) return panel;
    }
    return null;
  }

  protected void setWindowsMenuSelection(String panelName) {
    if (windowsMenu == null) return;
    JMenuItem menuThing;
    Component[] menuThings = windowsMenu.getMenuComponents();
    if (menuThings == null) Debug.noteln("null menu things");
    for (int i=0; i<menuThings.length; i++) {
      menuThing = (JMenuItem)menuThings[i];
      if (menuThing.getActionCommand() == panelName)
	if (!menuThing.isSelected()) menuThing.setSelected(true);
    }
    //Debug.noteln("Done setCurrentPanel in JFrameMTP");
  }
  
  
  /**
   * Adds a panel to the set of panels that can be displayed.
   * Sets the given parameters in the panel and adds the panel to the panel
   * list.<p>
   *
   * NOTE: ensure that the panel names are unique! <p>
   *
   * @param panel the panel to be added
   * @param name a string to be used as the panel's identifyier for the user
   * @param tttext the tool tip text to use for the panel
   */
  public void addPanel(JPanel panel, String name, String tttext) {
    //Debug.noteln("Adding panel", name);
    panel.setName(name);
    panel.setToolTipText(tttext);
    panels.add(panel);
    if (panelStyle.equals("tabbed")){
      //tabbedPane.removeChangeListener(this);
      isAdjusting = true;
      if (tabbedPane != null) tabbedPane.addTab(name, null, panel, tttext);
      isAdjusting = false;
      //tabbedPane.addChangeListener(this);
    }
    else if (panelStyle.equals("card")){
      if (cardsPanel != null) cardsPanel.addPanel(panel, name);
    }
    if (currentPanel == null) setCurrentPanel(panel);
  }

  /** Removes all panels from the frame. */
  public void removeAllPanels() {
    panels.clear();
    if (panelStyle.equals("tabbed")) {
      isAdjusting = true;
      if (tabbedPane != null) tabbedPane.removeAll();
      isAdjusting = false;
    }
    else if (panelStyle.equals("card")){
      if (cardsPanel != null) cardsPanel.removeAllPanels();
    }
    setCurrentPanel(null);
  }

  private void fillMainPanel() {
    //Debug.noteln("fillMainPanel with panel style", panelStyle);
    //tabbedPane.removeChangeListener(this);
    GridBagConstraints gbc = 
      new GridBagConstraints(0,0,1,1,1.0,1.0,
			     GridBagConstraints.NORTHWEST,
			     GridBagConstraints.BOTH,
			     new Insets(0,0,0,0),0,0);
    JComponent pane;
    if (panelStyle.equals("tabbed")) pane = fillTabbedPane(tabbedPane);
    else if (panelStyle.equals("card")) pane = fillCardPanel(cardsPanel);
    else if (panelStyle.equals("single")) pane = fillSimplePanel();
    else {
      UIUtil.notImplemented(this,"Panel style " + panelStyle);
      return;
    }
    mainJPanel.removeAll();
    if (pane != null) {
      mainJPanel.add(pane,gbc);
      pane.invalidate();
    }
    //pack();
    validate();
  } 
  
  private JTabbedPane fillTabbedPane(JTabbedPane pane) {
    pane.setEnabled(false);
    isAdjusting = true;
    //Debug.noteln("JFMTP: fillTabbedPane. No of tabs:", pane.getTabCount());
    pane.removeAll();
    JPanel panel;
    for (int i=0; i<panels.size(); i++){
      panel = (JPanel)panels.get(i);
      tabbedPane.addTab(panel.getName(), null, panel, panel.getToolTipText());
     }
    panel = getACurrentPanel();
    if (panel != null) {
      //Debug.noteln("JFMTP: setting selected component to", panel.getName());
      pane.setSelectedComponent(panel);
      panel.invalidate();
      panel.setVisible(true);
      panel.validate();
    }
    pane.setEnabled(true);
    isAdjusting = false;
    //tabbedPane.addChangeListener(this);
    return pane;
  }
  private JPanel fillCardPanel(CardPanel parent) {
    parent.setEnabled(false);
    isAdjusting = true;
    parent.isAdjusting = true;
    //Debug.noteln("Filling card pane.");
    parent.removeAllPanels();
    JPanel panel;
    for (int i=0; i<panels.size(); i++){
      panel = (JPanel)panels.get(i);
      parent.addPanel(panel, panel.getName());
     }
    //Debug.noteln("Filling card pane");
    panel = getACurrentPanel();
    if (panel != null) {
      parent.setPanel(currentPanel.getName());
      panel.invalidate();
    }
    parent.isAdjusting = false;
    isAdjusting = false;
    return parent;
  }
  private JPanel fillSimplePanel() {
    JPanel panel = getACurrentPanel();
    TitledBorder b;
    if (panel != null) {
      isAdjusting = true;
      panel.setVisible(false);
      mainJPanel.removeAll();
      mainJPanel.add(panel);
      panel.invalidate();
      panel.setVisible(true);
      panel.validate();
      isAdjusting = false;
    }
    return panel;
    /*
    if (panel != null) {
      b = (TitledBorder)panel.getBorder();
      if (b != null) b.setTitle(panel.getName());
      else {
	b = new TitledBorder(panel.getName());
	panel.setBorder(b);
      }
      return panel;
    }
    else return null;
    */
  }
  private JPanel getSimplePane() {
    return getACurrentPanel();
  }

  /**
   * Makes a default menubar and toolbar.
   * Menus: File, Edit, View (Style, Windows), Help
   */
  public void makeDefaultMenuToolbar(){
    super.makeDefaultMenuToolbar(); //makes File,Edit,View,Help
    makeStyleMenu(optionsMenu);
    //if (viewMenu.getItemCount() != 0) viewMenu.addSeparator();
    makeWindowsMenu(viewMenu);
    adaptDefaultMenuToolbar();
  }

  public void makeViewMenu() {
    super.makeViewMenu(); //makes nothing but may change
    if (viewMenu.getItemCount() != 0) viewMenu.addSeparator();
    makeWindowsMenu(viewMenu);
  }

  public void makeOptionsMenu() {
    super.makeOptionsMenu();
    makeStyleMenu(optionsMenu);
  }

  /**
   * Makes a default windows menu and no toolbar buttons.
   * Makes a single-selection radio menu ("Windows") with the names of
   * all panels.
   * <ul>
   * <li> Menu items: "Windows" menu, <panel-name>*
   * <li> Toolbar buttons: none
   * </ul>
   */
  public JMenu makeWindowsMenu(JComponent parentMenu) {
    return makeWindowsMenu(parentMenu,"Windows");
  }
  /**
   * As above, but with the given string as the menu title.
   */
  public JMenu makeWindowsMenu(JComponent parentMenu, String title) {
    windowsMenu = makeMenu(parentMenu,title);
    fillWindowsMenu();
    return windowsMenu;
  }
  /**
   * Makes a default style menu (all known styles) and no toolbar buttons.
   * Currently, known styles are "single", "tabbed", "card"
   * <ul>
   * <li> Menu items: simple, tabbed, card, 
   * <li> Toolbar buttons: none
   * </ul>
   */
  public JMenu makeStyleMenu(JComponent parentMenu) {
    //Debug.noteln("JFMTP: making style menu");
    styleMenu = makeMenu(parentMenu,"Panel Style");
    ButtonGroup bg = new ButtonGroup();
    IXToolItem item;
    JRadioButtonMenuItem rb;
    for (int i=0; i<panelStyles.size(); i++) {
      String style = (String)panelStyles.get(i);
      item = new IXToolItem(styleMenu,this, "style " + style, style, 
			    "Set style of panel switching to " + style);
      rb = item.makeRadioItem();
      bg.add(rb);
      item.showItem();
      if (style == panelStyle) rb.setSelected(true);
      
      //rb = makeRadioMenuItem(styleMenu,"style " + style, style);
      //bg.add(rb);
      //if (style == panelStyle) rb.setSelected(true);
    }
    //Debug.noteln(" finished making style menu");
    return styleMenu;
  }

  /**
   * Makes a single-selection radio menu ("Windows") with the names of
   * all panels.
   * 
   */
  public void fillWindowsMenu() {
    windowsMenu.setEnabled(false);
    ButtonGroup bg = new ButtonGroup();
    IXToolItem item;
    JRadioButtonMenuItem rb;
    for (int i=0; i<panels.size(); i++) {
      JPanel panel = (JPanel)panels.get(i);
      String name = panel.getName();
      item = new IXToolItem(windowsMenu,this, name, name, 
			    "Switch to " + name + " panel");
      rb = item.makeRadioItem();
      bg.add(rb);
      item.showItem();

      //rb = makeRadioMenuItem(windowsMenu,name,name);
      //bg.add(rb);
    }
    windowsMenu.setEnabled(true);
  }



  protected void changePanelStyle() {
    fillMainPanel();
  }

  /**
   * Listens to changes in which panel is selected in the cardsPanel.
   * Updates the selection in the Windows menu (if there is one).
   * Note: the card panel will have displayed the panel already.
   */
  public void itemStateChanged(ItemEvent event) {
    if (cardsPanel.isAdjusting) return;
    String panelName = (String)event.getItem();
    this.updateCurrentPanel(panelName);  
    
  }  

  /**
   * Listens to changes in which panel is selected in the TabbedPane
   */
  public void stateChanged(ChangeEvent e) {
    if (isAdjusting) return;
    //Debug.noteln("Got state change", e);
    Component source = (Component)e.getSource();
    if (!source.isEnabled()) return;
    //Debug.noteln("Got state change from", source);
    if (source.equals(tabbedPane)) {
      Component component = ((JTabbedPane)source).getSelectedComponent();
      //setCurrentPanel((JPanel)component);
      updateCurrentPanel((JPanel)component);
    }
    //else if (source.equals(cardsPanel)) {
    else Debug.noteln("Cannot process state change from", source);
  }

  /**
   * Implements the ActionListener interface.
   * First passes the event to the current DomainEditorPanel to see if it can
   * process it. If the panel does not recognise the event, this tries to do
   * the processing itself. If this cannot recognise the event either,
   * this passes it to its superclass (JFrameMT).
   */
  public void actionPerformed(java.awt.event.ActionEvent event) {
    if (isAdjusting) return;
    Object object = event.getSource();
    Object command = (Object)((AbstractButton)object).getActionCommand();
    //Debug.noteln("JFrameMTP: action object is",object);
    //first pass it to the panel to see if that can handle it.
    FramePanel fPanel = (FramePanel) getCurrentPanel();
    if ((fPanel != null) && (fPanel.frameActionPerformed(event))) return;

    // no panel or it couldn't handle the event so try handling it here
    Debug.noteln("JFrameMTP: action command is",command);
    if (command.equals("style tabbed")) setPanelStyle("tabbed");
    else if (command.equals("style card")) setPanelStyle("card");
    else if (command.equals("style single")) setPanelStyle("single");
    else {
      for (int i=0; i<panels.size(); i++) {
	JPanel panel = (JPanel)panels.get(i);
	if (command.equals(panel.getName())) {
	  setCurrentPanel(panel);
	  return;
	}
      }
      //event not recognised, so give the superclass a chance to do it
      super.actionPerformed(event);
    }
  }

}
/*
 * Document all public methods
 */
