/****************************************************************************
 * A renderer for list or table cells. Can use text, colour, image, symbol.
 *
 * @author Jussi Stader
 * @version 1.0
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

// needs class TwoField in ix.iface

import javax.swing.*;//jtable, list cell renderer etc
import ix.*;
import ix.util.Debug;
import ix.iface.util.*;

import javax.swing.table.*;//table cell renderer etc.
import java.awt.Component;
import java.awt.Color;
import java.awt.Font;
import javax.swing.border.Border;//label may have a border


/****************************************************************************
 * A renderer for list or table cells. Can use text, colour, image, symbol.
 * 
 * Expects to use a TwoField, ThreeField or FourField object to obtain up
 * to four values, which are represented using text, colour, an image and a 
 * symbol.<p>
 *
 * The IXLabelRenderer is created with a boolean argument which
 * determines whether the cell has a border or not. The border is used
 * to show whether the cell/row is selected or not.<p>
 *
 * The fields unselectedBorder, selectedBorder, and isBordered are
 * used to handle the borders of the cell if it has one.
 *
 * The class sub-classes RendererJLabel rather than JLabel for performance
 * reasons (see Java API documentation of TableCellRenderer or 
 * ListCellRenderer for more detail).
 *
 ****************************************************************************
 */
public abstract class IXLabelRenderer extends RendererLinkLabel 
    implements IXRenderer, TableCellRenderer, ListCellRenderer
{ 
  /** A border to provide the colour of an unselected border */	
  Border unselectedBorder = null;
  /** A border to provide the colour of a selected border */	
  Border selectedBorder = null;
  Border lineSelectedBorder = null;
  Border lineUnselectedBorder = null;
  /** A flag to show whether the cell has a border */	
  boolean isBordered = true;
  /** A flag to show whether the cell has a line */	
  boolean isLined = false;
  int borderWidth = 2;
  int borderHeight = 5;
  
  
    /** 
     * Makes a renderer that produces a border.
     * The border can be used to reflect whether the cell is selected
     *
     * @param isBordered determines whether the cell has a border.
     */
    public IXLabelRenderer(boolean isBordered) {
	super();
	this.isBordered = isBordered;
	this.setOpaque(true); //MUST do this for background to show up.
	setBold(false);
    }


  public void setLined(boolean isLined) {
    this.isLined = isLined;
  }
  public void setBorderSize(int width, int height) {
    borderWidth = width;
    borderHeight = height;
  }

  public void setBold(boolean bold) {
    Font font = getFont();
    Font newFont = font;
    if ((bold && font.isBold()) || (!bold && !font.isBold())) return;
    else if (bold) newFont = font.deriveFont(font.BOLD);
    else newFont = font.deriveFont(font.PLAIN);
    setFont(newFont);
  }


    /**
     * Renders a table cell in colour and text. (Public due to interface).
     */
    public Component getTableCellRendererComponent(JTable table, Object object,
						   boolean isSelected, 
						   boolean hasFocus, 
						   int row, int column) {
	Color selBack = table.getSelectionBackground();
	Color back = table.getBackground();
	return this.getCellRendererComponent(object, isSelected, hasFocus, 
					     back, selBack);
    }

    /**
     * Renders a list cell in colour and text. (Public due to interface).
     */
    public Component getListCellRendererComponent(JList list, Object object,
						  int index,
						  boolean isSelected, 
						  boolean hasFocus) {
	Color selBack = list.getSelectionBackground();
	Color back = list.getBackground();
	//Debug.noteln("Background and selection background are"
	//     + back.toString() + " and " + selBack);
	return this.getCellRendererComponent(object, isSelected, hasFocus, 
					     back, selBack);
    }

    /**
     * Actually does the painting of the cell.
     * Does not care whether it paints a table cell or a list cell.
     * Calls setInfo to provide the information to be displayed.
     */
    private Component getCellRendererComponent(Object object,
					       boolean isSelected, 
					       boolean hasFocus,
					       Color background,
					       Color selectionBackground){
      Border border = null;
      if (isSelected) {
	if (isBordered) {
	  if (selectedBorder == null) {
	    selectedBorder = 
	      BorderFactory.createMatteBorder(borderWidth, borderHeight,
					      borderWidth, borderHeight,
					      selectionBackground);
	  }
	  border = selectedBorder;
	}
	else if (isLined) {
	  if (lineSelectedBorder == null) 
	    lineSelectedBorder = 
	      BorderFactory.createLineBorder(Color.black, 1);
	  border = lineSelectedBorder;
	}
	if (border != null) this.setBorder(border);
      } 
      else {
	if (isBordered) {
	  if (unselectedBorder == null) {
	    unselectedBorder = 
	      BorderFactory.createMatteBorder(borderWidth, borderHeight,
					      borderWidth, borderHeight,
					      background);
	  }
	  border = unselectedBorder;
	}
	else if (isLined) {
	  if (lineUnselectedBorder == null) 
	    lineUnselectedBorder = 
	      BorderFactory.createLineBorder(background, 1);
	  border = lineUnselectedBorder;
	}
	if (border != null) this.setBorder(border);
      }
      if (isSelected) this.setBackground(selectionBackground);
      else this.setBackground(background);
	  
      this.setInfo(object); //overwrites background if desired
      return this;
    }
    
    /**
     * Provides the information to be displayed.
     * <p>Example1:
     * <PRE><code>
     *  protected void setInfo(Object object) {
     *    if (object == null) setText("");
     *    else setText(((UIRefinement)object).getName());
     *  }
     * </code></PRE>
     * Example2 (NDRenderer):
     * <PRE><code>
     *    protected void setInfo(Object object) {
     *        FourField fourField = (FourField) object;
     *        Color colour = (Color) fourField.toColour();
     *        if (colour != null) this.setBackground(colour);
     *        char fieldSymbol = fourField.toSymbol();
     *        String fieldString = fourField.toString();
     *        if (java.lang.Character.isDefined(fieldSymbol)) {
     *          this.setText((String) fieldString + " " + fieldSymbol);}
     *        else this.setText((String) fieldString);
     *        Icon fieldIcon = fourField.toImage();
     *        if (fieldIcon != null) this.setIcon(fieldIcon);
     *    }
     * </code></PRE>
     */
    abstract public void setInfo(Object object);

}

