/****************************************************************************
 * Support for editing sets of objects (e.g. constraints)
 *
 * File: AbstractSetEditor.java
 * @author Jussi Stader
 * @version 4.3
 * Updated: Tue Mar 13 12:35:30 2007
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iface.ui;

import java.util.*;
import java.lang.reflect.Field;
import java.awt.Component;
import java.awt.Cursor;
import javax.swing.JOptionPane;
import javax.swing.JFileChooser;
import java.io.*;

import ix.icore.domain.event.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.util.*;
import ix.util.match.*;
import ix.util.lisp.*;

/**
 * Support for editing sets of objects (e.g. constraints)
 * Looks after adding/removing/editing the objects
 * and keeping the set consistent. Handles overwrites, finding objects
 * in the set, working out whether the set has changed.
 */

public abstract class AbstractSetEditor { 

  public ObjectManager manager;  //the thing interested in the members

  private boolean overwrite = false; //may be set for bulk overwrts when loadng
  public boolean lazyOwn = false; //true if not all own copies are made

  protected String label = "construct"; // set for messages

  /** Constants for handling overwrite */
  protected static final int CANCEL = 0;
  protected static final int ADD = 1;
  protected static final int OVERWRITE = 2;
  protected static final int NEWNAME = 3;

  /**
   * List of objects in the editor. Needed to preserve input order.
   */
  protected StableHashMap members = new StableHashMap();

  /** list of Objects added to the set (unless removed).*/
  protected LinkedList added = new LinkedList();
  /** list of Objects edited in the set (unless added). */
  protected HashSet edited = new HashSet();
  /** list of Objects removed from the set (unless added). */
  protected HashSet removed = new HashSet();
  /** original to copy map; note that new Objects do *not* appear */
  protected StableHashMap originals = new StableHashMap();


  public AbstractSetEditor(ObjectManager manager, LinkedList objects) {
    super();
    setManager(manager, objects); //load objects and set up oritinals etc.
  }

  public void setManager(ObjectManager manager, LinkedList objects) {
    this.manager = manager;
    //setObjects below may need manager to know about us already. HACK
    manager.noteSetEditor(this);
    setObjects(objects); 
  }

  public void setObjects(LinkedList objects) {
    //get all relevant objects in their order and note them in nodes
    //Debug.noteln("SetEd: setting Objects", this);
    clear();
    if (manager != null) {
      members.clear();
      originals.clear();
      if (objects == null) return;
      for (Iterator i = objects.iterator(); i.hasNext(); ) {
	noteOriginal(i.next());
      }
    }
  }

  private void noteOriginal(Object original) {
    if (original != null) { //ignore null objects
      if (!lazyOwn) {
	Object own = makeOwnObject(original);
	members.put(getName(own), own); 
	originals.put(original, own);
      }
      else {
	members.put(getName(original), original);
	originals.put(original, null);
      }
    }
  }

  public Object defaultMakeOwnObject(Object original) {
    Object own = null;
    try { own = cloneConstruct(original);
    } catch (CloneNotSupportedException e) { }
    return own;
  }

  public void clear(){
    resetEdits();
    originals.clear();
  }

  public abstract Named getStringOriginal(String name);

  protected void loadOriginal(Object own) {
    Object original = getStringOriginal(getName(own));
    if (original != null) originals.put(original, own);
  }

  public void publishedEdits() {
    //remove removed from originals
    for (Iterator i = removed.iterator(); i.hasNext(); ) 
      originals.remove(i.next());

    //lookup added in originals, add to originals,
    // put reference to new original into own
    for (Iterator i = added.iterator(); i.hasNext(); ) 
      loadOriginal(i.next());

    //edited may need new baseRef (if name has changed)
    for (Iterator i = edited.iterator(); i.hasNext(); ) 
      loadOriginal(i.next());

    //then clear the lot
    resetEdits();
  }

  public void resetEdits() {
    added.clear();
    edited.clear();
    removed.clear();
  }

  public boolean isLoading() {
    return manager.isLoading();
  }
  public void setOverwrite(boolean onOff) {
    overwrite = onOff;
  }

  //-------------------------abstracts---------------------------------------
  /** Checks whether the given object is one of this editor's own,
   * e.g. UIRefinement */
  public abstract boolean isOwnObject(Object object); // true for own objects
  /** Checks whether the two given objects have the same specifications. */
  public abstract boolean sameConstruct(Object one, Object other);
  /** Makes an own object from the given one */
  public abstract Object makeOwnObject(Object object);
  /** Gets a string representation of the object to identify it uniquely */
  public abstract String getName(Object object);
  /** Sets a string representation of the object that identifies it uniquely */
  public abstract void setName(Object object, String name);
  /** Sets a string representation of the object that identifies it uniquely */
  public abstract void setLegalName(Object object, String name);
  /**
   * Finds a construct whose getName(Object) matches the given string.
   * @return an Object if it exists, otherwise null.
   */
  public abstract Named getNamedObject(String name);
  /** Checks whether the given object is empty */
  public abstract boolean isEmpty(Object object);
  /** Checks whether the given object is undefined */
  public abstract boolean isUndefined(Object object);
  /** Clones the given construct so as not to overwrite the original. */
  public abstract Object cloneConstruct(Object o) 
      throws CloneNotSupportedException;


  public boolean isCurrentOwnObject(Object o) {
    //not current of removed (it or its original)
    if ((removed.contains(o)) ||
	(hasOriginal(o) && removed.contains(getOriginal(o))))
      return false;
    //added or edited or as original, check if it matches
    if ((originals != null) && (originals.entrySet().contains(o))) 
      return true;
    else return (edited.contains(o) || added.contains(o)); 
  }
  public boolean isCurrentNamedObject(Object o) {
    if (o == null) return false;
    if (isOwnObject(o)) return isCurrentOwnObject(o);
    else if (removed.contains(o))//not current of removed (it or its original)
      return false;
    else {
      Object own = originals.get(o);
      if (own == null) return true; //not changed, so still valid
      String oName = getName(o);
      String cName = getName(own);
      //Debug.noteln("ASetEd: old and current names are", oName+", "+cName);
      //Debug.noteln("ASetEd: old object", o);
      //Debug.noteln(" current object", own);
      if (oName == "") return (cName == "");
      else return oName.equals(cName);
    }
  }


  /** Checks whether the given object has an original base object */
  public boolean hasOriginal(Object object) {
    if ((originals != null) 
	&& (originals.values() != null) 
	&& originals.values().contains(object)) return true;
    else return false;
  }
  /** 
   * Gets all original objects from the originals map. If this map is built 
   * in a lazy way (as for Refinements), make sure that this is overwiritten 
   * to look in the original domain.
   */
  public LinkedList getOriginals() {
    if ((originals == null) || (originals.keySet() == null)) return null;
    else return new LinkedList(originals.keySet());
  }
  /** 
   * Finds the original version of the given object in the originals map
   */
  public Object getOriginal(Object object) {
    if ((object != null) && (originals.values() != null)
	&& originals.values().contains(object)) {
      for (Iterator i = originals.keySet().iterator(); i.hasNext(); ) {
	Object original = i.next();
	if (object.equals(originals.get(original))) 
	  return original;
      }
    }
    return null;
  }
  /** 
   * Finds the original version of the given object in the originals map
   */
  public void setOriginal(Object own, Object original) {
    if (original != null) originals.put(original, own);
  }

  /** Checks whether there are objects in the set */
  public boolean isEmpty() {
    List objects = getAllConstructs();
    return ((objects == null) || objects.isEmpty());
  }
  

  //--------------------------Handling changes--------------------------------
  /**
   * Adds a given construct to the set unless it is already there or
   * undefined. Checks originals without removed, added and edited ones.
   */
  public void ensureConstruct(Object object) {
    //if added, edited, or noted as having original, nothing to do.
    //Debug.noteln("AbSE: Ensuring construct", object);
    if (added.contains(object)) return;
    if (edited.contains(object)) return;
    List origs = getOriginals();
    if (origs != null) {
      origs.removeAll(removed);
      //Debug.noteln("ASEd-ensure:", this);
      //Debug.noteln("  removed size ", removed.size());
      //Debug.noteln("  originals without removed size ", origs.size());
    }
    if ((origs != null) && origs.contains(object)) return;
    if ((originals.values() != null) && originals.values().contains(object)) 
      return;
    //add unless empty or undefined
    if (!isEmpty(object) && !isUndefined(object)) {
      addConstruct(object);
    }
  }
  /**
   * Adds a given construct to the set.
   * Checks two things: whether the object is empty (nothing done), and 
   * whether it should overwrite an existing one with the same name.
   * @return false if nothing done, true if added (inc mod)
   */
  public boolean addConstruct(Object object) {
    //Debug.noteln("SE: addConstruct", object.print());
    //Debug.noteln(" Added now: ", UIUtil.listToDisplay(added));
    //Debug.noteln(" Edited now: ", UIUtil.listToDisplay(edited));
    //Debug.noteln(" Removed now: ", UIUtil.listToDisplay(removed));
    int toAdd = handleOverwrite(object);
    if ((toAdd == NEWNAME) || (toAdd == OVERWRITE)) { //start again
      //if overwriting, old one has been removed
      return addConstruct(object);
    }
    else if (toAdd == CANCEL) return false; //stop
    
    //toAdd == ADD, so add normally
    
    members.put(getName(object), object);
    //toAdd == ADD, so add normally
    //Debug.noteln(" Base object is", object.getBaseObject());
    //previously removed, so just forget the remove but remember changes
    if (removed.contains(object)) {
      removed.remove(object);
      fireConstructEdited(object);
    }
    else {
      Object original = getOriginal(object);
      if (removed.contains(original)) {
	removed.remove(original);
	originals.put(original, object);
	fireConstructEdited(object);
      }
      else {
	if (hasOriginal(object)) originals.put(getOriginal(object), object);
	if (!added.contains(object) && added.add(object)) 
	  fireConstructAdded(object);
	else {
	  //Debug.noteln("SetEd: object already added");
	  fireConstructEdited(object); 
	}
      }
    }
    return true;
    //Debug.noteln(" Added now: ", UIUtil.listToDisplay(added));
    //Debug.noteln(" Edited now: ", UIUtil.listToDisplay(edited));
    //Debug.noteln(" Removed now: ", UIUtil.listToDisplay(removed));
  }

  /**
   * Determines whether the given object has a name that is already in
   * the set. Ensures that the old object has not been removed (if it has, 
   * its name can be reused).
   * @return the object with the same name if one can be found, null
   * if not.
   **/
  private Object getNameClashObject(Object object) {
    return getNameClashObject(object, getName(object)); 
  }
  /**
   * Determines whether the given object has a name that is already in
   * the set. Ensures that the old object has not been removed (if it has, 
   * its name can be reused).
   * @return the object with the same name if one can be found, null
   * if not.
   **/
  private Object getNameClashObject(Object object, String name) {
    if (name == "") return null;
    Named oldObject = getNamedObject(name); 
    //Debug.noteln("ASetEd: name clash object is", oldObject);
    if ((oldObject != null) && //there is an old object
	name.equals(oldObject.getName()) && //that has this name
	(!oldObject.equals(object)) && //not the new one
	(!oldObject.equals(getOriginal(object))) && //not new one's original
	(!removed.contains(oldObject))) { //not removed
      //check it is not an old edit (i.e. name still in use)
      //Debug.noteln(" old name", oldObject.getName());
      if (!isCurrentNamedObject(oldObject))
	oldObject = null;
      Debug.noteln("ASetEd: name clash object is", oldObject);
      return oldObject;
    }
    else return null;
  }

  /**
   * Handles overwrites for objects that are not alerady in the domain. Use 
   * the next (handleOverwrite(object, newName)) for ones that are.
   * @returns ADD if the object is to have the new name (no clash);
   * NEWNAME if the user has been asked for a new name (check this one)
   * CANCEL if the user cancelled the whole operation,
   * OVERWRITE if the clashing object has been removed, i.e. try again.
   */
  private int handleOverwrite(Object object) {
    return handleOverwrite(object, getName(object));
  }
  /**
   * Handles overwrites for objects that may alerady be in the domain. 
   * @returns ADD if the object is to have the new name (no clash);
   * NEWNAME if the user has been asked for a new name (check this one)
   * CANCEL if the user cancelled the whole operation,
   * OVERWRITE if the clashing object has been removed, i.e. try again.
   */
  //The first dialog should not be YES_NO_CANCEL but YES_NO_YES-ALL
  private int handleOverwrite(Object object, String newName) {
    String name = newName;
    //Debug.noteln("SetEd: handleOverwrite");
    //(new Throwable()).printStackTrace();
    /*
    if (((object == null) || isEmpty(object))
	&& ((newName == null) || newName.equals("")))
      return CANCEL; //dont put in empty objects
    */
    //no name: get a name and start again (or give up)
    if ((newName == null) || newName.equals("")) {
      //(new Throwable()).printStackTrace();
      //needs a name
      String answer = 
	JOptionPane.showInputDialog("Please enter a name for the " + label);
      if (answer == null) return CANCEL; //cancel ****check what happens!
      else { //got a new name
	try {
	  //object not in domain, so update it (in domain will update later)
	  if (((getName(object) == null) && (newName == null))
	      || getName(object).equals(newName)) 
	    setName(object, answer);
	  //check whether the new name is ok
	  return handleOverwrite(object, answer); 
	}
	catch (Exception e) {
	  Debug.noteException(e);
	  return CANCEL;
	}
      }
    }
    //else (above always returns)
    Object[] optionsA = {"Replace", "Change Name", "Replace All", "Cancel"};
    Object[] optionsB = {"Replace", "Change Name", "Cancel"};
    List options = null;
    Object oldObject = getNameClashObject(object, newName);
    if (oldObject != null) {
      Debug.noteln("ASE: we do have an overwrite situation for", object);
      Debug.noteln(" old object is", oldObject);

      int y = 0; //"Replace" slot; stays if loading and overwrite is set!
      if (isLoading()) {
	//Debug.noteln("ASE: we are loading");
	options = Arrays.asList(optionsA);
	if (!overwrite) {
	  String[] message = 
	    {"There already is a " + label + " called " + 
	     newName + ". Do you want to ",
	     "Replace the existing " + label + "?",
	     "Change the name of the new " + label + "?",
	     "Replace ALL duplicate " + label + "s during this open/insert?",
	     "Cancel the overwrite, i.e. use the existing " 
	     + label + ", forgetting the new one?"};
	  y = JOptionPane.showOptionDialog(null, message, 
					   "WARNING: Name clash",
					   JOptionPane.DEFAULT_OPTION, 
					   JOptionPane.WARNING_MESSAGE,
					   null, optionsA, optionsA[0]);
	}
      }
      else { //not loading, so ask for each time
	//Debug.noteln("ASetEd: handling overwrite");
        //(new Throwable()).printStackTrace();

	String[] message = {"There already is a " + label + " called " + 
			    newName + ". Do you want to ",
			    "Replace the existing " + label + "?",
			    "Change the name of the new " + label + "?",
			    "Cancel the overwrite, i.e. use the existing " 
			    + label + ", forgetting the new one?"};
	y = JOptionPane.showOptionDialog(null, message, 
					 "WARNING: Name clash",
					 JOptionPane.DEFAULT_OPTION, 
					 JOptionPane.WARNING_MESSAGE,
					 null, optionsB, optionsB[0]);
	options = Arrays.asList(optionsB);
      }
      //Debug.noteln("SetEd: y = .", y);
      if ((y == JOptionPane.CLOSED_OPTION) || "Cancel".equals(options.get(y)))
	{
	  //Debug.noteln("SetEd: Overwrite got cancelled");
	  return CANCEL;
	}
      if ("Replace All".equals(options.get(y)) || 
	  "Replace".equals(options.get(y))) {
	if ("Replace All".equals(options.get(y))) {
	  //Debug.noteln("SetEd: Overwrite setting replace all");
	  overwrite = true;
	}
	//remove the old one and do original plan with the new one
	//Debug.noteln("SetEd: Overwrite replacing");
	//removeConstructQuiet(oldObject); //remove then add
	removeConstruct(oldObject); //remove then add
	return OVERWRITE;
      }
      else { //new name
	//change the name then do original plan with the new one
	//Debug.noteln("SetEd: Overwrite changing name");
	String answer = 
	  JOptionPane.showInputDialog("Please enter the new name");
	if (answer == null) return CANCEL; 
	try {
	  setName(object, answer);
	  return NEWNAME;
	}
	catch (Exception e2) {
	  Debug.noteException(e2);
	  return CANCEL;
	}
      }
    }
    
    return ADD;
  }

  /**
   * The two given named objects clash if changeO changes its name to
   * newName. Offer choices: 
   * - change oldO name: legal-set changeO temporary name, 
   *   get oldO name from user and set it on oldO, set newName in changeO
   * - change newName: get name from user and setName that on changeO
   * - or forget the new name and keep changeO's old name.
   * @return true if changes are made, false for last option.
   */
  public boolean handleNameClash(Named changeO, String newName, Object old){
    Debug.noteln("ASetEd: name clash on", newName);
    Debug.noteln(" Between " + changeO.toString() + " and", old);
    //(new Throwable()).printStackTrace();
    Object[] options = {"Change this name", "Change other name", "Cancel"};
    String[] message = 
     {"There already is a " + label + " called " +newName+ ". Do you want to ",
      "Change the name of this " + label + "?",
      "Change the name of the other " + label + "?",
      "Cancel the name change and stay with the old name " 
      + changeO.getName()}; 
    int y = JOptionPane.showOptionDialog(null, message, 
					      "WARNING: Name clash",
					      JOptionPane.DEFAULT_OPTION, 
					      JOptionPane.WARNING_MESSAGE,
					      null, options, options[0]);
    if ((y == JOptionPane.CLOSED_OPTION) || (y == 2)) {
      //Debug.noteln("SetEd: name change got cancelled");
      return false;
    }
    else {//get a new name
      String replaceName = 
	JOptionPane.showInputDialog("Please enter the new name");
      if (replaceName == null) return false; //user changed mind
      //now we have a new name
      if (y == 0) //change new object and try again
	setName(changeO, replaceName);
      else {
	String oldName = changeO.getName();
	setLegalName(changeO, newName+"temporaryNameXX");
	Named oldO;
	if (isOwnObject(old)) oldO = (Named)old;
	else oldO = (Named)getOwnObject(old);
	setName(oldO, replaceName);
	setName(changeO, newName); //in case user used newName again!
	//see whether all worked ok. If not, put things back and forget
	String otherName = oldO.getName();
	if ((otherName == null) && (newName == null)) return true;
	if ((otherName == null) ||
	    !otherName.equals(replaceName)) {
	  setLegalName(changeO, oldName);
	  setLegalName(oldO, newName);   
	  return false;
	}
      }
      return true;
    }
  }

  public boolean hasChangedFromOriginal(Object o) {
    Object original = getOriginal(o);
    return !sameConstruct(o, original);
  }
  
  public void updateConstruct(Object object) {
    if (!isEmpty(object)) {
      //Debug.noteln("SetEd: updateConstruct");
      int toAdd = handleOverwrite(object);
      if ((toAdd == NEWNAME) || (toAdd == OVERWRITE)) { //start again
	updateConstruct(object);
	return;
      }
      else if (toAdd == CANCEL) return;
    }
    //at this point the object is fine as it is (no name clashes) or empty
    //if it has been added, ignore edits, else remember them
    if (hasOriginal(object)) {
      if (!added.contains(object)) edited.add(object);
      //Debug.noteln("SetEd: Edited construct", getName(object));
      //back to same as original, forget changes!
      if (!hasChangedFromOriginal(object)) edited.remove(object);
      originals.put(getOriginal(object), object);
      fireConstructEdited(object);
    }
    else addConstruct(object);
  }

  public void removeConstruct(Object object) {
    removeConstructQuiet(object);
    if (!hasOriginal(object) && isEmpty(object)) return;
    else fireConstructRemoved(object);
  }
  private void removeConstructQuiet(Object object) {
    Object ownObject;
    if (!isOwnObject(object)) { //request to remove base object
      ownObject = findOwnObject(object);
      if (ownObject == null) {//removing original with no own
	removed.add(object);
	return;
      } //else remove own version as below
    }

    //when we get here, either we have looked up the own object or it was a
    //request to remove own object 
    else ownObject = object; 
    

    //work on own object from here
    //if it has been added, forget this and be done (do fire removed)
    if (added.contains(ownObject)) {
      added.remove(ownObject);
      return;
    }

    //if it is empty and has no original - just forget it.(don't fire removed)
    else if (!hasOriginal(ownObject) && isEmpty(ownObject)) return;

    //else forget all edits (if any) and remember that it is removed
    else { //has original or is not empty
      edited.remove(ownObject);
      if (hasOriginal(ownObject)) removed.add(getOriginal(ownObject));
    }
    if (originals.containsValue(ownObject)) {
      originals.put(getOriginal(ownObject), null);
    }
  }


  /** 
   * Finds out whether any constructs in the set have changed.
   * The set has not changed if there are no added/edited/removed notes
   */
  public boolean hasChangedConstructs() {
    if (added.isEmpty() && removed.isEmpty() && edited.isEmpty())
      return false;
    else return true;				   
  }

  /** 
   * Collects descriptions of any changes in constructs in the set.
   */
  public List collectConstructChanges() {
    List changes = new ArrayList();
    if (hasChangedConstructs()) {
      changes.add("Changes in " + label);
      if (!added.isEmpty()) changes.add(" Added: " + UIUtil.show(added));
      if (!removed.isEmpty()) changes.add(" Removed: " + UIUtil.show(removed));
      if (!edited.isEmpty()) changes.add(" Edited: " + UIUtil.show(edited));
    }
    return changes;
  }


  //--------------------------Getting information-----------------------------


  public Object newOwnObject(Object object) {
    if (object == null) return null;
    Object ownObject = null;
    ownObject = makeOwnObject(object);
    if (ownObject != null)
      addedObject(ownObject, object);
    return ownObject;
  }


  public Object getOwnObject(Object object) {
    //find out whether there is an own version for this object
    Object oo = findOwnObject(object);
    if (oo == null) return newOwnObject(object);
    else return oo;
  }

  /**
   * Finds an ownObject from the given Object by looking it up in the
   * "originals" list.
   * @return - the ownObject found or null
   */
  public Object findOwnObject(Object object) {
    if (object == null) return null;
    if (isOwnObject(object)) return object; //already own
    String name = "";
    if (object instanceof String)  //prepare for name/symbol lookup
      name = (String)object;
    else if (object instanceof Symbol)
      name = ((Symbol)object).toString();
    
    if ((name != null) && !name.equals("")) {//look for object by name
      //Debug.noteln("SetEd: Looking for named object",name);
      Object found = findOwnObject(name, added);
      if (found == null) 
	found = findOwnObject(name, edited);
      if (found == null)
	found = findOwnObject(name, originals.values());
      return found;
    }

    else { //object given, not name, not own
      if (removed.contains(object)) return null;

      //Debug.noteln("SetEd: originals length:-------",originals.size());
      //Debug.noteln("SetEd: finding own object for ", getName(object));
      Object o = originals.get(object);
      if (o == null) //not found, search by name
	o = findOwnObject(getName(object));
      if (o != null) {
	//Debug.note("SetEd: found object for " + getName(object));
	//Debug.noteln(" -- object is", o);
      }
      return o;
    }
  }

  private Object findOwnObject(String name, Collection collection) {
    for (Iterator i = collection.iterator(); i.hasNext(); ) {
      try {
	Object o = i.next();
	//Debug.noteln(" checking", getName(o));
	if (getName(o).equals(name)) return o;
      }
      catch (Exception e) {} //dont worry if the object is not named
    }
    return null;
  }

  public void addedObject(Object own, Object original) {
    if (original == null) {
      //Debug.noteln("ASetEd: got null original for", own);
      addConstruct(own);
    }
    else {
      //Debug.note("SetEd: originals object " + getName(original));
      //Debug.noteln(" - with object", getName(own));
      originals.put(original, own); 
      //Debug.noteln("SetEd:             originals length:", originals.size());
      //fireConstructAdded(own);
    }
  }


  public LinkedList getAllConstructs() {
    //start with originals (they came first!) remove removed ones.
    // go through this list and use own ones where available 
    //add added in their own order

    //get originals, remove removed ones
    LinkedList currents = new LinkedList();
    LinkedList sofars = getOriginals();
    sofars.removeAll(removed);

    //use ui ones of originals 
    for (Iterator i = sofars.iterator(); i.hasNext(); ) {
      Object orig = i.next();
      if (!removed.contains(orig)) { //do nothing if it is removed
	Object own = findOwnObject(orig);
	if (own != null) {
	  //removed is not really necessary - should never contain own
	  if (!removed.contains(own) && !currents.contains(own)) 
	    //ui version exists so forget the domain version
	    currents.add(own);
	}
	else // no own
	  if (!currents.contains(orig)) currents.add(orig);
      }
    }
    
    // add added
    for (Iterator i = added.iterator(); i.hasNext();) {
      Object next = i.next();
      if (!removed.contains(next) &&!currents.contains(next)) 
	currents.add(next);
    }

    //Debug.noteln("AllConstructs are:", IVUtil.show(currents));
    return currents;
  }

  //--------------------------Notifying listeners-----------------------------


  public void fireConstructAdded(Object object) {
    //Debug.noteln("SetEd firing construct added", object.print());
    manager.fireConstructAdded(this, object);
  }
  public void fireConstructEdited(Object object) {
    //Debug.noteln("SetEd firing construct edited");
    manager.fireConstructEdited(this, object);
  }
  public void fireConstructRemoved(Object object) {
    //Debug.noteln("SetEd firing construct removed");
    manager.fireConstructRemoved(this, object);
  }



  //-------------------------- things---------------------------------

  public String print() { 	
    List cs = getAllConstructs();
    return "[" + cs.size() + " " + label + "s]";
  }
  public String printDetails() { 	
    List cs = getAllConstructs();
    return "[" + cs.size() + " " + label + "s]" + UIUtil.lineSeparator +
      "removed: "+UIUtil.show(removed)+ UIUtil.lineSeparator +
      "added: "+UIUtil.show(added)+ UIUtil.lineSeparator +
      "edited: "+UIUtil.show(edited);
  }
  public String printSet() { 	
    List cs = getAllConstructs();
    String p = "[";
    for (Iterator i = cs.iterator(); i.hasNext(); ) {
      Object o = i.next();
      p = p + getName(o) + ", ";
    }
    return p;
  }




}


// Issues:
// * tidy up load/insert domain stuff 
// * 
// * 
