/****************************************************************************
 * A simple panel with a label, a component, and optional buttons.
 *
 * @author Jussi Stader
 * @version 2.3+
 * Updated: Mon Mar 19 10:14:33 2007
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import java.awt.LayoutManager;
import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.Insets;
import java.awt.Dimension;
import java.awt.Container;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.text.JTextComponent;
import javax.swing.table.*;
import java.util.EventListener;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import ix.util.Debug;
import ix.util.Util;
import ix.iface.ui.table.*;
import ix.iface.ui.util.UIUtil;
    
/**
 * A simple panel with a label (title), a component, and optional buttons.
 * The given label will be at the top, the given component at the left and
 * buttons with given labels on the right.
 * Adds the given listener to the buttons and the list.
 */
public abstract class AbstractEditorPanel extends JPanel 
  implements IXEditorPanel {  //listens for larger/smaller

  public boolean isBordered = false;
  /** A sub-panel containing buttons */
  public IXButPanel butPanel;
  /** A scroll-panel for the component (if required) */
  protected JScrollPane scrollPan = null;
  /** The panel's label (title) */
  public JLabel nameLabel;
  /** The panel's border (if it has one) */
  public TitledBorder border;
  /** The panel's item that may contain data. There are methods for setting
   * and getting data for JList and JTextComponent items
   */
  public JComponent itemComponent;
        
  /**
   * Creates a panel with a label (top left) or a border (depending on the
   * default)
   * Other components (a JComponent and a button panel) are usually added via
   * one of the other constructors; the label text is set in other
   * constructors.
   */
  public AbstractEditorPanel(boolean minimisable){
    super(new GridBagLayout());
    makePanel(minimisable);
  }
  /**
   * Creates a panel as above but with a given border setting.
   * If there is a border, the title of the panel will appear in the border,
   * otherwise it will appear in a label at the top left of the panel.
   * Other components (a JComponent and a button panel) are usually added via
   * one of the other constructors; the label text is set in other
   * constructors.
   */
  public AbstractEditorPanel(boolean minimisable, boolean makeBorder) {
    super(new GridBagLayout());
    isBordered = makeBorder;
    makePanel(minimisable);
  }
  /**
   * Creates a panel with a title and a component (left).
   * If the border-flag is set, the title will go into the border, otherwise
   * a label will be used (top left).
   * If the component is a JList or a JTextArea, it is made scrollable
   * Also used by AbstractEditorPanel constructors that have buttons.
   *
   * @param ml the listener that is interested in component events
   * @param makeBorder a flag determining whether the title is set in a border
   *   (true) or a label (false).
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   */
  public AbstractEditorPanel(EventListener ml,boolean makeBorder, String label,
			     boolean minimisable, JComponent component){
    this(minimisable, makeBorder);
    makePanelComponent(ml, component);
    setLabel(label);
    this.setName(label);
  }
  public AbstractEditorPanel(EventListener ml, boolean makeBorder,
			     String label, JComponent component){
    this(ml, makeBorder, label, false, component);
  }
  /**
   * Creates a panel with a title and a component (left) as above with default
   * border/label.
   *
   * @param ml the listener that is interested in component events
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   */
  public AbstractEditorPanel(EventListener ml, String label, 
			     boolean minimisable, JComponent component) {
    this(ml, false, label, minimisable, component);
  }
  public AbstractEditorPanel(EventListener ml, String label,
			     JComponent component) {
    this(ml, label, false, component);
  }
  /**
   * Creates a panel with a label (top left) and buttons (right) only.
   *
   * @param ml the listener that is interested in button events
   * @param label a string that is used as the label (title of the panel).
   * @param buttons an array of strings that are button labels
   */
  public AbstractEditorPanel(EventListener ml, String label,
			     boolean minimisable, String[] buttons) {
    this(minimisable);
    setLabel(label);
    makeButtons(ml, buttons);
  }
  public AbstractEditorPanel(EventListener ml, String label,
			     String[] buttons) {
    this(ml, label, false, buttons);
  }     
  /**
   * Creates a panel with a title, component (left) and buttons (right).
   * The title goes either into a label (top left) or into a border, depening
   * on the parameter.
   * Calls AbstractEditorPanel(EventListener, label, JComponent)
   * and makes the buttons itself.
   *
   * @param ml the listener that is interested in component events
   * @param makeBorder a flag determining whether the title is set in a border
   *   (true) or a label (false).
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   * @param buttons an array of strings that are button labels
   */
  public AbstractEditorPanel(EventListener ml, boolean makeBorder,
			     String label, boolean minimisable, 
			     JComponent component, String[] buttons){
    this(ml, makeBorder, label, minimisable, component);
    makeButtons(ml, buttons);
  }
  public AbstractEditorPanel(EventListener ml, boolean makeBorder,
			     String label, JComponent component, 
			     String[] buttons){
    this(ml, makeBorder, label, false, component, buttons);
  }
  /**
   * Creates a panel as above but with the default border/label setting.
   *
   * @param ml the listener that is interested in component events
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   * @param buttons an array of strings that are button labels
   */
  public AbstractEditorPanel(EventListener ml, String label, 
			     boolean minimisable, JComponent component, 
			     String[] buttons){
    this(ml, label, minimisable, component);
    makeButtons(ml, buttons);
  }
  public AbstractEditorPanel(EventListener ml, String label, 
			     JComponent component, String[] buttons){
    this(ml, label, false, component, buttons);
  }

  /**
   * Makes an empty label and adds it to the panel;
   * The label is the title of the border if there is a border,
   * otherwise the label is at the top left of the panel.
   */
  private void makePanel(boolean minimisable) {
    if (isBordered) {
      border = BorderFactory.createTitledBorder("some label");
      this.setBorder(border);
    }
    else {
      GridBagConstraints gbc = 
	new GridBagConstraints(0,0,  GridBagConstraints.REMAINDER,1, //x,y, w,h
			       1.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,4,0,0),0,0); //insets and padding
      nameLabel = new JLabel("some label");
      this.add(nameLabel, gbc);

      if (minimisable) makeSizeIcons();
    }
  }

  protected ImageIcon minIcon;
  protected ImageIcon deminIcon; 
  protected ImageIcon xDeminIcon; 
  protected ImageIcon xMinIcon;
  protected JButton minBut; 
  protected JButton deminBut;

  private void makeSizeIcons() {
    try {
      minIcon = Util.resourceImageIcon("ide-symbol-minimise.gif");
      deminIcon = Util.resourceImageIcon("ide-symbol-deminimise.gif");
      xDeminIcon = Util.resourceImageIcon("ide-symbol-xdeminimise.gif");
      xMinIcon = Util.resourceImageIcon("ide-symbol-xminimise.gif");
      minBut = new JButton(minIcon);
      deminBut = new JButton(deminIcon);

      minBut.setToolTipText("Minimise component");
      deminBut.setToolTipText("De-minimise component");
      minBut.setActionCommand("minimise");
      deminBut.setActionCommand("deminimise");

      final JPopupMenu popupMax = new JPopupMenu();
      final JPopupMenu popup = new JPopupMenu();
      JMenuItem menuItem = new JMenuItem("de-minimise component");
      menuItem.addActionListener(new ActionListener() {
	  public void actionPerformed(ActionEvent ae) { 
	    AbstractEditorPanel.this.deMinimise();
	  }
	});
      popupMax.add(menuItem);

      menuItem = new JMenuItem("minimise component");
      menuItem.addActionListener(new ActionListener() {
	  public void actionPerformed(ActionEvent ae) { 
	    AbstractEditorPanel.this.minimise();
	  }
	});
      popup.add(menuItem);
      popup.addSeparator();
      menuItem = new JMenuItem("smaller");
      menuItem.addActionListener(new ActionListener() {
	  public void actionPerformed(ActionEvent ae) { 
	    AbstractEditorPanel.this.smaller();
	  }
	});
      popup.add(menuItem);      
      menuItem = new JMenuItem("larger");
      menuItem.addActionListener(new ActionListener() {
	  public void actionPerformed(ActionEvent ae) { 
	    AbstractEditorPanel.this.larger();
	  }
	});
      popup.add(menuItem);      


      minBut.addMouseListener(new MouseAdapter() {
	  public void mouseClicked(MouseEvent me) { 
	    //Debug.noteln("AEdPan: mouse clicked");
	    if (me.isPopupTrigger()) 
	      popup.show(me.getComponent(), me.getX(), me.getY());
	    else 
	      AbstractEditorPanel.this.minimise();
	  }
	  public void mousePressed(MouseEvent me) { 
	    //Debug.noteln("AEdPan: mouse pressed");
	    if (me.isPopupTrigger()) 
	      popup.show(me.getComponent(), me.getX(), me.getY());
	  }
	  public void mouseReleased(MouseEvent me) { 
	    //Debug.noteln("AEdPan: mouse pressed");
	    if (me.isPopupTrigger()) 
	      popup.show(me.getComponent(), me.getX(), me.getY());
	  }
	});
      deminBut.addMouseListener(new MouseAdapter() {
	  public void mouseClicked(MouseEvent me) { 
	    //Debug.noteln("AEdPan: mouse clicked");
	    if (me.isPopupTrigger()) 
	      popupMax.show(me.getComponent(), me.getX(), me.getY());
	    else 
	      AbstractEditorPanel.this.deMinimise();
	  }
	  public void mousePressed(MouseEvent me) { 
	    //Debug.noteln("AEdPan: mouse pressed");
	    if (me.isPopupTrigger()) 
	      popupMax.show(me.getComponent(), me.getX(), me.getY());
	  }
	  public void mouseReleased(MouseEvent me) { 
	    //Debug.noteln("AEdPan: mouse pressed");
	    if (me.isPopupTrigger()) 
	      popupMax.show(me.getComponent(), me.getX(), me.getY());
	  }
	});

      Dimension dim = new Dimension(minIcon.getIconWidth()+5, 
				    minIcon.getIconHeight()+3);
      //Debug.noteln("ButDim is", dim);
      minBut.setMinimumSize(dim);
      minBut.setMaximumSize(dim);
      minBut.setPreferredSize(dim);
      deminBut.setMinimumSize(dim);
      deminBut.setMaximumSize(dim);
      deminBut.setPreferredSize(dim);

      GridBagConstraints gbc;

      //Box iconBox = new Box(BoxLayout.X_AXIS);
      //iconBox.add(smallerBut);
      //iconBox.add(largerBut);


      gbc = new GridBagConstraints(5,0, 1,1, //x,y, w,h
				   0.0,0.0,  //weight x,y
				   GridBagConstraints.EAST, //anchor
				   GridBagConstraints.NONE,  //fill
				   new Insets(0,4,0,0),0,0); //insets, padding
      add(minBut, gbc);
      add(deminBut, gbc);
      deminBut.setVisible(false);
    }
    catch (Exception e) { 
      Debug.noteln("AEdPan: cannot make buttons to change size");
    }
  }

  /**
   * Puts the component into the panel (left) and sets the title.
   * If the border-flag is set, the title will go into the border, otherwise
   * a label will be used (top left).
   * If the component is a JList, JTable or a JTextArea, it is made scrollable
   * Also used by AbstractEditorPanel constructors that have buttons.
   *
   * @param ml the listener that is interested in component events
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   */
  protected void makePanelComponent(EventListener ml, JComponent component) {
    boolean horizontal = false;
    int fill = GridBagConstraints.BOTH;

    if (JTextField.class.isInstance(component)) {
      horizontal = true;
      fill = GridBagConstraints.HORIZONTAL;
    }
    else if (JCheckBox.class.isInstance(component)) {
      horizontal = true;
      fill = GridBagConstraints.NONE;
    }
    else if (component instanceof IXButPanel) {
      //Debug.noteln("AEdP: got butPan", component);
      if (((IXButPanel)component).getOrientation() != BoxLayout.Y_AXIS) 
	horizontal = true;
      fill = GridBagConstraints.NONE;
    }
    //variables fill and horizontal are set now

    GridBagConstraints gbc;

    if (isBordered) 
      gbc = new GridBagConstraints(0,0, 2,1,  //x,y, width,height
				   2.0,1.0,  //weight x,y
				   GridBagConstraints.NORTHWEST, //anchor
				   fill,// fill
				   new Insets(0,2,0,2),1,0); //insets, padding
    else if (horizontal) {
      //Debug.noteln("AEdP: horizontal");
      //change the label constraint
      gbc = new GridBagConstraints(0,0,  1,1, //x,y, w,h
				   0.0,0.0,  //weight x,y
				   GridBagConstraints.NORTHWEST, //anchor
				   GridBagConstraints.NONE,  //fill
				   new Insets(0,4,0,0),0,0); //insets, pad
      this.add(nameLabel, gbc);
      
      //make gbc for component next to label      
      gbc = new GridBagConstraints(1,0, 1,1,  //x,y, width,height
				   2.0,1.0,  //weight x,y
				   GridBagConstraints.NORTHWEST, //anchor
				   fill,// fill
				   new Insets(0,2,0,2),1,0); //insets, padding

    }
    else 
      gbc = new GridBagConstraints(0,1, 3,1,  //x,y, width,height
				   2.0,1.0,  //weight x,y
				   GridBagConstraints.WEST, fill, //anchor,fill
				   new Insets(0,2,0,2),1,0); //insets, padding
    if (isScrollItem(component)) {
      // JLists and JTextAreas should be scrollable!
      //Debug.noteln("AEdP: scoll item", component);
      scrollPan = new JScrollPane();
      scrollPan.getViewport().add(component);
      this.add(scrollPan, gbc);
    }
    else this.add(component, gbc);

    //strictly this bit should be obsolete with the above
    JComponent insideComponent = null;
    if (component instanceof JScrollPane) {
      scrollPan = (JScrollPane)component;
      //if the component is a scroll pane, the real component is the one inside
      try { //try to get the component inside
	insideComponent = (JComponent)
	  ((JScrollPane)component).getViewport().getComponent(0);
      }
      catch (ArrayIndexOutOfBoundsException e) { }
    }
    if (insideComponent == null) //no scroll pane or no inside component found
      itemComponent = component;
    else itemComponent = insideComponent;

    addComponentListener(ml);
  }
  private void makeButtons(EventListener ml, String[] buttons){
    if ((buttons != null) && (buttons.length > 0)) {
      butPanel = new IXButPanel(ml, (Object[])buttons);
      GridBagConstraints gbc = 
	new GridBagConstraints(3,1,  //x,y
			       3,1,   //width,height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.EAST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
      this.add(this.butPanel, gbc);
      makeRightClickMenu(butPanel.getButtons());
    }
  }

  JPopupMenu rightClickMenu;

  public JPopupMenu ensureRightClickMenu() {
    if (rightClickMenu == null) rightClickMenu = new JPopupMenu();
    return rightClickMenu;
  }
  public JPopupMenu makeRightClickMenu(HashSet buttons) {
    if (buttons == null) return null;
    ensureRightClickMenu();
    //add each button to the menu
    for (Iterator i = buttons.iterator(); i.hasNext(); ) {
      Object but = i.next();
      if (but != null) {
	JMenuItem item = null;
	if (but instanceof JMenuItem) item = (JMenuItem)but; //unlikely
        else if (but instanceof AbstractButton) {
	  item = new ListRightClickMenuItem((AbstractButton)but);
	  item.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent ae) { 
	     ((ListRightClickMenuItem)ae.getSource()).getButton().doClick();
	    }
	  });
	  item.setActionCommand(((AbstractButton)but).getActionCommand());
	}
	rightClickMenu.add(item);
      }
    }
    itemComponent.addMouseListener(new MouseAdapter() {
       public void mousePressed(MouseEvent me) { 
	   //Debug.noteln("AEdPan: mouse pressed");
	 if (me.isPopupTrigger()) 
	   rightClickMenu.show(me.getComponent(), me.getX(), me.getY());
	 //else Debug.noteln(" No popup trigger", me);
       }
       public void mouseReleased(MouseEvent me) { 
	   //Debug.noteln("AEdPan: mouse released");
	 if (me.isPopupTrigger()) 
	   rightClickMenu.show(me.getComponent(), me.getX(), me.getY());
	 //else Debug.noteln(" No popup trigger", me);
       }
     });
     return rightClickMenu;
  }

  public void addComponentListener(EventListener ml) {
      UIUtil.addListener(itemComponent, ml);
  }

  /** Sets the panel's label */
  public void setLabel(String label) {
    if (isBordered) border.setTitle(label);
    else nameLabel.setText(label);
  }
      
  /**
   * Gets the component of the panel.
   *
   * @return the JComponent that is at the left of the panel
   */
  public JComponent getItemComponent(){
    return this.itemComponent;
  }

  public boolean isListItem() {
    return isListItem(this.itemComponent);
  }
  public static boolean isListItem(JComponent component) {
    return ((component instanceof JList)
	    || (component instanceof JComboBox)
	    || (component instanceof JTable));
  }
  public static boolean isScrollItem(JComponent component) {
    return ((component instanceof JList)
	    || (component instanceof JTextArea)
	    || (component instanceof JTable));
  }

  /** Gets the panel's name (the label) */
  public String getName(){ return this.nameLabel.getText(); }

  /**
   * Makes the item (not) editable and enables (disables) the buttons 
   * and the component (needed for right-click)
   */
  public void setEnabled(boolean isEditable) {
    if (butPanel != null) this.butPanel.setEnabled(isEditable);
    if (itemComponent != null) itemComponent.setEnabled(isEditable);
  }


  private Dimension oldDim = null;
  /**
   * Minimises the component and the rest of the editor
   */
  public void minimise() {
    oldDim = getPreferredSize();
    if (scrollPan != null) scrollPan.setVisible(false);
    else itemComponent.setVisible(false);
    if (butPanel != null) butPanel.setVisible(false);
    minBut.setVisible(false);
    deminBut.setVisible(true);
    adviseMinimise();
    setPreferredSize(new Dimension((int)oldDim.getWidth(), 
				   nameLabel.getHeight()));
    revalidate();
    adjustMinParent(getParent());
  }
  public void deMinimise() {
    if (scrollPan != null) scrollPan.setVisible(true);
    else itemComponent.setVisible(true);
    if (butPanel != null) butPanel.setVisible(true);
    minBut.setVisible(true);
    deminBut.setVisible(false);
    adviseMinimise();
    setPreferredSize(oldDim);
    revalidate();
    adjustDeMinParent(getParent());
  }

  public void smaller() {
    //Debug.noteln("Making smaller");
    Dimension dim = getPreferredSize();
    double w = dim.getWidth();
    double h = dim.getHeight() - dim.getHeight()/3;
    dim.setSize(w, h);
    setPreferredSize(dim);
    revalidate();
    adjustParent(getParent());
  }
  public void larger() {
    //Debug.noteln("Making larger");
    Dimension dim = getSize();
    double w = dim.getWidth();
    double h = dim.getHeight() + dim.getHeight()/2;
    dim.setSize(w, h);
    setPreferredSize(dim);
    revalidate();
    adjustParent(getParent());
  }

  GridBagConstraints oldGBC = null;
  private void adjustMinParent(Container parent) {
    try {
      LayoutManager lm = parent.getLayout();
      if (lm instanceof GridBagLayout) {
	GridBagLayout layMan = (GridBagLayout)lm;
	GridBagConstraints gbc = layMan.getConstraints(this);
	oldGBC = (GridBagConstraints)gbc.clone();
	gbc.fill = GridBagConstraints.HORIZONTAL;
	gbc.weighty = 0.0;
	layMan.setConstraints(this, gbc);
      }
      adjustParent(parent);
    }
    catch (Exception e) {}
  }
  private void adjustDeMinParent(Container parent) {
    try {
      LayoutManager lm = parent.getLayout();
      if (lm instanceof GridBagLayout) {
	GridBagLayout layMan = (GridBagLayout)lm;
	layMan.setConstraints(this, oldGBC);
      }
      adjustParent(parent);
    }
    catch (Exception e) {}
  }

  private void adjustParent(Container parent) {
    parent.doLayout();
    parent.validate();
    parent.repaint();
  }

  public void adviseMinimise() {
    if (!isMinimisable()) return;
    if (isMinimised()) {
      if (hasData()) adviseMinToMax();
      else adviseMinOk();
    }
    else if (!hasData()) adviseMaxToMin();
    else adviseMaxOk();
  }

  private void adviseMaxOk() {
    if (minBut != null) {
      minBut.setIcon(minIcon);
      paintImmediately(minBut.getBounds());
    }
  }
  private void adviseMaxToMin() {
    //Debug.noteln("minimise", nameLabel.getText());
    /*
    minBut.setIcon(xMinIcon);
    paintImmediately(minBut.getBounds());
    */
    adviseMaxOk(); //no red for this!!
  }
  private void adviseMinOk() {
    if (deminBut != null) {
      deminBut.setIcon(deminIcon);
      deminBut.repaint();
    }
  }
  private void adviseMinToMax() {
    if (deminBut != null) {
      deminBut.setIcon(xDeminIcon);
      deminBut.repaint();
    }
  }

  public boolean isMinimised() {
    if (scrollPan != null) return !scrollPan.isVisible();
    else {
      JComponent comp = getItemComponent();
      if (comp != null) return !comp.isVisible();
      else return true; //nothing there, so assume it's gone
    }
  }
  public boolean isMinimisable() {
    return ((minBut != null) && (minBut.isVisible() || deminBut.isVisible()));
  }

  public boolean hasData() {
    Object data = getData();
    //Debug.noteln("AEdP: has data", data);
    return !UIUtil.isEmptyThing(data);
  }

  public void dataSet(Object data) {
    adviseMinimise();
  }

  public abstract Object getData();
  public abstract void setData(Object data);
  public abstract void removeData();
  public abstract boolean setRenderer(ListCellRenderer r);




    public class ListRightClickMenuItem extends JMenuItem {
	//public ListRightClickMenuItem() {}
	private AbstractButton button;
	public ListRightClickMenuItem(AbstractButton button) {
	    super(button.getText());
	    this.button = button;
	}
	public AbstractButton getButton() {return button;}
    }

}


  /*
  private void minimiseList() {
    ((JList)getItemComponent()).setVisibleRowCount(1);
    
  }

  //neither of these work!
  private void minimiseText() {
    try {
      JTextArea text = (JTextArea)getItemComponent();
      text.setRows(1);
      text.revalidate();
    }
    catch (Exception e) {}
  }
  private void minimiseTree() {
    ((JTree)getItemComponent()).setVisibleRowCount(1);
  }
  private void minimiseTable() {
    JTable table = (JTable)getItemComponent();
    Dimension dim = table.getPreferredScrollableViewportSize();
    dim.setSize(dim.getWidth(), table.getRowHeight());
    table.setPreferredScrollableViewportSize(dim);
  }
  */
