/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Sep 22 15:10:51 2009 by Jeff Dalton
 * Copyright: (c) 2005 - 2009, AIAI, University of Edinburgh
 */

package ix.util.http;

import org.jdom.Document;
import org.jdom.output.XMLOutputter;

import java.net.*;
import java.io.*;
import java.util.*;

import ix.ip2.Ip2XMLConfig;

import ix.util.ipc.ServiceAddress;

import ix.util.*;
import ix.util.xml.*;

/**
 * Provides some utility methods for HTTP-related code.
 */
public class HttpUtilities {

    private XMLTranslator xmlt =
        new CompactXMLTranslator(new Ip2XMLConfig().defaultClassSyntax());

    // { xmlt.setOmitImplAttributes(false); }

    public HttpUtilities() {
    }

    /**
     * Returns the name of the host machine that is running the application.
     *
     * @see Util#getHostName()
     */
    public String getHostName() {
	try { return Util.getHostName(); }
	catch (UnknownHostException e) {
	    throw new RethrownException(e);
	}
    }

    public URL makeMessageURL(ServiceAddress addr, String path) {
	try {
	    return new URL("http", addr.getHost(), addr.getPort(), path);
	}
	catch (MalformedURLException e) {
	    throw new RethrownException
		("Can't make a URL from " + addr + ". ", e);
	}
    }

    public URL makeMessageURL(URI serverBase, String path) {
        try {
            return serverBase.resolve(path).toURL();
        }
	catch (MalformedURLException e) {
	    throw new RethrownException
		("Can't make a URL from " + serverBase +
                 " and " + path + ". ", e);
	}
    }

    public URI followServerPointer(String pointer) {
        try {
            Queue<String> lines = Util.fileLines(pointer);
            if (lines.isEmpty())
                throw new IllegalArgumentException
                    (pointer + " is empty.");
            String baseURI = lines.element();
            if (baseURI.endsWith("/"))
                return URI.create(baseURI);
            else
                return URI.create(baseURI + "/");
        }
        catch (Exception e) {
            throw new RethrownException
                ("Problem with ipc-server-pointer " + pointer + ":",
                 e);
        }
    }

    /**
     * Encode an object using the I-X XML representation and
     * the UTF-8 character set.
     */
    public byte[] encodeForSend(Object obj) {
	return encodeForSend(obj, "UTF-8");
    }

    /**
     * Encode an object using the I-X XML representation and
     * the specified character set.
     */
    public byte[] encodeForSend(Object obj, String charsetName) {
	Document doc = xmlt.objectToDocument(obj);
	ByteArrayOutputStream bytes = new ByteArrayOutputStream(2048);
	try {
	    OutputStreamWriter out = new OutputStreamWriter(bytes, charsetName);
	    XMLOutputter outputter = XML.makePrettyXMLOutputter();
	    outputter.output(doc, out);
	    out.close();
	}
	catch (IOException e) {
	    throw new RethrownException(e);
	}
	return bytes.toByteArray();
    }

    public Object decodeReceived(String contents) {
	if (contents == null)
	    return null;
	else
	    return xmlt.objectFromXML(contents);
    }

    public String getContentCharset(String contentType) {
	String cse = "charset=";
	int i = contentType.indexOf(cse);
	if (i > 0)
	    return contentType.substring(i + cse.length());
	else
	    return null;
    }

    /**
     * @throws EOFException if len > 0 and there's less content than expected.
     * @throws IOException for various other reasons.
     */
    public String readContent(InputStream in, int len, String charset)
	   throws IOException {
	if (len < 0) {
	    Debug.noteln("No contet length; will read everything.");
	    Reader r = charset == null
		? new InputStreamReader(in)
		: new InputStreamReader(in, charset);
	    return readAllLines(r);
	}
	else {
	    return readText(in, len, charset);
	}
    }

    /**
     * @throws EOFException if there's less content than expected.
     * @throws IOException for various other reasons.
     */
    public String readText(InputStream in, int len, String charset)
	   throws IOException {
	byte[] content = new byte[len];
	int need = len;
	int at = 0;
	while (need > 0) {
	    int obtained = in.read(content, at, need);
	    if (obtained == -1) { // end reached prematurely
		throw new EOFException();
	    }
	    need -= obtained;
	    at += obtained;
	}
	return charset == null
            ? new String(content)
            : new String(content, charset);
    }

    public String readAllLines(Reader r) throws IOException {
	BufferedReader buf = r instanceof BufferedReader
	    ? (BufferedReader)r
	    : new BufferedReader(r);
	List lines = new LinkedList();
	String line;
	while ((line = buf.readLine()) != null)
	    lines.add(line);
	return Strings.joinLines(lines);
    }

}
