/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu May 28 14:00:51 2009 by Jeff Dalton
 * Copyright: (c) 2005 - 2009, AIAI, University of Edinburgh
 */

package ix.util.http;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.net.*;
import java.io.*;
import java.util.*;

import ix.util.*;

/**
 * Handles HTTP requests that can be regarded as sending a string
 * and receiving one in reply.
 *
 * <p>An HttpStringServelt is like an {@link HttpStringClient} but on the
 * server side.   Where a client sends a string, and receives a string reply,
 * the servlet receives a string, and sends one back.
 *
 * <p>At present, only POST requests are supported.
 *
 * @see HttpServer
 */
public abstract class HttpStringServlet extends HttpObjectServlet {

    public HttpStringServlet() {
	this.responseContentType = "text/plain";
    }

    @Override
    protected String handleRequest(HttpServletRequest req,
				   Object content)
	      throws HttpRequestException {
        return handleRequest(req, (String)content);
    }

    protected abstract String handleRequest(HttpServletRequest req,
                                            String content)
        throws HttpRequestException;

    @Override
    protected byte[] encodeForSend(Object obj)
	throws UnsupportedEncodingException {
	return ((String)obj).getBytes(responseCharsetName);
    }

    @Override
    protected String decodeReceived(String contents) {
	return contents;
    }

    /**
     * Main program for testing.
     */
    public static void main(String[] argv) {

	Parameters.processCommandLineArguments(argv);

	class TestServlet extends HttpStringServlet {
	    protected String handleRequest(HttpServletRequest req,
					   String content) {
		return "Thank you for '" + content + "'.";
	    }
	}

        class EchoServlet extends HttpStringServlet {
	    protected String handleRequest(HttpServletRequest req,
					   String content) {
		return content;
	    }
        }

	int port = Parameters.getInt("http-server-port", 0);
	HttpServer server = new HttpServer(port);
	server.addServlet(new TestServlet(), "/take");
	server.addServlet(new EchoServlet(), "/echo");
	server.addServlet(new HelloServlet("Hi!"), "/hello");
	server.start();

	//\/: N.B. HelloServlet doesn't handle POSTs.

    }

}
