/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Jun  4 17:14:14 2009 by Jeff Dalton
 * Copyright: (c) 2005 - 2009, AIAI, University of Edinburgh
 */

package ix.util.http;

import java.net.*;
import java.io.*;
import java.util.*;

import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.Lisp;	// for testing

/**
 * Makes HTTP requests that can be regarded as sending an object
 * and receiving one in reply.  The usual I-X XML encoding is used;
 * however, it's possible to change that in a subclass.
 *
 * <p>Certain reasonable assumptions are made about the requests:
 * <ul>
 * <li>The Content-Type when sending is "application/xml".
 * <li>The charset is UTF-8.
 * </ul>
 *
 * Both can be changed by calling "set" methods, or in a subclass.
 *
 * <p>Connect and read timeouts may also be set.
 *
 * <p>Both POST and GET requests can be made.
 *
 * @see HttpObjectServlet
 * @see HttpServer
 * @see XMLTranslator
 */
public class HttpObjectClient {

    protected HttpUtilities util = new HttpUtilities();

    protected String requestContentType = "application/xml";
    protected String requestCharsetName = "UTF-8";

    protected int connectTimeout = 0;
    protected int readTimeout = 0;

    public HttpObjectClient() {
    }

    /**
     * Set the Content-Type used when sending requests.
     */
    public void setRequestContentType(String type) {
	this.requestContentType = type;
    }

    /**
     * Set the character set used when sending requests.
     */
    public void setRequestCharsetName(String name) {
	this.requestCharsetName = name;
    }

    /**
     * Sets the timeout in millisecond for opening a communications link
     * when sending a request.  A timout of zero means to wait forever
     * and is the default.  A java.net.SocketTimeoutException is thrown
     * if the timeout expires.
     */
    public void setConnectTimeout(int milliseconds) {
        this.connectTimeout = milliseconds;
    }

    /**
     * Sets the timeout in millisecond for reading a response.
     * A timout of zero means to wait forever and is the default.
     * A java.net.SocketTimeoutException is thrown if the timeout
     * is exceeded before data is available to read.
     */
    public void setReadTimeout(int milliseconds) {
        this.readTimeout = milliseconds;
    }

    /**
     * Send an object and receive one in reply.  The usual I-X XML
     * encoding is used.  This makes a POST request.
     *
     * @see #sendGetRequest(URL)
     */
    public Object sendRequest(URL url, Object contentsToSend)
	throws RethrownIOException {
	try {
	    return do_sendRequest(url, contentsToSend);
	}
	catch (IOException e) {
	    throw new RethrownIOException
		("Trouble requesting an object from " + url + ":", e);
	}
    }

    private Object do_sendRequest(URL url, Object contentsToSend)
	throws IOException, HttpRequestException {

	Debug.noteln("Sending to", url);
	Debug.noteln("Contents:", contentsToSend);

	// Convert the content to bytes so that we can see
	// how long it is.
	byte[] bytes = encodeForSend(contentsToSend);

	// Set up for a connection to the URL.
	HttpURLConnection conn = (HttpURLConnection)url.openConnection();

        // Set timeouts.
        conn.setConnectTimeout(connectTimeout);
        conn.setReadTimeout(readTimeout);

	// Send.
	sendBytes(bytes, url, conn);

	// Describe the response (debug output).
	describeResponse(conn);

	// Read the reply.
	if (handleResponseCode(conn))
	    return decodeReceived(readReply(conn));
	else
	    return null;

    }

    /**
     * Turns an object into request contents.
     *
     * @see #decodeReceived(String contents)
     */
    protected byte[] encodeForSend(Object contents)
	throws UnsupportedEncodingException {
	return util.encodeForSend(contents, requestCharsetName);
    }

    /**
     * Utility method that writes bytes to an HttpURLConnection.
     * The URL is used to set the requests "Host" header field.
     */
    protected void sendBytes(byte[] bytes, 
			     URL url,
			     HttpURLConnection conn)
	throws IOException {

	conn.setDoInput(true);
	conn.setDoOutput(true);
	conn.setUseCaches(false);
	conn.setRequestMethod("POST");
	conn.setRequestProperty("Content-Length", ""+bytes.length);
	conn.setRequestProperty
	    ("Content-Type", requestContentType +
	                         "; charset=" + requestCharsetName);
        int requestPort = url.getPort();
        if (requestPort > 0)
            conn.setRequestProperty
                ("Host", url.getHost() + ":" + url.getPort());
        else
            conn.setRequestProperty
                ("Host", url.getHost());

	// Connect.
	conn.connect();

	// Send the request.
	OutputStream out = conn.getOutputStream();
	out.write(bytes);
	out.flush();
	out.close();

        Debug.noteln("Request sent to", url);

    }

    /**
     * Send a GET request and get an object in reply.  The usual
     * I-X XML syntax should be used in the reply.  To send an object
     * using a POST request, use {@link #sendRequest(URL, Object)}.
     */
    public Object sendGetRequest(URL url)
	throws RethrownIOException {
	try {
	    return do_sendGetRequest(url);
	}
	catch (IOException e) {
	    throw new RethrownIOException
		("Trouble requesting an object from " + url + ":", e);
	}
    }

    private Object do_sendGetRequest(URL url)
	throws IOException, HttpRequestException {

	Debug.noteln("Sending to", url);

	// Set up for a connection to the URL.
	HttpURLConnection conn = (HttpURLConnection)url.openConnection();

        // Set timeouts.
        conn.setConnectTimeout(connectTimeout);
        conn.setReadTimeout(readTimeout);

	// Setup.
	conn.setDoInput(true);
	conn.setUseCaches(false);
	conn.setRequestMethod("GET");
        int requestPort = url.getPort();
        if (requestPort > 0)
            conn.setRequestProperty
                ("Host", url.getHost() + ":" + url.getPort());
        else
            conn.setRequestProperty
                ("Host", url.getHost());

	// Connect.
	conn.connect();

        Debug.noteln("Request sent to", url);

	// Describe the response (debug output).
	describeResponse(conn);

	// Read the reply.
	if (handleResponseCode(conn))
	    return decodeReceived(readReply(conn));
	else
	    return null;

    }

    /**
     * Utility method that produces debugging output that describes
     * the response.
     */
    protected void describeResponse(HttpURLConnection conn)
	throws IOException {
	// Describe the response.
        Debug.noteln("Read timeout", conn.getReadTimeout());
	Debug.noteln("\nResponse:",
		     conn.getResponseCode() + " " +
		         conn.getResponseMessage());
	Debug.noteln("\nReply Headers:");
	for (Iterator i = conn.getHeaderFields().entrySet().iterator()
		 ; i.hasNext(); ) {
	    Map.Entry e = (Map.Entry)i.next();
	    Debug.noteln(e.getKey() + ": " + e.getValue());
	}
    }

    /** 
     * Checks whether the response is a success or a failure and
     * whether there's content to be read.  This method just gets
     * the response code (status) and message from the connection
     * and calls {@link #handleResponseCode(int status, String message)}.
     */
    protected boolean handleResponseCode(HttpURLConnection conn)
	      throws HttpRequestException, 
		     IOException {
	int status = conn.getResponseCode();         // can throw IOException
	String message = conn.getResponseMessage();  // can throw IOException
	return handleResponseCode(status, message);
    }

    /**
     * Checks whether the response is a success or a failure and
     * whether there's content to be read.  Failure is indicated
     * by throwing an {@link HttpRequestException}; otherwise,
     * the true/false result indicates whether there's content
     * to be read or not.  It's possible to have a successful
     * request that does not return any content; normally, the
     * response code is then NO_CONTENT (204).
     *
     * <p>As defined in the HttpObjectClient class, this method returns
     * true if the status is HTTP_OK and throws an HttpRequestException
     * if it isn't.  Subclasses should override this method if they
     * want to accept anything other than HTTP_OK or if they want to
     * return false in some case.
     *
     * @return true if the response indicates there's content to be read,
     *         otherwise false.
     *
     * @throws HttpRequestException for more serious failures.
     */
    protected boolean handleResponseCode(int status, String message)
	      throws HttpRequestException {
	if (status == HttpURLConnection.HTTP_OK)
	    return true;
	else {
	    Debug.noteln("Rejecting response " + status + " " + message);
	    throw new HttpRequestException(status, message);
	}
    }

    /**
     * Utility method that reads a reply from the connection and
     * returns it as a String.
     *
     * N.B. Can return null in NO_CONTENT cases.
     */
    protected String readReply(HttpURLConnection conn)
	throws EOFException,            // from util.readContent(...)
               IOException {

	// Read the reply contents.
	InputStream in = conn.getInputStream();
	String contentType = conn.getContentType();
	String charset = util.getContentCharset(contentType);
	int len = conn.getContentLength();
	try {
	    String result = util.readContent(in, len, charset);
	    Debug.noteln("\nReply:\n", result);
	    return result;
	}
	finally {
	    in.close();
	}

    }

    /**
     * Turns response contents into an object.  For example, the
     * content might be an XML representation of an object.
     *
     * @see #encodeForSend(Object contents)
     */
    protected Object decodeReceived(String contents) {
	return util.decodeReceived(contents);
    }

    /**
     * Main program for testing.
     */
    public static void main(String[] argv) throws Exception {
	
	class LispParser implements Fn1<String,Object> {
	    public Object funcall(String arg) {
		return Lisp.readFromString(arg);
	    }
	}

	mainLoop(argv, new HttpObjectClient(), new LispParser());

    }

    protected static void mainLoop(String[] argv,
				   HttpObjectClient http,
				   Fn1<String,Object> preprocess)
              throws Exception {

	Parameters.processCommandLineArguments(argv);

	URL url = new URL(Parameters.getParameter("url"));

	for (;;) {
	    String contentText = Util.askLines("\nContent:");
	    if (contentText.equals(""))
		return;
	    Object content = preprocess.funcall(contentText);
	    Object reply = http.sendRequest(url, content);
	    System.out.println(reply.getClass() + " reply:\n");
	    System.out.println(Strings.quote(reply.toString()));
	}

    }

}
