/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Aug 25 15:11:03 2010 by Jeff Dalton
 * Copyright: (c) 2007 - 2009, AIAI, University of Edinburgh
 */

package ix.iserve.ipc.sl;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.io.*;
import java.net.*;

import java.util.*;
import java.util.concurrent.*;

import ix.icore.*;

import ix.ichat.ChatMessage;

import ix.iserve.IServe;

import ix.iserve.ipc.*;

import static ix.iserve.ipc.sl.SLHttpHeader.*;

import ix.test.LongToBytes;

import ix.test.xml.VirtualWorld;

import ix.iface.util.Reporting;

import ix.util.*;
import ix.util.ipc.*;
import ix.util.xml.*;
import ix.util.http.*;
import ix.util.lisp.*;

import ix.util.PropertyLogger.LoggingMap;

/**
 * Handles messages for a {@link IServeCommStrategy}.
 */
public class IServeSLCommServer extends IServeCommServer {

    // Header properties to record in message-wrappers
    final List<SLHttpHeader> headerProperties =
        Lisp.list(XSL_OBJECT_NAME, XSL_OBJECT_KEY,
                  XSL_REGION,
                  XSL_OWNER_NAME, XSL_OWNER_KEY);

    private final IdentityCheckerFactory slIdentityCheckerFactory =
        new SLIdentityCheckerFactory();

    private final WorldTable worlds = new WorldTable();

    private PropertyLogger propertyLogger;

    public IServeSLCommServer() {
        super();
    }

    /** Constructor used when making an {@link IXAgentExtension}. */
    public IServeSLCommServer(IServe containingAgent) {
        super(containingAgent);
    }

    public static void main(String[] argv) {
	Util.printGreeting("IServe Message Server");
        do_main(argv, IServeSLCommServer.class);
    }

    @Override
    protected void processCommandLineArguments() {
        super.processCommandLineArguments();

        // Read world descriptions
        String worldsResource = Parameters.getParameter("virtual-worlds");
        if (worldsResource != null) {
            for (Object w: XML.readObject(List.class, worldsResource))
                worlds.add(Util.mustBe(VirtualWorld.class, w));
        }

        // Create a PropertyLogger and read any logged proeprties.
        String propFileName = "logs/" + getName() + "-properties";
        propertyLogger = new PropertyLogger(propFileName);
        propertyLogger.readLogIfExists();
    }

    @Override
    protected void addServlets() {
        super.addServlets();
        addServlet(new SLRegistrationServelt(), "/ipc/sl/register");
        addServlet(new SLSendServlet(), "/ipc/sl/send");
	addServlet(new SLNextMessageServlet(), "/ipc/sl/get-next");
        addServlet(new SLChatbotTestServlet(), "/ipc/sl/chat");
        addServlet(new SLReplyServlet(), "/ipc/sl/reply");
    }

    class SLRegistrationServelt extends HttpStringServlet {

        SLRegistrationServelt() {
        }

	@Override
	protected String handleRequest(HttpServletRequest req,
				       String contents)
	          throws HttpRequestException {
            MessageWrapper w = makeMessageWrapper(Command.REGISTER, req);
            w.setContents(contents);
	    ensureSLUser(w.getFrom())
                .register(w);
            return "OK";
        }

    }

    class SLSendServlet extends HttpStringServlet {

        SLSendServlet() {
        }

	@Override
	protected String handleRequest(HttpServletRequest req,
				       String contents)
	          throws HttpRequestException {
            MessageWrapper w = makeMessageWrapper(Command.SEND_TO, req);
            ensureSLUser(w.getFrom())
                .checkForSend(w);
            // /\/: Need to decode contents sent from SL ...
	    ensureUser(w.getTo())
		.addMessage(w);
            return "OK";
        }

    }

    class SLNextMessageServlet extends HttpStringServlet {

	SLNextMessageServlet() {
	}

	@Override
	protected String handleRequest(HttpServletRequest req,
				       String contents)
	          throws HttpRequestException {
	    // Debug.noteln("Server received", contents);
            MessageWrapper w = makeMessageWrapper(Command.GET_MESSAGE, req);
            User slUser = ensureSLUser(w.getFrom());
            setSequenceNunberIfPresent(w, contents);
	    Object reply = slUser.nextMessage(w);
            Debug.noteln("SLNextMessageServlet obtained", reply);
	    if (reply instanceof AbandonRequest) {
		Debug.noteln("Told to abandon request");
                throw (HttpRequestException)reply;
	    }
	    Debug.noteln("Sending next message to", w.getFrom());
            MessageWrapper r = (MessageWrapper)reply;
            //\/: If the wrapper didn't specify an ack, assume one
            // for the message we're about to send.  It's really the
            // next get-message request that should ack the message
            // we're about to send, but ... /\/
//             if (w.getSequenceNumber() == null)
//                 slUser.assumeAcknowledged(r);
            // /\/: Need to encode the reply for SL ...
            String replyContents = encodeForSL(r.getContents());
            long seqNo = r.getSequenceNumber().longValue();
            String seqId = LongToBytes.encode(seqNo);
            replyContents = seqId + "^" + replyContents;
            Debug.noteln("Will send " + slUser, replyContents);
            return replyContents;
	}
        void setSequenceNunberIfPresent(MessageWrapper w, String contents) {
            if (contents.startsWith("Received ")) {
                String seqId = Strings.afterFirst("Received ", contents);
                long seqNo = LongToBytes.decode(seqId);
                w.setSeqNo((int)seqNo);
            }
        }
        String encodeForSL(Object contents) {
            // At present, we just return a string, without any indication
            // of what it represents.  /\/
            if (contents instanceof ChatMessage)
                return ((ChatMessage)contents).getText();
            else if (contents instanceof Activity)
                return PatternParser
                    .unparse(((Activity)contents).getPattern());
            else if (contents instanceof Issue)
                return PatternParser
                    .unparse(((Issue)contents).getPattern());
            else if (contents instanceof Report)
                return ((Report)contents).getText();
//          else if (contents instanceof Constraint)
//              return ;
            else
                throw new IllegalArgumentException
                    ("Can't encode for SL: " + contents);
        }

    }

    /**
     * Sends a {@link ChatMessage} to whatever agent has registered
     * as "chat-service".
     */
    class SLChatbotTestServlet extends HttpStringServlet {

        SLChatbotTestServlet() {
        }

	@Override
	protected String handleRequest(HttpServletRequest req,
				       String contents)
	          throws HttpRequestException {
            MessageWrapper w = makeMessageWrapper(Command.SEND_TO, req);
            w.setSendDate(new Date()); // /\/
            ChatMessage chat = new ChatMessage(contents, w.getFrom());
            w.setContents(chat);
            w.setTo("chat-service"); // /\/
	    ensureSLUser(w.getFrom())
                .checkForSend(w);
            ensureUser(w.getTo())
                .addMessage(w);
            return "OK";
        }

    }

    /**
     * Sends a reply to whoever sent the last message the replying
     * agent has received.
     */
    class SLReplyServlet extends HttpStringServlet {

        SLReplyDecoder decoder = new SLReplyDecoder();

        SLReplyServlet() {
        }

	@Override
	protected String handleRequest(HttpServletRequest req,
				       String contents)
	          throws HttpRequestException {
            MessageWrapper w = makeMessageWrapper(Command.SEND_TO, req);
            w.setSendDate(new Date()); // /\/
	    User user = ensureSLUser(w.getFrom());
            synchronized (user) {
                user.checkForSend(w);
                MessageWrapper last = user.getLastAckdMessage();
                // Fill-in contents and destination.
                try {
                    decoder.fillInReply(w, contents, last);
                }
                catch (Exception e) {
                    throw new HttpRequestException
                        (HttpURLConnection.HTTP_BAD_REQUEST,
                         Debug.describeException(e));
                }
            }
            ensureUser(w.getTo())
                .addMessage(w);
            return "OK";
        }

    }

    protected MessageWrapper makeMessageWrapper(Command command,
                                                HttpServletRequest req)
              throws HttpRequestException {
        MessageWrapper w = new MessageWrapper();
        w.setCommand(command.text());
        w.setFrom(XSL_OBJECT_NAME.requireHeader(req));
        fillInRequestInfo(w, req);
        for (SLHttpHeader h: headerProperties) {
            String value = h.getHeader(req);
            if (value != null && value.indexOf("Loading...") < 0)
                setHeaderAnnotation(w, h, value);
        }
        VirtualWorld world = worlds.findWorld(req);
        if (world != null)
            w.setAnnotation(VIRTUAL_WORLD, world);
        return w;
    }

    protected void setHeaderAnnotation(MessageWrapper w,
                                       SLHttpHeader h,
                                       String value) {
        w.setAnnotation(h.text(), value);
    }

    protected String getHeaderAnnotation(MessageWrapper w, SLHttpHeader h) {
        return (String)w.getAnnotation(h.text());
    }

    protected User ensureSLUser(String name) {
        // This should be called only for the user who sent the
        // request to this server.
        User user;
	synchronized (userNameToUserMap) {
	    user = getUser(name);
	    if (user == null) {
		user = new User(name);
		recordUser(user);
	    }
        }
        synchronized (user) {
            // The identity-checker is what makes it an SL user
            // rather than an ordinary user.
            user.setIdentityChecker(slIdentityCheckerFactory);
            // See how long we ought to sleep to avoid exceeding SL's
            // limits on how frequent HTTP requests can be.
            Date then = user.getLastContactDate();
            if (then !=  null) {
                long now = System.currentTimeMillis();
                long howLong = now - then.getTime();
                // Estimate how long it takes to get from SL to us & back.
                // long tripTime = howLong * 2;
                long tripTime = howLong;
                // Delay things enough that SL doesn't throttle requests /\/
                if (tripTime < 1000) {
                    try {
                        Thread.sleep(1000 - tripTime);
                    }
                    catch (InterruptedException e) {
                    }
                }
            }
	    return user;
	}
    }

    /*
     * SL User properties
     */

    private static final Symbol
        LAST_WAKEUP_DATE = Symbol.intern("last-wakeup-date"),
        XML_RPC_CHANNEL = Symbol.intern("xml-rpc-channel"),
        VIRTUAL_WORLD = Symbol.intern("virtual-world");

    // Last Wakeup Date

    private Date getLastWakeupDate(User user) {
        return (Date)user.getProperty(LAST_WAKEUP_DATE);
    }

    private void setLastWakeupDate(User user, Date date) {
        user.setProperty(LAST_WAKEUP_DATE, date);
    }

    // Wakeup Channel

    private String getWakeupChannel(User user) {
        String result = (String)user.getProperty(XML_RPC_CHANNEL);
        return result != null
            ? result
            : (String)propertyLogger.get(XML_RPC_CHANNEL.toString(),
                                         user.getName());
    }

    private void setWakeupChannel(User user, String channelId) {
        user.setProperty(XML_RPC_CHANNEL, channelId);
        propertyLogger.putIfNew(XML_RPC_CHANNEL.toString(),
                                user.getName(),
                                channelId);
    }

    // Virtual World

    private VirtualWorld getVirtualWorld(Annotated a) {
        // This method is used to get it from MessageWrappers.  /\/
        // Should Users be Annotated? /\/
        return (VirtualWorld)a.getAnnotation(VIRTUAL_WORLD);
    }

    private VirtualWorld getVirtualWorld(User user) {
        VirtualWorld result = (VirtualWorld)user.getProperty(VIRTUAL_WORLD);
        if (result != null)
            return result;
        else {
            String worldName = propertyLogger.get(VIRTUAL_WORLD.toString(),
                                                  user.getName());
            if (worldName != null)
                return worlds.worldNamed(worldName);
            else
                return null;
        }
    }

    private void setVirtualWorld(User user, VirtualWorld world) {
        user.setProperty(VIRTUAL_WORLD, world);
        propertyLogger.putIfNew(VIRTUAL_WORLD.toString(),
                                user.getName(),
                                world.getName());
    }

    /*
     * Identity-checker
     */

    protected class SLIdentityCheckerFactory
              implements IdentityCheckerFactory {

        @Override
        public boolean existingCheckerIsOk(IdentityChecker checker) {
            return checker instanceof SLIdentityChecker;
        }

        @Override
        public IdentityChecker makeIdentityChecker(User user) {
            return new SLIdentityChecker(user);
        }

    }

    protected class SLIdentityChecker extends IdentityChecker {

        protected String ownerKey = null;
        protected String ownerName = null;

        SLIdentityChecker(User user) {
            super(user);
        }

        @Override
        public void changingIdentityCheckerFrom(IdentityChecker oldChecker) {
            user.noteln("Changing identity-checker from " + oldChecker +
                        " to " + this);
        }

        @Override
        public void init() {
            // This is our chance to do some SL-user-specific init.  /\/
        }

        @Override
        public void checkRegisterRequest(MessageWrapper req)
               throws HttpRequestException {
            checkMessageSource(req);
            // /\/: Assume contents is the channel id for XML-RPC
            String channelId = (String)req.getContents();
            setWakeupChannel(user, channelId);
        }

        @Override
        public void checkAddMessage(MessageWrapper req) {
            maybeSendWakeup(user);
        }

        @Override
        public void status(Date now, List<String> lines) {
            Date lastWakeupDate = getLastWakeupDate(user);
	    if (lastWakeupDate != null)
		lines.add("  Last wakeup call: " +
			  agoTime(lastWakeupDate, now));
            if (getWakeupChannel(user) == null)
                lines.add("  No wakeup channel");
            VirtualWorld world = getVirtualWorld(user);
            if (world != null)
                lines.add ("  World: " + world.getName());
            if (ownerName != null)
                lines.add("  Owner name: " + ownerName);
        }

        @Override
        protected void checkMessageSource(MessageWrapper w)
                  throws HttpRequestException {
            // super.checkMessageSource(w);

            // N.B. The Object- (?) and Owner-Name headers sometimes
            // come through as "(Loading...)", in which case we don't
            // attach the annotation.  So w_ownerName can be null.

            String w_ownerKey = getHeaderAnnotation(w, XSL_OWNER_KEY);
            String w_ownerName = getHeaderAnnotation(w, XSL_OWNER_NAME);

            // Check owner key.
            if (ownerKey == null) {
                user.noteln("Recording owner key", w_ownerKey);
                ownerKey = w_ownerKey;
            }
            else if (!w_ownerKey.equals(ownerKey)) {
                user.noteln("Changing owner key to", w_ownerKey);
                ownerKey = w_ownerKey;
            }

            // Check owner name.
            if (ownerName == null) {
                user.noteln("Recording owner name", w_ownerName);
                ownerName = w_ownerName;
            }
            else if (w_ownerName != null && !w_ownerName.equals(ownerName)) {
                user.noteln("Changing owner name to", w_ownerName);
                ownerName = w_ownerName;
            }

            // Check virtual world.
            checkMessageWorld(w);

        }

        protected void real_checkMessageSource(MessageWrapper w)
                  throws HttpRequestException {
            // super.checkMessageSource(w);
            String w_ownerKey = getHeaderAnnotation(w, XSL_OWNER_KEY);
            String w_ownerName = getHeaderAnnotation(w, XSL_OWNER_NAME);
            // Check owner key.
            if (ownerKey == null) {
                user.noteln("Recording owner key", w_ownerKey);
                ownerKey = w_ownerKey;
            }
            else if (!w_ownerKey.equals(ownerKey)) {
                throw new HttpRequestException
                    (HttpURLConnection.HTTP_FORBIDDEN,
                     "Wrong owner key");
            }
            // Check owner name.
            if (ownerName == null) {
                user.noteln("Recording owner name", w_ownerName);
                ownerName = w_ownerName;
            }
            // Check virtual world.
            checkMessageWorld(w);
        }

        private void checkMessageWorld(MessageWrapper req) {
            VirtualWorld req_world = getVirtualWorld(req);
            if (req_world == null)
                user.noteln("Could not find virtual world");
            else {
                VirtualWorld user_world = getVirtualWorld(user);
                if (user_world == null)
                    setVirtualWorld(user, req_world);
                else if (user_world != req_world) {
                    user.noteln("Changing world from " + user_world +
                                " to " + req_world);
                    setVirtualWorld(user, req_world);
                }
            }
        }

    }

    /*
     * Wakeups
     */

    ExecutorService wakeupPool = Executors.newFixedThreadPool(3);

    SLRpc rpcClient = new SLRpc();

    // If the last time the SL side said it received a
    // message was > n minutes, assume it may be asleep.
    // Note that our n here must be < the time the SL
    // side waits before going to sleep, so that there
    // isn't a 'gap' during which it's asleep but we don't
    // try to wake it up.

    // However, since we know the 'receive' script will wake
    // up if it sees a link message, which it will if the object
    // sends to another agent, and will similarly wake up if
    // the object (re)registers, we can look at those dates
    // as well as lastAckDate to see how recently it was awake.
    // For now, we don't look at registration dates.  /\/

    // Note that we cannot use lastContactDate, because it includes
    // get-next requests that didn't get anything, which are what
    // the SL object counts when deciding to sleep.

    // /\/: Maybe reorganize to put the test of whether we think
    // the SL side might be asleep into a separate method.

    void maybeSendWakeup(User user) {
        if (user.getLastContactDate() == null) {
            // If we've not yet heard from the SL side at all,
            // assume we don't need to wake it up.  Perhaps it
            // doesn't even exist yet.  /\/: That will be wrong if
            // it talked with a previous incarnation of this server.
            return;
        }
        // See when the SL side last seemed awake.
        Date lastAck = user.getLastAckDate();
        Date lastSend = user.getLastSendDate();
        Date now = new Date();
        if (lastAck == null && lastSend == null) {
            // If the SL side has not send or received anything,
            // then it may be asleep.
            tryToWakeup(user, now);
            return;
        }
        // See how long it's been.
        // Find the most recent of lastAck and lastSend.
        Date lastAwake = lastAck == null ? lastSend
                       : lastSend == null ? lastAck
                       : lastAck.compareTo(lastSend) > 0 ? lastAck
                       : lastSend;
        long milliscondsAgo = now.getTime() - lastAwake.getTime();
        if (milliscondsAgo >= 2 * 60 * 1000) {
            // If it's long enough ago, the agent may need
            // awakening.
            tryToWakeup(user, now);
        }
    }

    void tryToWakeup(User user, Date now) {
        // We've decided the SL side may need a wakeup, however
        // we don't want to send it one if we've already done that
        // for some other recent message.
        Date lastWakeup = getLastWakeupDate(user);
        if (lastWakeup != null
              && now.getTime() - lastWakeup.getTime()
                   < 1 * 60 * 1000) {
            user.noteln("Skipping wakeup because we've recently sent one.");
            return;
        }
        VirtualWorld world = getVirtualWorld(user);
        // We can't send a wakeup if we don't know the server URL.
        if (world == null) {
            user.noteln("Skipping wakeup because we don't know the world.");
            return;
        }
        if (world.getXmlRpcUrl() == null) {
            user.noteln("Skipping wakeup because we don't know the URL.");
            return;
        }
        // We can't send a wakeup if we don't have a channel.
        if (getWakeupChannel(user) == null) {
            user.noteln("Skipping wakeup because we don't have a channel id.");
            return;
        }
        // Give a wakeup task to the thread pool.
        Debug.noteln("Decided to try to wake up", user);
        setLastWakeupDate(user, now);
        wakeupPool.execute(new Awakener(user));
    }

    class Awakener implements Runnable {
        User user;
        Awakener(User u) {
            this.user = u;
        }
        public void run() {
            Debug.noteln("Trying to wake up", user);
            // Note that by the time we get here, a wakeup may no longer
            // be needed.  /\/
            URL server = getVirtualWorld(user).getXmlRpcUrl();
            String channel = getWakeupChannel(user);
            SLRpcMessage reply;
            try {
                reply = rpcClient.sendRequest(server, channel, 0, "Wake up");
                Debug.noteln("Wakeup reply from " + user + ": " + reply);
            }
            catch (Exception e) {
                Debug.noteln("Problem while trying to wake up", user);
                Debug.noteException(e);
            }
        }
    }


}
