/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jan 27 18:43:32 2010 by Jeff Dalton
 * Copyright: (c) 2003, 2007 - 2009, AIAI, University of Edinburgh
 */

package ix.iserve.ipc;

import java.net.*;
import java.io.*;
import java.util.*;
import java.util.concurrent.*;
import javax.swing.SwingUtilities;

import ix.Release;

import ix.icore.Annotated;
import ix.icore.IXAgent;
import ix.ip2.Ip2;

import ix.iface.util.Reporting;
import ix.iface.util.ToolController;

import ix.util.*;
import ix.util.ipc.ServiceAddress;
import ix.util.xml.*;
import ix.util.http.*;
import ix.util.lisp.*;

/**
 * A communication strategy that sends all messages via a server.
 * An agent that uses this strategy must specify an "ipc-server"
 * parameter as host:port or an "ipc-server-pointer" parameter
 * that's the URL of a file that contains the base URL for
 * messages to the server.
 *
 * <p>The server side is the class {@link IServeCommServer}.</p>
 *
 * @see Parameters
 */
public class IServeCommStrategy implements IPC.CommunicationStrategy {

    // Annotation key
    // /\/: A send-date should be part of Sendable.
    public static Symbol SEND_DATE = Symbol.intern("send-date");

    // protected IXAgent agent;
    protected IPC.MessageListener messageListener;

    protected URI serverBase;
    // protected ServiceAddress serverAddr;
    protected URL sendUrl;
    protected URL nextMessageUrl;
    protected URL registerUrl;

    protected IServeCommTool tool; // a user interface.

    protected ReceiveThread receiveThread = null;

    protected int receiveThreadCount = 0;

    protected boolean ableToSend = false;

    protected String uuid = null;

    // One higher than the highest we've received so far.
    // Should be modified only in the receive thread.  /\/
    // Volatile because 'register' refers to it and is called
    // in the GUI event thread.  /\/
    protected volatile int expectedSeqNo = 0;

    // We need to know which messages we've received, but all
    // we really need to keep is a record of their sequence numbers --
    // the keys in the messageMap. /\/ For now, we are keeping the
    // values as well.
    protected MessageMemory messageMap = new MessageMemory();

    protected HttpObjectClient sendClient = makeSendClient();

    protected HttpUtilities httpUtil = new HttpUtilities();

    public IServeCommStrategy() {
    }

    void setTool(IServeCommTool tool) {
        // Called by the tool when it is constructed.
	// This method exists because the tool doesn't exist until
	// the user asks for it.
	this.tool = tool;
    }

    protected String getAgentName() {
	return IXAgent.getAgent().getAgentSymbolName();
    }

    synchronized boolean isAbleToSend() {
	return ableToSend;
    }

    synchronized void setIsAbleToSend(boolean v) {
	ableToSend = v;
    }

    synchronized String getMessageUrlBase() {
	return Strings.beforeLast("/", sendUrl.toString()); // ? /\/
    }

    public void setupServer(Object destination, IPC.MessageListener listener) {

	messageListener = listener;

        serverBase = determineServerBase();
        Debug.noteln("Comm server base URI", serverBase);

	sendUrl = httpUtil.makeMessageURL(serverBase, "ipc/ix/send");
	nextMessageUrl = httpUtil.makeMessageURL(serverBase,
                                                 "ipc/ix/get-next");
        registerUrl = httpUtil.makeMessageURL(serverBase, "ipc/ix/register");

        sendClient.setReadTimeout(60 * 1000); // 60 seconds

        installCommTool();

        // startReceiving();

    }

    protected void installCommTool() {
	IXAgent agent = IXAgent.getAgent();
        String title = agent.getAgentDisplayName() 
                         + " I-Serve Communications";
        ToolController tc = new IServeCommTool.Controller(this, title);
	agent.addTool(tc);
        tc.ensureTool();
        Debug.expect(tool != null);
    }

    protected void initialCommToolTranscript() {
        // Called by the tool.
        tool.transcript("I-X " + Release.version + " " + Release.date +
                        " " + Release.time);
	tool.transcript("Communications base URL: " + getMessageUrlBase());
    }
    
    protected URI determineServerBase() {
        String pointer = Parameters.getParameter("ipc-server-pointer");
        String ipcServer = Parameters.getParameter("ipc-server");
        if (pointer != null) {
            return httpUtil.followServerPointer(pointer);
        }
        else if (ipcServer != null) {
            ServiceAddress addr = new ServiceAddress(ipcServer);
            return URI.create
                ("http://" + addr.getHost() + ":" + addr.getPort() + "/");
        }
        else
            throw new ParameterException
            ("An ipc-server-pointer or ipc-server parameter " +
             "must be supplied.");
    }

    public void register(String password) {
        
        // Normally called only from the communication tool.
	Debug.expect(SwingUtilities.isEventDispatchThread(), "not in Swing");

	// Debug.expect(!isAbleToSend(), "Registering when already registered");
        transcript("Trying to register with the comm server.");
	Debug.noteln("Trying to register with password", password);

        HttpObjectClient http = sendClient; //\/ was new Httpobjectclient();

	MessageWrapper request =
	    MessageWrapper.makeRegister(getAgentName(), password);

	// Let the message-server know the highest-numbered message
	// we've received (in case the server exited and was restarted).
	// N.B. If we haven't yet received anything, the value will be -1.
	request.setSeqNo(expectedSeqNo - 1);

	request.setSendDate(new Date());
        request.setUUID(uuid);

	Object reply = http.sendRequest(registerUrl, request);
        Registration reg = Util.mustBe(Registration.class, reply);
	if (!reg.isSuccess())
	    throw new IPC.IPCException
               ("Registration failure: " + reg.getStatus());

        uuid = reg.getUUID();

	transcript("Registered as " + getAgentName() + ", uuid = " + uuid);
	// agent.setAgentSymbolName(name);
	setIsAbleToSend(true);
        ensureReceiving();

    }

    public void sendObject(Object destination, Object contents) {

	if (!isAbleToSend())
	    throw new UnsupportedOperationException
		("Cannot send to " + destination +
		 " until you have registered with the message server.");

	MessageWrapper command =
	    MessageWrapper.makeSend(getAgentName(), (String)destination,
                                    contents);

        // Note that we do not set a sequence number in this case.

	command.setSendDate(new Date());
        command.setUUID(uuid);

	Object reply = sendClient.sendRequest(sendUrl, command);
	Debug.noteln("Received", reply);
	if (!reply.equals("OK"))
	    throw new IPC.IPCException
		("Unexpected reply from message-server: " + reply);

    }

    protected HttpObjectClient makeSendClient() {
        return new HttpObjectClient();
    }

    protected HttpObjectClient makeReceiveClient() {
        return new HttpObjectClient();
    }

    synchronized void ensureReceiving() {
        if (receiveThread == null)
            startReceiving();
    }

    protected synchronized void startReceiving() {
        transcript("Preparing to receive messages.");
        if (receiveThread != null)
            throw new ConsistencyException("Non-null receiveThread");
        else {
            receiveThread = new ReceiveThread();
            receiveThread.start();
            // awaitStartedReceiving();
        }
    }

    protected synchronized void awaitStartedReceiving() {
        try {
            receiveThread.awaitStart();
        }
        catch (InterruptedException e) {
            throw new RethrownException
                (e, "Interrupted while starting receiving.");
        }
    }

    public synchronized void stopReceiving() {
        if (receiveThread != null)
            receiveThread.stopRunning();
    }

    protected synchronized void stoppedReceiving() {
        transcriptLater("Stopped receiving.");
        receiveThread = null;
    }

    // N.B. Receive-threads are always created in a synchronized method
    // of the comm strategy object.

    class ReceiveThread extends Thread {

        final HttpObjectClient receiveClient = makeReceiveClient();

	final String ourName = getAgentName();

        final CountDownLatch started = new CountDownLatch(1);

        volatile boolean keepRunning = true;

	MessageWrapper message = null;

        ReceiveThread() {
            super("I-Serve comm receiver " + receiveThreadCount++);
            // receiveClient.setReadTimeout(10 * 1000);
            setDaemon(true);
        }
        public void awaitStart() throws InterruptedException {
            started.await();
        }

        public void stopRunning() {
            keepRunning = false;
            this.interrupt();   // will this help? /\/
        }

	public void run() {

            started.countDown();
            // transcript("Starting to ask for messages.");

            int errs = 0; // for counting failures in a row.

	    messageLoop:
	    while(keepRunning) {

		// Get a message
		try {
		    message = nextMessage(message);
                    errs = 0;
		}
                catch (RethrownIOException io) {
                    try { throw io.getCause(); }
                    catch (SocketTimeoutException timeout) {
                        Debug.noteln("Timeout when requesting next message");
                        continue messageLoop;
                    }
                    catch (ConnectException e) {
                        if (errs < 3) {
                            Debug.noteException(e, false);
                            errs++;
                            Util.sleepSeconds(10);
                            continue messageLoop;
                        }
                        reportProblem("Can't connect to server", e);
                        break messageLoop;
                    }
                    catch (SocketException e) {
                        // See if "Unexpected end of file from server"
                        // or "Connection reset".
                        String m = e.getMessage();
                        if (m.indexOf("end of file") >= 0
                              || m.startsWith("Connection reset")) {
                            Debug.noteException(e, false);
                            continue messageLoop;
                        }
                        reportProblem("Unexpected", e);
                        break messageLoop;
                    }
                    catch (HttpRequestException e) {
                        // Normally the request times out rather than
                        // getting the superseded status result.
//\/: We're not getting the reason that was given on the other end.
//                      if (e.getStatus() == HttpURLConnection.HTTP_ACCEPTED
//                          && e.getReason()
//                              .equals("Request was superseded.")) {
                        if (e.getStatus() == HttpURLConnection.HTTP_ACCEPTED) {
                            Debug.noteException(e, false);
			    continue messageLoop;
			}
                        else if (e.getStatus() ==
                                   HttpURLConnection.HTTP_FORBIDDEN) {
                            // This means the get-next request was from
                            // the wrong host or had the wrong UUID.
                            reportIdentityCheckFailure(e);
                            break messageLoop;
                        }
                        else {
                            reportProblem
                                ("Problem when requesting next message", e);
			    break messageLoop;
			}
                    }
                    catch (Throwable t) {
                        reportProblem
                            ("Problem requesting next message", t);
                        break messageLoop;
                    }
                }
                catch (ThreadDeath t) {
                    // Rethrow so thread exits.  This happens in applets.
                    throw t;
                }
		catch (Throwable t) {
                    reportProblem
			("Problem while requesting next message", t);
		    break messageLoop;
		}

		// Handle the message
		try {
                    handleMessage(message);
		}
		catch (Throwable t) {
                    reportProblem("Problem while handling message", t);
		}

	    }
            stoppedReceiving();
	}

	MessageWrapper nextMessage(MessageWrapper previous) {

	    MessageWrapper request = 
		MessageWrapper.makeGetNextMessage(ourName);

            request.setUUID(uuid);

	    // Let the message-server know we received the previous message.
	    if (previous != null)
		request.setSeqNo(previous.getSeqNo());

	    Object reply = receiveClient.sendRequest(nextMessageUrl, request);
	    if (!(reply instanceof MessageWrapper))
		throw new ClassCastException
		    ("Received an instance of " + reply.getClass().getName() +
		     " instead of a MessageWrapper.");

	    // The message should have the form
	    //   MessageWrapper(fromName, "send-to", [ourName, message])
	    MessageWrapper next = (MessageWrapper)reply;
	    Debug.expect(next.getCommand().equals("send-to"));
	    Debug.expect(next.getTo().equals(ourName));
	    return next;

	}

	void handleMessage(MessageWrapper message) {
	    // Perhaps we've already received this message, but
	    // the server didn't know we had and so sent it again.
	    // Or we may have missed a message while the server
	    // thought it had been sent successfully.
	    Integer mSequence = message.getSequenceNumber();
	    int mSeq = message.getSeqNo();
	    Debug.expect(mSeq == mSequence.intValue());
	    if (mSeq == expectedSeqNo) {
		Debug.expect(!messageMap.containsKey(mSequence));
		expectedSeqNo++;
	    }
	    else if (mSeq > expectedSeqNo) {
		Debug.noteln("Missing messages between " + expectedSeqNo +
			     " and " + mSeq);
		expectedSeqNo = mSeq + 1;
	    }
	    else {
		Debug.expect(mSeq < expectedSeqNo);
		if (messageMap.containsKey(mSequence)) {
		    // We've already received this message.
		    transcript("Received " + mSeq + " again");
		    // So ignore it.
		    return;
		}
		else {
		    // This message fills a gap in the previous messages.
		    Debug.noteln("Received gap message", mSeq);
		}
	    }
	    // We have a new message.
	    Object contents = getReceivedContents(message);
	    transcript("Received " + mSeq + ": " +
		       Reporting.description(contents));
	    messageMap.remember(message); 	//\/ won't GC message
	    messageListener.messageReceived
		(new IPC.BasicInputMessage(contents));
	}

    }

    protected Object getReceivedContents(MessageWrapper message) {
        Object contents = message.getContents();
        if (contents instanceof Annotated) {
            if (message.getSendDate() != null) {
                ((Annotated)contents)
                    .setAnnotation(SEND_DATE, message.getSendDate());
            }
        }
        return contents;
    }

    public void reportProblem(String prefix, Throwable t) {
        transcript(prefix + ": " + Debug.describeException(t));
        Debug.displayException(prefix, t);
    }

    protected void reportIdentityCheckFailure(HttpRequestException e) {
        String[] message = new String[] {
            "Request refused because it had the wrong host or uuid.",
            "You will have to re-register."
        };
        transcript(Strings.joinLines(message));
        Util.displayAndWait(null, message);
    }

    public void transcript(final String line) {
	Debug.noteln("Transcript:", line);
	Util.swingAndWait(new Runnable() {
	    public void run() {
		do_transcript(Reporting.dateString() + " " + line);
	    }
	});
    }

    public void transcriptLater(final String line) {
	Debug.noteln("Transcript Later:", line);
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		do_transcript(Reporting.dateString() + " " + line);
	    }
	});
    }

    protected void do_transcript(String line) {
        try {
            tool.transcript(line);
        }
        catch (Throwable t) {
            Debug.displayException
                ("Problem adding transcript line " + Strings.quote(line), t);
        }
    }

}
