/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Sep 22 15:22:34 2009 by Jeff Dalton
 * Copyright: (c) 2003, 2007 - 2009, AIAI, University of Edinburgh
 */

package ix.iserve.ipc;

// import com.google.gson.Gson;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import java.util.*;
import java.io.*;
import java.net.*;

import ix.icore.IXAgentExtension;

import ix.iserve.IServe;

import ix.iface.util.HtmlWriter;
import ix.iface.util.HtmlStringWriter;
import ix.iface.util.Reporting;

import ix.util.*;
import ix.util.ipc.*;
import ix.util.xml.*;
import ix.util.http.*;
import ix.util.lisp.*;

/**
 * Handles messages for a {@link IServeCommStrategy}.
 */
public class IServeCommServer implements IXAgentExtension {

    private HttpServer httpServer;

    private final HttpUtilities httpUtil = new HttpUtilities();

    private final IdentityCheckerFactory defaultIdentityCheckerFactory =
        new DefaultIdentityCheckerFactory();

    private final IdentityCheckerFactory ixIdentityCheckerFactory =
        new IXIdentityCheckerFactory();

    private final Date startupDate = new Date();

    private final IServe iserve;

    // private final Gson gson = new Gson();

    // private TextAreaFrame textFrame;

    private final int max_waiting_messages_to_list_in_status = 6;

    /** Constructor used when making a standalone comm server. */
    public IServeCommServer() {
        this(null);
    }

    /** Constructor also used when making an {@link IXAgentExtension}. */
    public IServeCommServer(IServe containingAgent) {
        this.iserve = containingAgent;
    }

    /** Called when installing as an {@link IXAgentExtension}. */
    public synchronized void installExtension() {
        httpServer = iserve.getHttpServer();
        Debug.expect(httpServer != null, "No HTTP server");
        processCommandLineArguments();
        addServlets();
    }

    /** 
     * Used to run a standalone comm server.
     */
    public static void main(String[] argv) {
	Util.printGreeting("IServe Message Server");
        do_main(argv, IServeCommServer.class);
    }

    protected static void do_main
            (String[] argv, 
             Class<? extends IServeCommServer> serverClass) {

	Parameters.setIsInteractive(false);
	Parameters.processCommandLineArguments(argv);
	Debug.noteThreads = true;

	//\/ When exiting, could try starting a new server using 
	// Runtime.getRuntime().exec(...)

	IServeCommServer commServer = null;
	try {
	    commServer = Util.makeInstance(serverClass);
	    commServer.start();
	}
	catch (Throwable t) {
	    Debug.noteException(t);
	    System.exit(1);
	}
// 	Debug.noteln("Exiting main method for no good reason");
// 	System.exit(2);
        if (Parameters.isInteractive()) {
            while (true) {
                Util.askLine("Server:");
                System.out.println(commServer.makeServerStatus());
                System.out.println(commServer.describeAccessibleThreads());
                System.out.println("");
            }
        }
    }

    /*
     * The server
     */

    /** Used to start the server when running standalone. */
    protected void start() {

        processCommandLineArguments();

	// Server side
        int listenPort = getListenPort();
	httpServer = new HttpServer(listenPort);

        // Static content, if we should serve that too.
	String docRoot = Parameters.getParameter("http-root-directory");
	if (docRoot != null)
	    httpServer.parseDocRoot(docRoot, "files");

	// Logging
        String logName = getName();
	httpServer.setLogPath("logs", logName);

	// Servlets
        addServlets();

	// Start the server and see what port it's actually using.
	httpServer.start();

    }

    protected int getListenPort() {
        String pointer = Parameters.getParameter("ipc-server-pointer");
        if (pointer != null) {
            URI serverBase = httpUtil.followServerPointer(pointer);
            int serverPort = serverBase.getPort(); // -1 == no port
            return serverPort == -1 ? 80 : serverPort;
        }
        return Parameters.getInt("http-server-port", 0);
    }

    protected String getName() {
        if (iserve != null && iserve.getAgentSymbolName() != null)
            return iserve.getAgentSymbolName();
        else {
            for (String pname:
                     Arrays.asList("log-name", "symbol-name", "ipc-name")) {
                if (Parameters.haveParameter(pname))
                    return Parameters.getParameter(pname);
            }
            return "ix-comm-server"; // default
        }
    }

    protected void processCommandLineArguments() {
    }

    protected void addServlets() {
        addServlet(new RegistrationServlet(), "/ipc/ix/register");
 	addServlet(new SendServlet(), "/ipc/ix/send");
 	addServlet(new NextMessageServlet(), "/ipc/ix/get-next");
        addServlet(new StatusServlet(), "/ipc/status");
        addServlet(new ControlServlet(), "/ipc/control");
    }

    protected void addServlet(HttpServlet servlet, String pathSpec) {
        httpServer.addServlet(servlet, pathSpec);
    }

    protected static enum Command {
        REGISTER ("register"),
        SEND_TO ("send-to"),
        GET_MESSAGE ("get-message");

        private String text;

        Command(String text) {
            this.text = text;
        }

        public String text() {
            return text;
        }

    }

    /**
     * Provides a "status" page for the server.
     */
    class StatusServlet extends HttpServlet {

        StatusServlet() {
        }

        @Override
        protected void doGet(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {
            resp.setContentType("text/plain");
            resp.setStatus(HttpServletResponse.SC_OK);
            resp.getWriter().println(makeServerStatus());
        }

    }

    /**
     * Provides a "control" page for the server.  It is similar to
     * the "status" page but provides commands that can affect users
     * in various ways.
     */
    class ControlServlet extends HttpStringServlet {

        // The main reason we use an HttpStringServlet is to take
        // advantage of the way it turns exceptions into HTTP responses.

        // We "turn off" part of what it does: readRequest.

        ControlServlet() {
            setResponseContentType("text/html");
        }

        @Override
        protected void doGet(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException {

            doPost(req, resp);

        }

        @Override
        protected String readRequest(HttpServletRequest req) {
            return "";
        }

        @Override
        protected String handleRequest(HttpServletRequest req,
                                       String content)
            throws HttpRequestException {

            // Interpret the command, if one was given.
            String commandName = req.getParameter("command");
            Debug.noteln("Server command", commandName);
            if (commandName != null) {
                ServerCommand cmd = ServerCommand.valueOf(commandName);
                // ^ throws an exception if there's no value of that name.
                cmd.eval(req, IServeCommServer.this);
            }

            // Return a status-like description that also allows commands.
            // resp.setContentType("text/html");
            // resp.setStatus(HttpServletResponse.SC_OK);
            // Writer w = resp.getWriter();
            // HtmlWriter html = new HtmlWriter(w);
            HtmlStringWriter html = new HtmlStringWriter();
            Date now = new Date();

            html.tag("pre");
            // html.write(makeServerStatus());
            html.writeln("Status as of " + now);
            html.writeln("Started running " + agoTime(startupDate, now));
            html.end("pre");

            html.newLine();
            html.tag("form",
                     new String[][] {
                         {"method", "get"},
                         {"action", "control"}});
            html.newLine();
            html.tag("table");
            for (User u: getUsers()) {
                String userName = XML.encodeText(u.getName());
                html.tag("tr", "valign=\"top\"");
                html.tag("td");
                html.empty("input",
                           new String[][] {
                               {"type", "checkbox"},
                               {"name", "user"},
                               {"value", userName}});
                html.end("td");
                html.newLine();
                // html.tagged("td", userName);
                html.tag("td");
                html.tagged("pre", XML.encodeText(u.status(now)));
                html.end("td");
                html.end("tr");
                html.newLine();
            }
            html.end("table");
            html.newLine();
            html.select("command", ServerCommand.values(),
                                   ServerCommand.Select_A_Command);
            html.newLine();
            html.empty("input", "type=\"submit\" value=\"Submit\"");
            html.newLine();
            html.end("form");
            return html.toString();
        }

    }

    enum ServerCommand {

        Select_A_Command {
            void eval(HttpServletRequest req, IServeCommServer server) {
                // Do nothing.
            }
        },

        Delete_User {
            void eval(HttpServletRequest req, IServeCommServer server) {
                for (String name: selectedUsers(req)) {
                    server.removeUser(name);
                }
            }
        },

        Clear_Messages {
            void eval(HttpServletRequest req, IServeCommServer server) {
                for (String name: selectedUsers(req)) {
                    server.requireUser(name).clearUnacknowledgedMessages();
                }
            }
        };

        abstract void eval(HttpServletRequest req,
                           IServeCommServer server);

        String[] selectedUsers(HttpServletRequest req) {
            String[] names = req.getParameterValues("user");
            return names != null ? names : new String[]{};
        }

    }

    class RegistrationServlet extends HttpObjectServlet {

	RegistrationServlet() {
	}

	@Override
	protected Object handleRequest(HttpServletRequest req,
				       Object contents)
                  throws HttpRequestException {
	    Debug.noteln("Server asked to register", contents);
	    MessageWrapper w = (MessageWrapper)contents;
            fillInRequestInfo(w, req);
            requireCommand(Command.REGISTER, w);
            User user = ensureIXUser(w.getFrom());
            user.register(w);
	    Debug.noteln("Returning \"OK\"");
	    return Registration.makeSuccess(user.getUUID());
	}

    }

    class SendServlet extends HttpObjectServlet {

	SendServlet() {
	}

	@Override
	protected Object handleRequest(HttpServletRequest req,
				       Object contents)
                  throws HttpRequestException {
	    Debug.noteln("Server asked to send", contents);
            // Debug.noteln("JSON", gson.toJson(contents));
	    MessageWrapper w = (MessageWrapper)contents;
            fillInRequestInfo(w, req);
            requireCommand(Command.SEND_TO, w);
            ensureIXUser(w.getFrom())
                .checkForSend(w);
	    ensureUser(w.getTo())
		.addMessage(w);
	    Debug.noteln("Returning \"OK\"");
	    return "OK";
	}

    }

    class NextMessageServlet extends HttpObjectServlet {

	NextMessageServlet() {
	}

	@Override
	protected Object handleRequest(HttpServletRequest req,
				       Object contents)
                  throws HttpRequestException {
	    Debug.noteln("Server received", contents);
            // Debug.noteln("JSON", gson.toJson(contents));
            // noteAccessibleThreads();
	    MessageWrapper w = (MessageWrapper)contents;
            fillInRequestInfo(w, req);
            requireCommand(Command.GET_MESSAGE, w);
	    Object reply = ensureIXUser(w.getFrom())
		             .nextMessage(w);
            Debug.noteln("NextMessageServlet obtained", reply);
	    if (reply instanceof AbandonRequest) {
		Debug.noteln("Told to abandon request");
                // Normally the request times out rather than
                // getting the superseded status result, but we
                // have to do something to complete our handling
                // of the request, and this is it.
                throw (HttpRequestException)reply;
	    }
	    Debug.noteln("Sending next message to", w.getFrom());
	    return reply;
	}

    }

    protected void fillInRequestInfo(MessageWrapper w,
                                     HttpServletRequest req)
              throws HttpRequestException {
        w.setRemoteAddr(req.getRemoteAddr());
        w.setRemoteHost(req.getRemoteHost());
        if (w.getFrom() == null)
            throw new HttpRequestException
                (HttpURLConnection.HTTP_BAD_REQUEST,
                 "Missing the name of the sending agent.");
    }

    protected void requireCommand(Command command, MessageWrapper w)
              throws HttpRequestException {
        if (w.getCommand() == null)
            throw new HttpRequestException
                (HttpURLConnection.HTTP_BAD_REQUEST,
                 "No command; expected " + command);
        if (!w.getCommand().equals(command.text()))
            throw new HttpRequestException
                (HttpURLConnection.HTTP_BAD_REQUEST,
                 "Wrong command " + w.getCommand() +
                 "; expected " + command.text());
    }

    public static class AbandonRequest extends HttpRequestException {
	AbandonRequest() {
	    // 202 = Accepted
	    super(HttpURLConnection.HTTP_ACCEPTED,
                  "Request was superseded.");
	}
	public void printStackTrace(PrintStream s) {
	    // It doesn't represent an error, so a backtrace
	    // is misleading.
	}
    }

    protected String makeServerStatus() {
	// /\/: We don't call checkSender(m) in this case,
	// because MessageWrapper doesn't have one.  (Indeed,
	// the request might have come from a random browser.)
	List lines = new LinkedList();
	Date now = new Date();
	lines.add("Status as of " + now);
	// Startup date
	lines.add("Started running " + agoTime(startupDate, now));
// 	// Log file name
// 	lines.add("Log file: " +
// 		  Parameters.getParameter("server-log-file", "none"));
	// User status
	List<User> users = getUsers();
	if (users.isEmpty())
	    lines.add("No users");
	else {
	    lines.add("");
	    lines.add("Users:");
	    for (User u: users) {
		lines.add(u.status(now));
		lines.add("");
	    }
	}
        lines.add(Thread.activeCount() + " active threads.");
	return Strings.joinLines(lines);
    }

    /*
     * Utilities
     */

    protected String agoTime(Date then, Date now) {
	Duration delta = new Duration(then, now).roundToMinutes();
	return then +
	    (delta.asMilliseconds() > 0 	// means minutes > 0
	         ? ", " + delta + " ago"
	         : "");
    }

    private final Thread[] accessibleThreads = new Thread[100];

    private void noteAccessibleThreads() {
        Debug.noteln(describeAccessibleThreads());
    }

    private String describeAccessibleThreads() {
        // We are supposedly only able to get threads in our own ThreadGroup
        // and, presumably, its subgroups.  The 'enumerate' method includes
        // threads from subgroups.
        int n = Thread.enumerate(accessibleThreads);
        if (! (n < accessibleThreads.length))
            throw new ConsistencyException("too many threads: " + n);
        StringBuilder message = new StringBuilder();
        message.append(n);
        message.append(" threads: ");
        for (int i = 0; i < n; i++) {
            message.append(accessibleThreads[i].getName());
            if (i < n-1)
                message.append(", ");
        }
        message.append(".");
        return message.toString();
    }

    /*
     * User profiles
     */

    protected final Map<String,User> userNameToUserMap =
        new TreeMap<String,User>();

    protected User ensureIXUser(String name) {
        // This should be called only for the user who sent the
        // request to this server.
	synchronized(userNameToUserMap) {
	    User user = getUser(name);
	    if (user == null) {
		user = new User(name);
		recordUser(user);
	    }
            user.setIdentityChecker(ixIdentityCheckerFactory);
	    return user;
	}
    }

    protected User ensureUser(String name) {
	synchronized(userNameToUserMap) {
	    User user = getUser(name);
	    if (user == null) {
		user = new User(name);
                user.setIdentityChecker(defaultIdentityCheckerFactory);
		recordUser(user);
	    }
	    return user;
	}
    }

    protected User getUser(String name) {
	synchronized(userNameToUserMap) {
	    return userNameToUserMap.get(name);
	}
    }

    protected User requireUser(String name) {
        User user = getUser(name);
        if (user == null)
            throw new IllegalStateException
                ("There is no user named " + Strings.quote(name));
        else
            return user;
    }

    protected List<User> getUsers() {
	synchronized(userNameToUserMap) {
	    return new LinkedList<User>(userNameToUserMap.values());
	}
    }

    protected void recordUser(User user) {
	user.noteln("Recording user");
	synchronized(userNameToUserMap) {
	    String name = user.getName();
	    Debug.expect(getUser(name) == null,
			 "User " + name + " already exists");
	    userNameToUserMap.put(name, user);
	    // user.startMessageThread();
	}
    }

    protected void removeUser(String userName) {
        synchronized(userNameToUserMap) {
            removeUser(requireUser(userName));
        }
    }

    protected void removeUser(User user) {
	user.noteln("Removing user");
	synchronized(userNameToUserMap) {
            synchronized(user) {
                // user.killMessageThread();
                user.deleted();
                userNameToUserMap.remove(user.getName());
            }
	}
    }

    protected class User {

	String name;
        String uuid;

	MessageQueue responseQueue = null;
 	MessageMemory unacknowledged = new MessageMemory();

 	List<Date> registrationDates = new LinkedList<Date>();
	Date lastContactDate = null;    // last contact with this server
	Date lastSendDate = null; 	// last sent to another agent
	Date lastAckDate = null; 	// last new ack of a received message
        MessageWrapper lastAckdMessage = null;

        // This tells us whether we've ever sent anything in response
        // to a GET_NEXT request.  See maybeFixSequenceNumbers.
        boolean haveSentGetNext = false;

        // seqNo is used to number messages from other agents to this user.
        // It is the number that the next message to this user will get,
        // and so it is normally 1 greater than the sequence number we
        // gave to the most recent such message we've seen.
	int seqNo = 0;

        // /\/: At present, the identity-checker is the only aspect of
        // the user that can come in different classes.  Some of what it
        // does should be handled by having User subclasses instead.
        IdentityChecker checker;

        // Properties -- for things that don't fit into any field.
        // This should eventually be eliminated.  /\/
        Map<Symbol,Object> properties = new TreeMap<Symbol,Object>();

        public User(String name) {
            this.name = name;
        }

        public synchronized void setIdentityChecker(IdentityCheckerFactory f) {
            if (checker == null || !f.existingCheckerIsOk(checker)) {
                IdentityChecker c = f.makeIdentityChecker(this);
                noteln("Setting id checker to " + c);
                if (checker != null)
                    c.changingIdentityCheckerFrom(checker);
                checker = c;
                checker.init();
            }
        }

	public synchronized String getName() { return name; }

        public synchronized String getUUID() {
            Debug.expect(uuid != null, this + " has no uuid");
            return uuid;
        }

        public synchronized void setUUID(String uuid) {
            this.uuid = uuid;
        }

        public synchronized void assignUUID() {
            setUUID(UUID.randomUUID().toString());
        }

        //\/: Some methods are public only because used in IServeSLCommServer.

        public synchronized Object getProperty(Symbol name) {
            return properties.get(name);
        }

        public synchronized void setProperty(Symbol name, Object value) {
            noteln("set-property " + name + " = " + value);
            properties.put(name, value);
        }

        public synchronized Date getLastContactDate() {
            return lastContactDate;
        }

	public synchronized List getRegistrationDates() {
	    return new ArrayList(registrationDates); // snapshot
	}

        public synchronized Date getLastSendDate() {
            return lastSendDate;
        }

        public synchronized Date getLastAckDate() {
            return lastAckDate;
        }

        public synchronized MessageWrapper getLastAckdMessage() {
            return lastAckdMessage;
        }

        public synchronized void deleted() {
            // Called when this user's being deleted.
            // /\/: Note that an active user will quickly reappear,
            // since after it sees the AbandonRequest, it will just
            // send a new request for its next message.
	    if (responseQueue != null) {
		responseQueue.send(new AbandonRequest());
		responseQueue = null;
	    }
        }

        public synchronized void clearUnacknowledgedMessages() {
            unacknowledged.clear();
        }

	public synchronized void addMessage(MessageWrapper m) {
            // This is a message from another user, so we don't learn
            // anything new about this one.
	    noteln("Adding message " + seqNo);
	    m.setSeqNo(seqNo++); // give the message a sequence number
            checker.checkAddMessage(m);
	    unacknowledged.remember(m);
            // If something is waiting to get a message for this
            // user, give it one.
            maybeSendFirstUnacknowledgedMessage();
	}

	public synchronized void register(MessageWrapper w)
            throws HttpRequestException {
	    noteln(registrationDates.isEmpty()
                   ? "new registration" : "re-registration");
            checker.checkRegisterRequest(w);
            maybeFixSequenceNumbers(w);
	    Date rdate = sent(w);
            registrationDates.add(rdate);
        }

        private void maybeFixSequenceNumbers(MessageWrapper w) {
	    // If the user is new to this server, but has received
	    // some messages [seq > -1], that probably means it
	    // talked with an earlier server that then exited.
	    // If the user's new to us, we can't have sent it anything,
	    // so we have to make any messages we have (or get)
	    // look new to the user's agent by giving them numbers
	    // above those the agent's already received.
            if (w.getSequenceNumber() == null) {
                noteln("No sequence number in wrapper");
                return;
            }
	    int seq = w.getSeqNo();
	    noteln("Seq numbers: " + w.getCommand() + " request says " + seq +
                   ", we have " + seqNo);
            // We used to check lastContactDate == null here, but if the
            // 1st thing an agent new to us does is to send to another
            // agent (rather than register or ask for its next message),
            // that will set its last contact date, and then the 1st ack
            // will look like it was from an agent we'd been talking to
            // even though it wasn't.  So it would be treated as
            // acknowledging a message we hadn't actually sent.
	    if (!haveSentGetNext && seq > -1) {
		noteln("Renumbering messages");
		final int delta = seq + 1;
		// Renumber any pending messages.  /\/ Since we don't
		// yet allow mail to unregistered users, there won't
		// be any messages.
                unacknowledged.renumberMessages(delta);
		seqNo += delta;
		noteln("Revised next-message number " + seqNo);
	    }
	}

        public synchronized void checkForSend(MessageWrapper w)
            throws HttpRequestException {
            // Called when we get a request from this user's agent
            // to send to another agent.
            checker.checkSendRequest(w);
            sent(w);
        }

	private Date sent(MessageWrapper w) {
            // Called whenever this user's agent sends anything to the server.
            // We have to be careful when we call this, because it
            // sets lastContactDate and can set lastSendDate.
	    Date now = new Date();
	    lastContactDate = now;
	    if (w.getCommand().equals("send-to"))
		lastSendDate = now;
            return now;
	}

        public Object nextMessage(MessageWrapper request)
            throws HttpRequestException {

            // Note that this method is *not* synchronized,
            // but getResponseQueue is.

            return getResponseQueue(request).nextMessage();

        }

	private
        synchronized MessageQueue getResponseQueue(MessageWrapper req)
            throws HttpRequestException {

            noteln("getResponseQueue called");

	    // This is called only by nextMessage and only when we
            // get a new request for this user's next message.
            // The request must come from this user's agent,
            // but we don't at present have a definitive way
            // to check.  /\/
            checker.checkNextMessageRequest(req);
            maybeFixSequenceNumbers(req);
            sent(req);

            // The sequence number tells us what the agent that sent
            // the get-message most recently received.  So now we know
            // the agent really received it.
            if (req.getSequenceNumber() != null)
                acknowledgedReceipt(req.getSequenceNumber());

	    // Assume anything that's already waiting has been superseded
	    // by the new request.
	    if (responseQueue != null) {
		responseQueue.send(new AbandonRequest());
		responseQueue = null;
	    }

	    // Make a queue for the new request
	    responseQueue = new MessageQueue();

            MessageQueue result = responseQueue;

            // Give the request a message if there is one.
            // If there is a message, this call will put it
            // into responseQueue and make responseQueue = null;
            maybeSendFirstUnacknowledgedMessage();

	    // The request will wait until we put something into its
            // queue (which we may have just done).
	    return result;

	}

        /*
        public synchronized void assumeAcknowledged(MessageWrapper sent) {
            //\/: This is a hopfully temporary favility for communication
            // with agents, such as some in SL. that cannot acknowledge
            // properly.
            Integer key = sent.getSequenceNumber();
            if (key != null) {
                noteln("Assuming ack for", key);
                acknowledgedReceipt(key);
            }
        }
        */

        // It's possible that we are a new server talking with an agent
        // that had interacted with an earlier incarnation of the server,
        // so that its ack refers to a message we don't have even if
        // the ack's seq number does match one of ours.  We deal with
        // this by renumbering the 'unacknowledged' messages to have
        // higher numbers than the one the agent says it received.
        // See calls to maybeFixSequenceNumbers.  /\/

        // This problem would be eliminated if we preserved state from
        // earlier incarnations.  /\/

	private void acknowledgedReceipt(Integer key) {
            // Note that it's possible to get more than one ack
            // for the same message.
	    Date now = new Date();
	    lastContactDate = now;
	    if (unacknowledged.containsKey(key)) {
		noteln("Acknowledged receipt of message", key);
		lastAckDate = now;
                lastAckdMessage = unacknowledged.getMessage(key);
		unacknowledged.forgetKey(key);
	    }
	}

        private void maybeSendFirstUnacknowledgedMessage() {
            // See if anything's waiting for a message
            if (responseQueue != null) {
                // If we have a message waiting,
                // give it to the request.
                MessageWrapper m = getFirstUnacknowledgedMessage();
                if (m != null) {
                    // unacknowledged.forget(m); // <---- temp for SL /\/
                    responseQueue.send(m);
                    // The request is no longer waiting.
                    responseQueue = null;
                    // And we have sent something
                    haveSentGetNext = true;
                }                
            }
        }

	private MessageWrapper getFirstUnacknowledgedMessage() {
	    MessageWrapper m = unacknowledged.getFirstRemainingMessage();
	    if (m != null)
		noteln("Getting 1st message; it has key",
		       m.getSequenceNumber());
	    return m;
	}

        public void noteln(String line) {      // convenience method
	    Debug.noteln(this + ": " + line);
	}

	public void noteln(String text, Object about) { // convenience method
	    Debug.noteln(this + ": " + text, about);
	}

	public synchronized String toString() {
	    return "User[" + name + "]";
	}

	synchronized String status(Date now) {
	    final List<String> lines = new LinkedList<String>();
	    // Name and address
	    lines.add(name);
	    // Times since ...
            if (lastContactDate != null)
                lines.add("  Time of last active contact: " +
                          agoTime(lastContactDate, now));
	    if (lastSendDate != null)
		lines.add("  Last send to another agent:  " +
			  agoTime(lastSendDate, now));
	    if (lastAckDate != null)
		lines.add("  Last receipt acknowledgment: " +
			  agoTime(lastAckDate, now));
	    // Registration dates
	    List regs = registrationDates;
	    if (!regs.isEmpty()) {
		// Probably an error if it was empty. /\/
		lines.add("  Registration times:");
		for (Iterator i = regs.iterator(); i.hasNext();) {
		    Date d = (Date)i.next();
		    lines.add("    " + agoTime(d, now));
		}
	    }
            // Anything the identity-checker wants to add
            checker.status(now, lines);
	    // Waiting messages
            if (!unacknowledged.isEmpty()) {
                lines.add("  Waiting messages:");
                addWaitingMessageStatus(lines);
            }
	    return Strings.joinLines(lines);
	}

        private void addWaitingMessageStatus(List<String> lines) {
            int n = unacknowledged.size();
            int max = max_waiting_messages_to_list_in_status;
            if (n <= max + 1) {
                // + 1 because we may as well give the mssage
                // rather than say "... (1 other message) ...".
                for (MessageWrapper w: unacknowledged) {
                    lines.add(describeMessage(w));
                }
                return;
            }
            int cut = max / 2;
            int uncut = n - cut;
            int i = 0;
            for (MessageWrapper w: unacknowledged) {
                if (i == cut)
                    lines.add("... (" + (n - max) + " other messages) ...");
                else if (i < cut || i >= uncut)
                    lines.add(describeMessage(w));
                i++;
            }
        }

	private String describeMessage(MessageWrapper m) {
	    Object contents = m.getContents();
	    Class c = contents.getClass();
	    String className = XML.nameForClass(c);
	    return m.getSeqNo() + ": "
                +  m.getSendDate() + " " + className + " from " + m.getFrom()
		+  ": " + contents;
	}

    }

    protected interface IdentityCheckerFactory {

        public boolean existingCheckerIsOk(IdentityChecker checker);

        public IdentityChecker makeIdentityChecker(User user);

    }

    protected class DefaultIdentityCheckerFactory
              implements IdentityCheckerFactory {

        public boolean existingCheckerIsOk(IdentityChecker checker) {
            // An existing checker is always OK, and so a user
            // should never change *to* a DefaultIdentityChecker.
            return checker instanceof IdentityChecker;
        }

        public IdentityChecker makeIdentityChecker(User user) {
            return new DefaultIdentityChecker(user);
        }

    }

    protected class IXIdentityCheckerFactory
              implements IdentityCheckerFactory {

        public boolean existingCheckerIsOk(IdentityChecker checker) {
            return checker instanceof IXIdentityChecker;
        }

        public IdentityChecker makeIdentityChecker(User user) {
            return new IXIdentityChecker(user);
        }

    }

    public class IllegalIdentityCheckerChange extends IllegalStateException {

        public IllegalIdentityCheckerChange(User user,
                                            IdentityChecker oldChecker,
                                            IdentityChecker newChecker) {
            super("Illegal identity-checker change for " + user + ": " +
                  "from " + Util.aClass(oldChecker) +
                  " to "  + Util.aClass(newChecker));
        }

    }

    protected abstract class IdentityChecker {

        protected final User user;

        protected String password;

	protected String hostName;
	protected String hostAddr;

        protected IdentityChecker(User user) {
            this.user = user;
        }

        public void changingIdentityCheckerFrom(IdentityChecker oldChecker) {
            throw new IllegalIdentityCheckerChange(user, oldChecker, this);
        }

        public void init() {
        }

        // The current rules are:
        // 1. An agent must start by registering.  Other agents can send it
        //    messages before that, but it can't get them or send any of its
        //    own.  The agent's host is recorded when it registers, and it
        //    is given a new UUID.
        // 2. If an agent moves to a different host, or forgets its UUID
        //    (maybe it exited and was restarted), it has to register again.
        // 3. Once one of an agent's resistrations has supplied a password,
        //    that password must be given in all subsequent registrations.
        //    The password needn't be given in send or next-message requests,
        //    only in registrations.
        // 4. The correct UUID must be included in every send or next-message
        //    request, and they must come from the correct host (where
        //    "correct" was established by the most recent registration).
        //    (Perhaps the correct UUID should be enough on its own?)

        public void checkRegisterRequest(MessageWrapper req)
               throws HttpRequestException {

            checkMessagePassword(req);
            // If we reach this point, the password, if any, was correct.

            if (hostAddr == null) {
                // If we don't already know the host addr, assume
                // this message has the right one.
                hostAddr = req.getRemoteAddr();
                hostName = req.getRemoteHost();
                user.noteln("Host is " + hostName + " " + hostAddr);
            }
            else if (hostAddr.equals(req.getRemoteAddr())) {
                // The request came from the same host as before.
                ;
            }
            else {
                // The request came from a different host, but we'll
                // change the host. (We already know the password,
                // if there is one, is correct.)
                hostAddr = req.getRemoteAddr();
                hostName = req.getRemoteHost();
                user.noteln("Host is now" + hostName + " " + hostAddr);
            }

            // Generate a new random UUID for the user.

            // * When the agent registers, it should get a new UUID to
            // distinguish it from any agent-instance using the same name
            // that might still be out there and have the old UUID.

            // * We assign a new UUID here, rather than in the User class,
            // because a different type of identity-checker might handle
            // UUIDs differently.
            user.assignUUID();

        }

        public void checkSendRequest(MessageWrapper req)
               throws HttpRequestException {
            checkMessageSource(req);
        }

        public void checkAddMessage(MessageWrapper req) {
            // Normally, this is always ok.
        }

        public void checkNextMessageRequest(MessageWrapper req)
               throws HttpRequestException {
            checkMessageSource(req);
        }

        public void status(Date now, List<String> statusLines) {
            ;
        }

        /**
         * Checks whether the message came from an acceptable source.
         * If an agent (name) starts sending from a different host, for
         * example, then this method might object unless something (such
         * as a password) shows that it is the same agent rather than an
         * impostor.
         *
         * <p>This method is typically called by the public 'check' methods
         * that check the user who sent the request to the server.
         *
         * @throws HttpRequestException if the message was sent by
         *   an agent that this server does not recognise as a valid
         *   source for that message.
         */
        protected void checkMessageSource(MessageWrapper w)
                  throws HttpRequestException {
            // Passwords are now relevant only in registration messages.
            // checkMessagePassword(w);
            // // If we reach this point, the password, if any, was correct.
            checkMessageHost(w);
            checkMessageUUID(w);
        }

        protected void checkMessageHost(MessageWrapper w)
                  throws HttpRequestException {
            if (hostAddr == null) {
                throw new ConsistencyException(user + " has no host addr");
            }
            if (w.getRemoteAddr() == null) {
                throw new ConsistencyException
                    ("Failed to attach host addr to message for " + user);
            }
            if (!hostAddr.equals(w.getRemoteAddr())) {
                // The request came from the wrong host.
                throw new HttpRequestException
                    (HttpURLConnection.HTTP_FORBIDDEN, "Wrong host");
            }
        }

        protected void checkMessageUUID(MessageWrapper w)
                  throws HttpRequestException {
            String uuid = user.getUUID();
            if (uuid == null) {
                throw new ConsistencyException(user + " has no UUID.");
            }
            if (w.getUUID() == null) {
                // The request did not include a UUID.
                throw new HttpRequestException
                    (HttpURLConnection.HTTP_FORBIDDEN, "Missing UUID");
            }
            if (!uuid.equals(w.getUUID())) {
                // The request had the wrong UUID.
                throw new HttpRequestException
                    (HttpURLConnection.HTTP_FORBIDDEN, "Wrong UUID");
            }
        }

        protected void checkMessagePassword(MessageWrapper w)
                  throws HttpRequestException {
            // This method will return if the message has the right password,
            // or if it has no password; otherwise an exception will be
            // thrown.  If the message has a password and this user doesn't
            // already have one, the message's password is assumed to
            // be correct and is recorded as this user's password.
            if (password == null) {
                if (w.getPassword() != null) {
                    password = w.getPassword();
                    user.noteln("Set password to", password);
                }
            }
            else if (w.getPassword() != null) {
                if (!password.equals(w.getPassword()))
                    throw new HttpRequestException
                        (HttpURLConnection.HTTP_FORBIDDEN,
                         "Invalid password");
            }
        }

    }

    protected class DefaultIdentityChecker extends IdentityChecker {

        DefaultIdentityChecker(User user) {
            super(user);
        }

        @Override
        protected void checkMessageSource(MessageWrapper w) {
            throw new IllegalStateException
                ("The default identity-checker should not be in force " +
                 "once an agent has sent something to the server.  " +
                 "The user involved is " + user);
        }

    }

    protected class IXIdentityChecker extends IdentityChecker {

        IXIdentityChecker(User user) {
            super(user);
        }

        @Override
        public void changingIdentityCheckerFrom(IdentityChecker oldChecker) {
            if (! (oldChecker instanceof DefaultIdentityChecker))
                throw new IllegalIdentityCheckerChange(user, oldChecker, this);
        }

    }

}
