;;; Author: Jeff Dalton <J.Dalton@ed.ac.uk>
;;; Updated: Tue Nov 16 19:13:37 2004 by Jeff Dalton

(cl:in-package :util)

;;;; Concatenation and other string operations

(defun concat-name (s1 s2 &rest more)
  (intern (apply #'concatenate 'string
		 (string s1) (string s2) (mapcar #'string more))))

(defun concat-string (s1 s2 &rest more)
  (apply #'concatenate 'string
	 (string s1) (string s2) (mapcar #'string more)))

(defun big-string-concat (strings)
  (let ((len 0))
    (dolist (s strings) (incf len (length s)))
    (let ((big-string (make-string len))
	  (i 0))
      (dolist (s strings big-string)
	(replace big-string s :start1 i)
	(incf i (length s))))))

(defun string->keyword (s)
  (intern (string-upcase s) (find-package :keyword)))

(defun string->int (s)
  (handler-case (values (parse-integer s))
    (error (c)
      (error "Can't convert ~S to an int because: ~A" s c))))

(defun int->string (i)
  (write-to-string i))


;;; Length=1 is used to test whether a list contains only one element.
;;; It should be faster than actually calling length, and it may remove
;;; the temptation to just say (null (cdr x)) w/o being sure x is a cons
;;; (or when relying on an earlier test in a COND that might change).
;;; /\/: A different name, such as singleton-p, might be better.

(defun-inline length=1 (x)
  (and (consp x) (null (cdr x))))

(defun-inline length>1 (x)
  (and (consp x) (consp (cdr x))))


;;; Last-element does what last doesn't, namely return the last element.

(defun-inline last-element (lis)
  (car (last lis)))

;;; Last-elt returns the last element of a sequence.

(defun-inline last-elt (seq)
  (if (listp seq)
      (last-element seq)
    (let ((len (length seq)))
      (declare (fixnum len))
      (if (= len 0)
	  nil
	(elt seq (the fixnum (1- len)))))))


;;; Output comma-separtated items on separate lines.

;;; /\/: The O-Plan version does the newlines before each prefix,
;;; rather than after each comma.

(defun output-comma-separated (prefix strings &optional final-suffix)
  (when strings
    (output prefix (car strings))
    (cond ((null (cdr strings))
	   (when final-suffix
	     (output final-suffix)))
	  (t
	   (output "," //)
	   (output-comma-separated prefix (cdr strings) final-suffix)))))


;;; Sequence functions

;;; /\/: Move replace-subseq to util-function.lsp?

;;; /\/: Will type-of always return something concatenate will like?

(defun replace-subseq (new old seq)
  (let ((pos (search old seq)))
    (if (null pos)
        seq
      (concatenate (type-of seq)
	  (subseq seq 0 pos)
          new
          (replace-subseq new
			  old
			  (subseq seq (+ pos (length old))))))))

(defun sequence-begins (init-seg seq)
  (let ((m (mismatch init-seg seq)))
    (or (not m)
	(eql m (length init-seg)))))

(defun sequence-after (init-seg seq)
  (subseq seq (length init-seg)))

(defun sequence-after-first (subseq seq)
  (let ((pos (search subseq seq)))
    (if (null pos)
	(error "Sequence ~S does not contain a subsequence ~S"
	       seq subseq)
      (subseq seq (+ pos (length subseq))))))

(defun sequence-after-last (subseq seq)
  (let ((pos (search subseq seq :from-end t)))
    (if (null pos)
	(error "Sequence ~S does not contain a subsequence ~S"
	       seq subseq)
      (subseq seq (+ pos (length subseq))))))

(defun sequence-before-first (subseq seq)
  (let ((pos (search subseq seq)))
    (if (null pos)
	(error "Sequence ~S does not contain a subsequence ~S"
	       seq subseq)
      (subseq seq 0 pos))))

(defun sequence-before-last (subseq seq)
  (let ((pos (search subseq seq :from-end t)))
    (if (null pos)
	(error "Sequence ~S does not contain a subsequence ~S"
	       seq subseq)
      (subseq seq 0 pos))))

(defun equal-as-sets (seq1 seq2)
  (and (= (length seq1) (length seq2))
       (every #'(lambda (x) (find x seq2 :test #'equal))
	      seq1)))

;;; *char=
;;;
;;; /\/: *char= is needed at least in AKCL.  With *char= we get
;;; if(!((V1)==(((V2))->ust.ust_self[V6]))) ...; otherwise we get
;;; if(!(char_code(code_char(V1))==char_code(code_char(((V2))->
;;; ust.ust_self[fix(V7)]))))... .  Why we have fix(V7) is not clear.
;;; [This is from (disassemble 'break-string-at).]

(defmacro *char= (c1 c2)
  `(char= (the character ,c1) (the character ,c2)))

;;; (Break-string-at char string) takes a string containing fields
;;; separated by char and returns a list of the fields, in order, as
;;; strings.

(defun break-string-at (char string)
  (declare (character char) (simple-string string))
  (let ((start 0)
	(parts '()))
    (declare (fixnum start))
    (fix-dotimes (i (length string))
      (when (*char= char (schar string i))
	(push (subseq string start i) parts)
	(setq start (fix1+ i))))
    (nreverse (cons (subseq string start) parts))))

;;; (Break-string-at-first char string) takes a string containing fields
;;; separated by char and returns two values: the substring before the
;;; first occurrence of the char, and the substring after.  Neither
;;; substring contains the char.  If the char does not appear in the
;;; string at all, the values are the string and "".

(defun break-string-at-first (char string)
  (declare (character char) (simple-string string))
  (fix-dotimes (i (length string) (values string ""))
    (when (*char= char (schar string i))
      (return
	(values (subseq string 0 i)
		(subseq string (fix1+ i)))))))

;;; (break-string-at-last char string) is like a break-string-at-first
;;; that ignores all instances of char except the last.  So if the char
;;; does not appear at all, the values are string and "".

(defun break-string-at-last (char string)
  (declare (character char) (simple-string string))
  (let ((j (length string)))
    (declare (fixnum j))
    (fix-dotimes (i j (values string ""))
      (setq j (fix1- j))
      (when (*char= char (schar string j))
        (return
	  (values (subseq string 0 j)
		  (subseq string (fix1+ j))))))))

(defun concat-strings-with-separator (sep-string strings)
  (if (null strings)
      ""
    (big-string-concat
      (cons (car strings)
	    (flatten-one-level
	      (mapcar #'(lambda (s) (list sep-string s))
		      (cdr strings)))))))


;;;; Time and date formatting.

;;; Day and month names

;;; Note that in Common Lisp, Monday is day 0 but January is month 1.

(defvar *days* '#(monday tuesday wednesday thursday friday saturday sunday))

(defvar *months* '#(january february march april may june july
		    august september october november december))

(defun int->day (i)
  (check-day-int i)
  (string (elt *days* i)))

(defun check-day-int (i)
  (if (<= 0 i 6)
      i
    (error "~S is not a valid number for a day." i)))

(defun int->month (i)
  (check-month-int i)
  (string (elt *months* (1- i))))

(defun check-month-int (i)
  (if (<= 1 i 12)
      i
    (error "~S is not a valid number for a month." i)))

(defun time-and-date-string (&optional (utime (get-universal-time)))
  ;; Similiar output to unix date command.
  (multiple-value-bind (sec min hr day month year day-of-week dst-p time-zone)
      (decode-universal-time utime)
    (declare (ignore dst-p time-zone))
    (format nil "~:(~A ~A~) ~2D ~2,'0D:~2,'0D:~2,'0D ~4D"
       (subseq (int->day day-of-week) 0 3)
       (subseq (int->month month) 0 3)
       day
       hr
       min
       sec
       year)))

;;; End
